﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Projects
{
    using NintendoWare.SoundFoundation.Core.Parameters;

    public class StreamSoundBase : Sound
    {
        /// <summary>
        ///
        /// </summary>
        public StreamSoundBase()
        {
        }

        /// <summary>
        /// エンコード
        /// </summary>
        public WaveEncoding Encoding
        {
            get { return GetValue<WaveEncoding>(ProjectParameterNames.WaveEncoding); }
            set { SetValue<WaveEncoding>(ProjectParameterNames.WaveEncoding, value); }
        }

        ///--------------------------------
        /// <summary>
        /// パンモード
        /// </summary>
        public PanMode PanMode
        {
            get { return GetPanModeValue(ProjectParameterNames.PanMode); }
            set { SetPanModeValue(ProjectParameterNames.PanMode, value); }
        }

        ///--------------------------------
        /// <summary>
        /// パンカーブ
        /// </summary>
        public PanCurve PanCurve
        {
            get { return GetPanCurveValue(ProjectParameterNames.PanCurve); }
            set { SetPanCurveValue(ProjectParameterNames.PanCurve, value); }
        }

        /// <summary>
        /// ピッチ
        /// </summary>
        public float Pitch
        {
            get { return GetValue<float>(ProjectParameterNames.Pitch); }
            set { SetValue<float>(ProjectParameterNames.Pitch, value); }
        }

        /// <summary>
        /// センド設定を取得または設定します。
        /// </summary>
        public Sends Sends
        {
            get { return this.GetValue<Sends>(ProjectParameterNames.Sound.Sends); }
            set { this.SetValue<Sends>(ProjectParameterNames.Sound.Sends, value); }
        }

        /// <summary>
        /// プリフェッチ出力の設定を取得または設定します。
        /// </summary>
        public bool IsPrefetchEnabled
        {
            get { return this.GetBoolValue(ProjectParameterNames.StreamSound.IsPrefetchEnabled); }
            set { this.SetBoolValue(ProjectParameterNames.StreamSound.IsPrefetchEnabled, value); }
        }

        /// <summary>
        /// リサンプルの設定を取得または設定します。
        /// </summary>
        public bool IsResampleEnabled
        {
            get { return this.GetBoolValue(ProjectParameterNames.IsResampleEnabled); }
            set { this.SetBoolValue(ProjectParameterNames.IsResampleEnabled, value); }
        }

        /// <summary>
        /// サンプルレートを取得または設定します。
        /// </summary>
        public int SampleRate
        {
            get { return this.GetIntValue(ProjectParameterNames.SampleRate); }
            set { this.SetValue(ProjectParameterNames.SampleRate, value); }
        }

        /// <summary>
        /// ダウンミックス設定を取得または設定します。
        /// </summary>
        public bool IsDownMixEnabled
        {
            get { return this.GetBoolValue(ProjectParameterNames.IsDownMixEnabled); }
            set { this.SetBoolValue(ProjectParameterNames.IsDownMixEnabled, value); }
        }

        /// <summary>
        /// パラメータを初期化します。
        /// </summary>
        protected override void InitializeParameters()
        {
            base.InitializeParameters();

            SetInstanceBoolValue(ProjectParameterNames.SndEdit, false);
            SetInstanceBoolValue(ProjectParameterNames.SndEditEnabled, true);

            SetInstanceWaveEncodingValue(ProjectParameterNames.WaveEncoding, WaveEncoding.Adpcm);
            SetInstanceBoolValue(ProjectParameterNames.StreamSound.IsPrefetchEnabled, false);
            SetInstanceBoolValue(ProjectParameterNames.IsResampleEnabled, false);
            SetInstanceParameterValue(ProjectParameterNames.SampleRate, new SampleRateParameterValue());
            SetInstanceBoolValue(ProjectParameterNames.IsDownMixEnabled, false);

            SetInstancePanModeValue(ProjectParameterNames.PanMode, PanMode.Balance);
            SetInstancePanCurveValue(ProjectParameterNames.PanCurve, PanCurve.Sqrt);
            this.Parameters.AddValue(ProjectParameterNames.Pitch,
                                      new PitchParameterValue(1.0F, 0.00001F, float.MaxValue));

            SendsParameterValue sendsParam = new SendsParameterValue(new Sends());
            Parameters.AddValue(ProjectParameterNames.Sound.Sends, sendsParam);

            // ストリームバイナリ依存パラメータ
            this.SetParameterAttributes(ProjectParameterNames.Name, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.WaveEncoding, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.IsResampleEnabled, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.SampleRate, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.IsDownMixEnabled, ParameterAttributes.ComputeHash);
            this.SetParameterAttributes(ProjectParameterNames.StreamSound.IsPrefetchEnabled, ParameterAttributes.ComputeHash);

            // sndedit 転送対象パラメータ
            this.SetParameterAttributes(ProjectParameterNames.PanMode, ParameterAttributes.ComputeSndeditTransferHash);
            this.SetParameterAttributes(ProjectParameterNames.PanCurve, ParameterAttributes.ComputeSndeditTransferHash);
            this.SetParameterAttributes(ProjectParameterNames.Pitch, ParameterAttributes.ComputeSndeditTransferHash);
            this.SetParameterAttributes(ProjectParameterNames.Sound.Sends, ParameterAttributes.ComputeSndeditTransferHash);

            // StreamSoundTrackBase.FilePath を参照するため、削除します。
            Parameters.RemoveValue(ProjectParameterNames.FilePath);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SetInstanceWaveEncodingValue(string name, WaveEncoding value)
        {
            SetInstanceParameterValue(name, new WaveEncodingParameterValue(value));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void SetInstanceLoopTypeValue(string name, LoopType value)
        {
            SetInstanceParameterValue(name, new LoopTypeParameterValue(value));
        }
    }
}
