﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using NintendoWare.SoundFoundation.Core.Parameters;
using NintendoWare.SoundFoundation.Operations;

namespace NintendoWare.SoundFoundation.Parameters
{
    /// <summary>
    /// パラメータを設定するオペレーションです。
    /// </summary>
    public class SetParameterOperation : OperationImpl
    {
        private IParameterDictionary _target = null;              // 対象パラメータディクショナリ
        private string _parameterKey = string.Empty;      // 設定するパラメータキー
        private object _newValue = null;              // 新しく設定する値
        private object _oldValue = null;              // 設定する前の値
        private string _newValueText = null;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="target">対象パラメータディクショナリ。</param>
        /// <param name="parameterKey">設定するパラメータキー。</param>
        /// <param name="value">新しく設定する値。</param>
        public SetParameterOperation(IParameterDictionary target, string parameterKey, object value)
        {
            if (null == target) { throw new ArgumentNullException("target"); }
            if (null == parameterKey) { throw new ArgumentNullException("parameterKey"); }
            if (0 == parameterKey.Length) { throw new ArgumentException("parameterKey.Length must not be 0."); }
            if (null == value) { throw new ArgumentNullException("value"); }

            _target = target;
            _parameterKey = parameterKey;
            _newValue = value;

            if (target.ContainsKey(parameterKey))
            {
                _oldValue = target[parameterKey].Value;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public SetParameterOperation(IParameterDictionary target, string parameterKey, string value)
        {
            if (null == target) { throw new ArgumentNullException("target"); }
            if (null == parameterKey) { throw new ArgumentNullException("parameterKey"); }
            if (0 == parameterKey.Length) { throw new ArgumentException("parameterKey.Length must not be 0."); }
            if (null == value) { throw new ArgumentNullException("value"); }

            _target = target;
            _parameterKey = parameterKey;
            _newValue = null;
            _newValueText = value;

            if (target.ContainsKey(parameterKey))
            {
                _oldValue = target[parameterKey].Value;
            }
        }

        /// <summary>
        /// 対象パラメータディクショナリを取得します。
        /// </summary>
        public IParameterDictionary TargetDictionary
        {
            get { return _target; }
        }

        /// <summary>
        /// 設定するパラメータキーを取得します。
        /// </summary>
        public string ParameterKey
        {
            get { return _parameterKey; }
        }

        /// <summary>
        /// 新しく設定する値を取得します。
        /// </summary>
        public object Value
        {
            get { return _newValue; }
        }

        /// <summary>
        /// オペレーションを実行します。
        /// </summary>
        /// <returns>処理された場合は true、処理できなかった場合は false。</returns>
        protected override bool ExecuteInternal()
        {
            if (null != _newValue)
            {
                _target[_parameterKey].Value = _newValue;
            }
            else
            {
                Debug.Assert(null != _newValueText, "unexpected error");
                _target[_parameterKey].Parse(_newValueText);
            }
            return true;
        }

        /// <summary>
        /// オペレーションをロールバックします。
        /// </summary>
        /// <returns>処理された場合は true、処理できなかった場合は false。</returns>
        protected override bool RollbackInternal()
        {
            _target[_parameterKey].Value = _oldValue;
            return true;
        }
    }
}
