﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Diagnostics;

namespace NintendoWare.SoundFoundation.Operations
{
    /// <summary>
    /// 複数のオペレーションを束ねたトランザクション機能を提供します。
    /// </summary>
    public class Transaction : Operation, IDisposable
    {
        private bool _disposed = false;

        private OperationCollection _operations = new OperationCollection();   // オペレーションリスト
        private string _description = string.Empty;                // トランザクションの説明
        private bool _executed = false;                       // 実行済フラグ

        private TransactionStackCollection _stacks = new TransactionStackCollection();  // 関連するトランザクションスタック

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public Transaction() : this(string.Empty) { }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="description">トランザクションの説明。</param>
        public Transaction(string description)
        {
            if (null == description) { throw new ArgumentNullException("description"); }
            _description = description;
        }

        /// <summary>
        /// トランザクションの説明を取得します。
        /// </summary>
        public string Description
        {
            get { return _description; }
        }

        /// <summary>
        /// オペレーション一覧を取得します。
        /// </summary>
        public IOperationCollection Operations
        {
            get { return _operations; }
        }

        /// <summary>
        /// このトランザクションが追加されているスタックのコレクションを取得します。
        /// </summary>
        public TransactionStackCollection Stacks
        {
            get { return _stacks; }
        }

        /// <summary>
        /// このトランザクションが連動するかどうかを取得します。
        /// </summary>
        public bool IsInterlocked
        {
            get { return (1 < _stacks.Count); }
        }

        /// <summary>
        /// オペレーションが実行されたかどうかを取得します。
        /// </summary>
        public sealed override bool IsExecuted
        {
            get
            {
                if (0 == _operations.Count)
                {
                    return _executed;
                }
                else
                {

                    foreach (Operation operation in _operations)
                    {
                        if (!operation.IsExecuted) { return false; }
                    }

                    return true;

                }
            }
        }

        /// <summary>
        /// オペレーションを実行します。
        /// </summary>
        /// <returns>処理された場合は true、処理できなかった場合は false。</returns>
        public sealed override bool Execute()
        {
            if (!CanExecute()) { return false; }

            bool executed = true;

            try
            {

                executed = ExecuteInternal();

            }
            catch (Exception)
            {

                // 例外が発生した場合、ロールバックする
                try
                {
                    RollbackInternal();
                }
                catch
                {
                    Debug.Fail("Failed to cancel execute.");
                }

                throw;

            }

            if (!executed)
            {

                // 実行されなかった場合、ロールバックする
                try
                {
                    Rollback();
                }
                catch
                {
                    Debug.Fail("Failed to cancel execute.");
                }

            }

            return executed;
        }

        /// <summary>
        /// オペレーションをロールバックします。
        /// </summary>
        /// <returns>処理された場合は true、処理できなかった場合は false。</returns>
        public sealed override bool Rollback()
        {
            if (!CanRollback()) { return false; }

            bool executed = true;

            try
            {

                executed = RollbackInternal();

            }
            catch (Exception)
            {

                // 例外が発生した場合、実行（ロールバックをキャンセル）する
                try
                {
                    ExecuteInternal();
                }
                catch
                {
                    Debug.Fail("Failed to cancel rollback.");
                }

                throw;

            }

            if (!executed)
            {

                // ロールバックされなかった場合、実行（ロールバックをキャンセル）する
                try
                {
                    Execute();
                }
                catch
                {
                    Debug.Fail("Failed to cancel rollback.");
                    throw;
                }

            }

            return executed;
        }

        /// <summary>
        /// 現在、オペレーションを実行可能かどうかを取得します。
        /// </summary>
        /// <returns>実行可能な場合は true、実行不可能な場合は false。</returns>
        public sealed override bool CanExecute()
        {
            return CanExecuteInternal();
        }

        /// <summary>
        /// 現在、オペレーションをロールバック可能かどうかを取得します。
        /// </summary>
        /// <returns>ロールバック可能な場合は true、ロールバック不可能な場合は false。</returns>
        public sealed override bool CanRollback()
        {
            return CanRollbackInternal();
        }

        /// <summary>
        /// トランザクションを破棄します。
        /// </summary>
        public void Dispose()
        {
            Dispose(true);
        }

        /// <summary>
        /// オペレーション一覧を取得します。
        /// </summary>
        protected OperationCollection OperationsInternal
        {
            get { return _operations; }
        }

        /// <summary>
        /// 現在、オペレーションを実行可能かどうかを取得します。
        /// </summary>
        /// <returns>実行可能な場合は true、実行不可能な場合は false。</returns>
        protected virtual bool CanExecuteInternal()
        {
            foreach (Operation operation in _operations)
            {
                if (operation.CanExecute()) { return true; }
            }

            return false;
        }

        /// <summary>
        /// 現在、オペレーションをロールバック可能かどうかを取得します。
        /// </summary>
        /// <returns>ロールバック可能な場合は true、ロールバック不可能な場合は false。</returns>
        protected virtual bool CanRollbackInternal()
        {
            foreach (Operation operation in _operations)
            {
                if (operation.CanRollback()) { return true; }
            }

            return false;
        }

        /// <summary>
        /// オペレーションを実行します。このメソッドは例外を処理しません。
        /// </summary>
        /// <returns>処理された場合は true、処理できなかった場合は false。</returns>
        protected virtual bool ExecuteInternal()
        {
            bool executed = true;

            foreach (Operation operation in _operations)
            {

                if (operation.IsExecuted) { continue; }

                executed = operation.Execute();
                if (!executed) { break; }

            }

            return executed;
        }

        /// <summary>
        /// オペレーションをロールバックします。このメソッドは例外を処理しません。
        /// </summary>
        /// <returns>処理された場合は true、処理できなかった場合は false。</returns>
        protected virtual bool RollbackInternal()
        {
            bool executed = true;

            for (int i = _operations.Count - 1; i >= 0; i--)
            {

                Operation operation = _operations[i];

                if (!operation.IsExecuted) { continue; }

                executed = operation.Rollback();
                if (!executed) { break; }

            }

            return executed;
        }

        /// <summary>
        /// トランザクションを破棄します。
        /// </summary>
        /// <param name="disposing">Dispose中の場合 true、Finalize中の場合 false。</param>
        private void Dispose(bool disposing)
        {
            if (_disposed) { return; }

            if (disposing)
            {
                foreach (Operation operation in _operations)
                {
                    if (!(operation is IDisposable)) { continue; }
                    (operation as IDisposable).Dispose();
                }
                _operations.Clear();
            }

            _disposed = true;
        }

        /// <summary>
        /// オペレーションコレクション読み取り専用インターフェイス
        /// </summary>
        public interface IOperationCollection : IEnumerable<Operation>, IEnumerable
        {
            /// <summary>
            /// オペレーション数を取得します。
            /// </summary>
            int Count { get; }
        }

        /// <summary>
        /// オペレーションコレクション
        /// </summary>
        public class OperationCollection : Collection<Operation>, IOperationCollection { }

        /// <summary>
        /// トランザクションスタックコレクション
        /// </summary>
        public class TransactionStackCollection : Collection<TransactionStack> { }
    }
}
