﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace NintendoWare.SoundFoundation.Operations
{
    /// <summary>
    /// オペレーションクラスの実装をサポートします。
    /// </summary>
    public abstract class OperationImpl : Operation
    {
        private bool _executed = false;

        /// <summary>
        /// オペレーションが実行されたかどうかを取得します。
        /// </summary>
        public override bool IsExecuted
        {
            get { return _executed; }
        }

        /// <summary>
        /// オペレーションを実行します。
        /// </summary>
        /// <returns>処理された場合は true、処理できなかった場合は false。</returns>
        public sealed override bool Execute()
        {
            if (!CanExecute()) { return false; }

            if (!ExecuteInternal()) { return false; }
            _executed = true;

            return true;
        }

        /// <summary>
        /// オペレーションをロールバックします。
        /// </summary>
        /// <returns>処理された場合は true、処理できなかった場合は false。</returns>
        public sealed override bool Rollback()
        {
            if (!CanRollback()) { return false; }

            if (!RollbackInternal()) { return false; }
            _executed = false;

            return true;
        }

        /// <summary>
        /// 現在、オペレーションを実行可能かどうかを取得します。
        /// </summary>
        /// <returns>実行可能な場合は true、実行不可能な場合は false。</returns>
        public sealed override bool CanExecute()
        {
            return CanExecuteInternal() & !_executed;
        }

        /// <summary>
        /// 現在、オペレーションをロールバック可能かどうかを取得します。
        /// </summary>
        /// <returns>ロールバック可能な場合は true、ロールバック不可能な場合は false。</returns>
        public sealed override bool CanRollback()
        {
            return CanRollbackInternal() & _executed;
        }

        /// <summary>
        /// オペレーションを実行します。
        /// </summary>
        /// <returns>処理された場合は true、処理できなかった場合は false。</returns>
        protected virtual bool ExecuteInternal()
        {
            return CanExecute();
        }

        /// <summary>
        /// オペレーションをロールバックします。
        /// </summary>
        /// <returns>処理された場合は true、処理できなかった場合は false。</returns>
        protected virtual bool RollbackInternal()
        {
            return CanRollback();
        }

        /// <summary>
        /// 現在、オペレーションを実行可能かどうかを取得します。
        /// </summary>
        /// <returns>実行可能な場合は true、実行不可能な場合は false。</returns>
        protected virtual bool CanExecuteInternal()
        {
            return true;
        }

        /// <summary>
        /// 現在、オペレーションをロールバック可能かどうかを取得します。
        /// </summary>
        /// <returns>ロールバック可能な場合は true、ロールバック不可能な場合は false。</returns>
        protected virtual bool CanRollbackInternal()
        {
            return true;
        }
    }
}
