﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace NintendoWare.SoundFoundation.Operations
{
    /// <summary>
    /// オペレーションの履歴を管理します。
    /// </summary>
    public class OperationHistory
    {
        private TransactionStack _stack = new TransactionStack();  // トランザクションスタック
        private UserTransaction _uncompletedTransaction = null;                    // 現在作成中のトランザクション

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public OperationHistory()
        {
            _stack.ItemsChanged += OnItemsChangedInternal;
            _stack.DirtyChanged += OnDirtyChangedInternal;
            _stack.CurrentItemChanged += OnCurrentItemChangedInternal;
            _stack.SavedItemChanged += OnSavedItemChangedInternal;
        }

        /// <summary>
        /// トランザクションスタックが変更されたときに発生します。
        /// </summary>
        public event EventHandler ItemsChanged;

        /// <summary>
        /// Dirty 状態が変更されたときに発生します。
        /// </summary>
        public event EventHandler DirtyChanged;

        /// <summary>
        /// 現在位置のトランザクションが変更されたときに発生します。
        /// </summary>
        public event EventHandler CurrentItemChanged;

        /// <summary>
        /// 保存位置のトランザクションが変更されたときに発生します。
        /// </summary>
        public event EventHandler SavedItemChanged;

        /// <summary>
        /// ドキュメントが編集されているかどうかを調べます。
        /// </summary>
        public bool IsDirty
        {
            get { return _stack.IsDirty; }
        }

        /// <summary>
        /// トランザクションの IEnumerator を取得します。
        /// </summary>
        public IEnumerable<Transaction> Transactions
        {
            get { return _stack.AllItems; }
        }

        /// <summary>
        /// Undo 対象のトランザクションを取得します。
        /// </summary>
        public Transaction UndoTransaction
        {
            get { return _stack.UndoItem; }
        }

        /// <summary>
        /// Redo 対象のトランザクションを取得します。
        /// </summary>
        public Transaction RedoTransaction
        {
            get { return _stack.RedoItem; }
        }

        /// <summary>
        /// 保存位置のトランザクションを取得します。
        /// </summary>
        public Transaction SavedTransaction
        {
            get { return _stack.SavedItem; }
        }

        /// <summary>
        /// オペレーションを追加します。
        /// BeginTransaction と EndTransaction の間に挟まった場合は、
        /// 作成中のトランザクションに指定オペレーションを追加し、スタックには追加しません。
        /// それ以外の場合は、指定オペレーションのみをもつ新しいトランザクションを作成しスタックに追加します。
        /// </summary>
        /// <param name="operation">オペレーション</param>
        public void AddOperation(Operation operation)
        {
            if (null == operation) { throw new ArgumentNullException("operation"); }

            if (null != _uncompletedTransaction)
            {
                _uncompletedTransaction.Operations.Add(operation);
                return;
            }

            Transaction transaction;

            if (operation is Transaction)
            {
                transaction = operation as Transaction;
            }
            else
            {
                transaction = new UserTransaction();
                (transaction as UserTransaction).Operations.Add(operation);
            }

            AddTransactionInternal(transaction);
        }

        /// <summary>
        /// トランザクションを追加します。
        /// BeginTransaction と EndTransaction の間に挟むことはできません。
        /// </summary>
        /// <param name="transaction"></param>
        public void AddTransaction(Transaction transaction)
        {
            if (null == transaction) { throw new ArgumentNullException("transaction"); }

            if (null != _uncompletedTransaction)
            {
                throw new Exception("InternalError : must not add transaction to transaction.");
            }

            AddTransactionInternal(transaction);
        }

        /// <summary>
        /// 新しいトランザクションの作成を開始します。
        /// </summary>
        public void BeginTransaction()
        {
            BeginTransaction(new UserTransaction());
        }

        /// <summary>
        /// 既存のトランザクションを使ってトランザクションの作成を開始します。
        /// </summary>
        /// <param name="transaction"></param>
        public void BeginTransaction(UserTransaction transaction)
        {
            if (null != _uncompletedTransaction) { throw new Exception("InternalError : failed to begin transaction."); }

            _uncompletedTransaction = transaction;
        }

        /// <summary>
        /// トランザクションの作成を終了し、スタックに追加します。
        /// </summary>
        public void EndTransaction()
        {
            if (null == _uncompletedTransaction) { return; }

            if (0 < _uncompletedTransaction.Operations.Count)
            {
                AddTransactionInternal(_uncompletedTransaction);
            }

            _uncompletedTransaction = null;
        }

        /// <summary>
        /// トランザクションの作成を中止します。
        /// </summary>
        public void CancelTransaction()
        {
            if (null != _uncompletedTransaction)
            {
                _uncompletedTransaction.Rollback();
            }

            _uncompletedTransaction = null;
        }

        /// <summary>
        /// オペレーションの履歴をクリアします。
        /// </summary>
        public void Clear()
        {
            _stack.Clear();
        }

        /// <summary>
        /// 保存位置を現在位置にリセットします。
        /// </summary>
        public void ResetSavedTransaction()
        {
            _stack.ResetSavedItem();
        }

        /// <summary>
        /// Undo します。
        /// </summary>
        /// <returns>成功した場合は true、中止または失敗した場合は false。</returns>
        public bool Undo()
        {
            if (null == _stack.UndoItem) { return false; }

            if (!_stack.UndoItem.IsInterlocked)
            {
                return _stack.Undo();
            }
            else
            {
                return InterlockedUndo(_stack.UndoItem);
            }
        }

        /// <summary>
        /// Redo します。
        /// </summary>
        /// <returns>成功した場合は true、中止または失敗した場合は false。</returns>
        public bool Redo()
        {
            if (null == _stack.RedoItem) { return false; }

            if (!_stack.RedoItem.IsInterlocked)
            {
                return _stack.Redo();
            }
            else
            {
                return InterlockedRedo(_stack.RedoItem);
            }
        }

        /// <summary>
        /// Undo できるかどうかを調べます。
        /// </summary>
        /// <returns>Undo できる場合は true、Undo できない場合はfalse。</returns>
        public bool CanUndo()
        {
            if (null == _stack.UndoItem) { return false; }

            if (!_stack.UndoItem.IsInterlocked)
            {
                return _stack.CanUndo();
            }
            else
            {
                return CanInterlockedUndo(_stack.UndoItem);
            }
        }

        /// <summary>
        /// Redo できるかどうかを調べます。
        /// </summary>
        /// <returns>Redo できる場合は true、Redo できない場合はfalse。</returns>
        public bool CanRedo()
        {
            if (null == _stack.RedoItem) { return false; }

            if (!_stack.RedoItem.IsInterlocked)
            {
                return _stack.CanRedo();
            }
            else
            {
                return CanInterlockedRedo(_stack.RedoItem);
            }
        }

        /// <summary>
        /// トランザクションスタックが変更されたときに実行されます。
        /// </summary>
        /// <param name="e">イベントパラメータ。</param>
        protected virtual void OnItemsChanged(EventArgs e)
        {
            if (null == e) { throw new ArgumentNullException("e"); }

            if (null != ItemsChanged)
            {
                ItemsChanged(this, e);
            }
        }

        /// <summary>
        /// Dirty 状態が変更されたときに実行されます。
        /// </summary>
        /// <param name="e">イベントパラメータ。</param>
        protected virtual void OnDirtyChanged(EventArgs e)
        {
            if (null == e) { throw new ArgumentNullException("e"); }

            if (null != DirtyChanged)
            {
                DirtyChanged(this, e);
            }
        }

        /// <summary>
        /// 現在位置のトランザクションが変更されたときに実行されます。
        /// </summary>
        /// <param name="e">イベントパラメータ。</param>
        protected virtual void OnCurrentItemChanged(EventArgs e)
        {
            if (null == e) { throw new ArgumentNullException("e"); }

            if (null != CurrentItemChanged)
            {
                CurrentItemChanged(this, e);
            }
        }

        /// <summary>
        /// 保存位置のトランザクションが変更されたときに実行されます。
        /// </summary>
        /// <param name="e">イベントパラメータ。</param>
        protected virtual void OnSavedItemChanged(EventArgs e)
        {
            if (null == e) { throw new ArgumentNullException("e"); }

            if (null != SavedItemChanged)
            {
                SavedItemChanged(this, e);
            }
        }

        /// <summary>
        /// トランザクションを追加します。
        /// </summary>
        /// <param name="transaction"></param>
        private void AddTransactionInternal(Transaction transaction)
        {
            if (null == transaction) { throw new ArgumentNullException("transaction"); }
            _stack.Push(transaction);
        }

        private bool InterlockedUndo(Transaction transaction)
        {
            if (null == transaction) { throw new ArgumentNullException("transaction"); }
            if (!transaction.IsExecuted) { return false; }
            if (!transaction.IsInterlocked) { return false; }

            foreach (TransactionStack stack in transaction.Stacks)
            {

                while (stack.UndoItem != transaction)
                {
                    if (!stack.Undo()) { throw new Exception("InternalError : failed to interlocked undo."); }
                }

            }

            return true;
        }

        private bool InterlockedRedo(Transaction transaction)
        {
            if (null == transaction) { throw new ArgumentNullException("transaction"); }
            if (transaction.IsExecuted) { return false; }
            if (!transaction.IsInterlocked) { return false; }

            foreach (TransactionStack stack in transaction.Stacks)
            {

                while (stack.RedoItem != transaction)
                {
                    if (!stack.Redo()) { throw new Exception("InternalError : failed to interlocked redo."); }
                }

            }

            return true;
        }

        private bool CanInterlockedUndo(Transaction transaction)
        {
            if (null == transaction) { throw new ArgumentNullException("transaction"); }
            if (!transaction.IsExecuted) { return false; }
            if (!transaction.IsInterlocked) { return false; }

            foreach (TransactionStack stack in transaction.Stacks)
            {
                if (!CanUndo(stack.UndoItems, transaction)) { return false; }
            }

            return true;
        }

        private bool CanInterlockedRedo(Transaction transaction)
        {
            if (null == transaction) { throw new ArgumentNullException("transaction"); }
            if (transaction.IsExecuted) { return false; }
            if (!transaction.IsInterlocked) { return false; }

            foreach (TransactionStack stack in transaction.Stacks)
            {
                if (!CanRedo(stack.RedoItems, transaction)) { return false; }
            }

            return true;
        }

        private bool CanUndo(IEnumerable<Transaction> transactions, Transaction target)
        {
            if (null == transactions) { throw new ArgumentNullException("transactions"); }
            if (null == target) { throw new ArgumentNullException("target"); }

            foreach (Transaction transaction in transactions)
            {
                if (target == transaction) { return true; }
                if (!transaction.CanRollback()) { return false; }
            }

            throw new Exception("InternalError : target transaction not found in transaction stack.");
        }

        private bool CanRedo(IEnumerable<Transaction> transactions, Transaction target)
        {
            if (null == transactions) { throw new ArgumentNullException("transactions"); }
            if (null == target) { throw new ArgumentNullException("target"); }

            foreach (Transaction transaction in transactions)
            {
                if (target == transaction) { return true; }
                if (!transaction.CanExecute()) { return false; }
            }

            throw new Exception("InternalError : target transaction not found in transaction stack.");
        }

        /// <summary>
        /// トランザクションスタックが変更されたときに実行されます。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">イベントパラメータ。</param>
        private void OnItemsChangedInternal(object sender, EventArgs e)
        {
            OnItemsChanged(e);
        }

        /// <summary>
        /// Dirty 状態が変更されたときに実行されます。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">イベントパラメータ。</param>
        private void OnDirtyChangedInternal(object sender, EventArgs e)
        {
            OnDirtyChanged(e);
        }

        /// <summary>
        /// 現在位置のトランザクションが変更されたときに実行されます。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">イベントパラメータ。</param>
        private void OnCurrentItemChangedInternal(object sender, EventArgs e)
        {
            OnCurrentItemChanged(e);
        }

        /// <summary>
        /// 保存位置のトランザクションが変更されたときに実行されます。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">イベントパラメータ。</param>
        private void OnSavedItemChangedInternal(object sender, EventArgs e)
        {
            OnSavedItemChanged(e);
        }
    }
}
