﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.Linq;

namespace NintendoWare.SoundFoundation.Core
{
    /// <summary>
    /// 複数発行されたCollectionChangedイベントを溜めこみ、一つにまとめて発行し直すクラスです。
    ///
    /// イベントをまとめるルール
    /// 1. イベントが届いたときに、溜めこんでいなかったらイベントを溜める
    /// 2. 次に違う種類(Add > Remove)にイベントの種類が届いたとき or 同じ種類でも条件に該当しなかったときには溜めこんだイベントを発行する
    /// 3. 届いたイベントは溜めこむ
    /// 4. 2.に戻る
    ///
    /// * 追加の場合
    /// 同じ場所(index)、次の場所に追加するイベントはまとめる
    ///
    /// * 削除の場合
    /// 同じ場所から削除するイベントはまとめる
    /// </summary>
    public class CollectionChangedEventCompressor
    {
        private bool enabled = false;
        private NotifyCollectionChangedAction? action = null;
        private int startingIndex = -1;
        private List<object> items = null;

        public void Begin()
        {
            this.enabled = true;
        }

        public NotifyCollectionChangedEventArgs End()
        {
            if (this.enabled == false)
            {
                return null;
            }

            this.enabled = false;
            return CreateEventArgs();
        }

        public IEnumerable<NotifyCollectionChangedEventArgs> Push(NotifyCollectionChangedEventArgs e)
        {
            if (this.enabled == false)
            {
                yield return e;
                yield break;
            }

            if (this.action == null)
            {
                var args = Setup(e);
                if (args != null)
                {
                    yield return args;
                }
            }
            else
            {
                if (this.action == e.Action)
                {
                    switch (e.Action)
                    {
                        case NotifyCollectionChangedAction.Add:
                            var index = e.NewStartingIndex;
                            if (index >= this.startingIndex && index <= this.startingIndex + this.items.Count)
                            {
                                index -= this.startingIndex;
                                this.items.InsertRange(index, e.NewItems.Cast<object>());
                            }
                            else
                            {
                                yield return CreateEventArgs();
                                var args = Setup(e);
                                if (args != null)
                                {
                                    yield return args;
                                }
                            }
                            break;

                        case NotifyCollectionChangedAction.Remove:
                            foreach (var item in e.OldItems)
                            {
                                this.items.Add(item);
                            }
                            break;

                        //以下のイベントの圧縮に未対応です。
                        case NotifyCollectionChangedAction.Move:
                        case NotifyCollectionChangedAction.Replace:
                        case NotifyCollectionChangedAction.Reset:
                            yield return CreateEventArgs();
                            yield return e;
                            break;
                    }
                }
                else
                {
                    yield return CreateEventArgs();
                    var args = Setup(e);
                    if (args != null)
                    {
                        yield return args;
                    }
                }
            }

            yield break;
        }

        private NotifyCollectionChangedEventArgs Setup(NotifyCollectionChangedEventArgs e)
        {
            switch (e.Action)
            {
                case NotifyCollectionChangedAction.Add:
                    this.startingIndex = e.NewStartingIndex;
                    this.items = new List<object>();
                    foreach (var item in e.NewItems)
                    {
                        this.items.Add(item);
                    }
                    break;

                case NotifyCollectionChangedAction.Remove:
                    this.startingIndex = e.OldStartingIndex;
                    this.items = new List<object>();
                    foreach (var item in e.OldItems)
                    {
                        this.items.Add(item);
                    }
                    break;

                case NotifyCollectionChangedAction.Move:
                case NotifyCollectionChangedAction.Replace:
                case NotifyCollectionChangedAction.Reset:
                    return e;
            }

            this.action = e.Action;
            return null;
        }

        private NotifyCollectionChangedEventArgs CreateEventArgs()
        {
            if (this.items == null)
            {
                return null;
            }

            var args = new NotifyCollectionChangedEventArgs(this.action.Value, this.items, this.startingIndex);

            this.action = null;
            this.startingIndex = -1;
            this.items = null;

            return args;
        }
    }
}
