﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Core.Reflection
{
    using System;
    using System.Collections;
    using System.Collections.Generic;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// Type クラスの拡張メソッドを定義します。
    /// </summary>
    public static class TypeEx
    {
        /// <summary>
        /// 指定された型が構造体かどうかを調べます。
        /// </summary>
        /// <param name="type">型を指定します。</param>
        /// <returns>構造体の場合は true、それ以外の場合は false を返します。</returns>
        public static bool IsStruct(this Type type)
        {
            Ensure.Argument.NotNull(type);
            return type.IsValueType && !type.IsPrimitive && !type.IsEnum;
        }

        /// <summary>
        /// 指定された型が列挙可能かどうかを調べます。
        /// </summary>
        /// <param name="type">型を指定します。</param>
        /// <returns>列挙可能な場合は true、不可能な場合は false を返します。</returns>
        public static bool IsEnumerable(this Type type)
        {
            Ensure.Argument.NotNull(type);
            return type.HasInterface(typeof(IEnumerable));
        }

        /// <summary>
        /// 指定された型の要素型を取得します。
        /// </summary>
        /// <param name="type">型を指定します。</param>
        /// <returns>
        /// 指定された型が内包する要素を持つことができる場合は要素の型を、
        /// それ以外の場合は null を返します。</returns>
        public static Type GetElementTypeEx(this Type type)
        {
            Ensure.Argument.NotNull(type);

            if (!IsEnumerable(type))
            {
                return null;
            }

            // 配列の要素の型を取得します。
            Type elementType = type.GetElementType();

            if (elementType != null)
            {
                return elementType;
            }

            // ジェネリックなIEnumerable<T>の型を取得します。
            if (type.IsGenericType &&
                type.GetGenericTypeDefinition() == typeof(IEnumerable<>))
            {
                return type.GetGenericArguments()[0];
            }

            foreach (Type interfaceType in type.GetInterfaces())
            {
                if (!interfaceType.IsGenericType)
                {
                    continue;
                }

                if (interfaceType.GetGenericTypeDefinition() != typeof(IEnumerable<>))
                {
                    continue;
                }

                return interfaceType.GetGenericArguments()[0];
            }

            return typeof(object);
        }
    }
}
