﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Core.Parameters
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel;
    using System.Linq;
    using System.Security.Cryptography;
    using System.Text;
    using ToolDevelopmentKit;

    /// <summary>
    /// パラメータの値を格納します。
    /// </summary>
    public abstract class ParameterValue<TValue> : IParameterValue
    {
        private const string NullString = "null";

        private readonly HashSet<object> attributes = new HashSet<object>();

        private TValue value;
        private HashCode hashCode = HashCode.Empty;

        //-----------------------------------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ParameterValue() { }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="value">パラメータの値を指定します。</param>
        public ParameterValue(TValue value)
        {
            this.RegisterEventListener(value);
            this.value = value;
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 値が変更されると発生します。
        /// </summary>
        public event ParameterEventHandler ValueChanged;

        //-----------------------------------------------------------------

        /// <summary>
        /// パラメータ値を取得または設定します。
        /// </summary>
        public virtual TValue Value
        {
            get { return this.value; }
            set
            {
                Type valueType = typeof(TValue);

                // 参照型 or 値型の値が同じ場合は処理しません。
                if (valueType.IsClass || valueType.IsInterface || valueType.IsByRef)
                {
                    if ((object)value == (object)this.value) { return; }
                }
                else
                {
                    Assertion.Operation.True(this.value != null);
                    if (value.Equals(this.value)) { return; }
                }

                this.UnregisterEventListener(this.value);
                this.RegisterEventListener(value);

                this.value = (TValue)value;

                OnValueChanged(new ParameterEventArgs(string.Empty, this));
            }
        }

        /// <summary>
        /// パラメータの属性コレクションを取得します。
        /// </summary>
        public ICollection<object> Attributes
        {
            get
            {
                return this.attributes;
            }
        }

        /// <summary>
        /// パラメータ値を取得または設定します。
        /// </summary>
        object IConstParameterValue.Value
        {
            get { return this.Value; }
        }

        /// <summary>
        /// パラメータ値を取得または設定します。
        /// </summary>
        object IParameterValue.Value
        {
            get { return this.Value; }
            set
            {
                if (!(value is TValue)) { throw new ArgumentException("invalid value type"); }
                this.Value = (TValue)value;
            }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// パラメータ値のテキストを取得します。
        /// </summary>
        /// <returns></returns>
        public override string ToString()
        {
            if (null == this.value) { return NullString; }
            return this.value.ToString();
        }

        /// <summary>
        /// テキストからパラメータ値を設定します。
        /// </summary>
        /// <param name="text">テキストを指定します。</param>
        public void Parse(string text)
        {
            Ensure.Argument.NotNull(text);
            this.Value = this.ParseInternal(text);
        }

        /// <summary>
        /// テキストからパラメータ値を取得します。
        /// </summary>
        /// <param name="text">テキストを指定します。</param>
        public TValue ParseValue(string text)
        {
            Ensure.Argument.NotNull(text);
            return this.ParseInternal(text);
        }

        /// <summary>
        /// パラメータキーを含めたパラメータのハッシュコードを取得します。
        /// </summary>
        /// <param name="algorithm">ハッシュアルゴリズムを指定します。</param>
        /// <param name="key">パラメータキーを指定します。</param>
        /// <param name="filter">対象パラメータのフィルタを指定します。</param>
        /// <returns>ハッシュコードを返します。</returns>
        public HashCode GetParameterHashCode(
            HashAlgorithm algorithm,
            string key,
            Func<IParameterValue, bool> filter)
        {
            if (this.hashCode == HashCode.Empty)
            {
                this.hashCode = this.CreateHashCode(algorithm, key, filter);
            }

            var dynamicBytes = this.GetDynamicBytesForHashCode();

            if (dynamicBytes == null)
            {
                return this.hashCode;
            }

            // 静的なハッシュコードと動的なバイト列を連結します。
            var bytes = this.hashCode.Value.Concat(dynamicBytes);
            return new HashCode(algorithm.ComputeHash(bytes.ToArray()));
        }

        /// <summary>
        /// テキストからパラメータ値を取得します。
        /// </summary>
        /// <param name="text">テキストを指定します。</param>
        object IParameterValue.ParseValue(string text)
        {
            Ensure.Argument.NotNull(text);
            return this.ParseInternal(text);
        }

        /// <summary>
        /// パラメータ値を検証します。
        /// </summary>
        /// <returns>検証結果を返します。</returns>
        public ValidationResult Validate()
        {
            return this.ValidateInternal(this.value);
        }

        /// <summary>
        /// 指定した値を検証します。
        /// </summary>
        /// <param name="value">検証する値を指定します。</param>
        /// <returns>検証結果を返します。</returns>
        public ValidationResult ValidateValue(TValue value)
        {
            return this.ValidateInternal(value);
        }

        /// <summary>
        /// 指定した値を検証します。
        /// </summary>
        /// <param name="value">検証する値を指定します。</param>
        /// <returns>検証結果を返します。</returns>
        ValidationResult IParameterValue.ValidateValue(object value)
        {
            if (value != null && !(value is TValue))
            {
                return new ValidationResult(false, "invalid parameter value type.");
            }

            return this.ValidateInternal((TValue)value);
        }

        /// <summary>
        /// テキストからパラメータ値を取得します。
        /// </summary>
        /// <param name="text">テキストを指定します。</param>
        protected abstract TValue ParseInternal(string text);

        /// <summary>
        /// 指定した値を検証します。
        /// </summary>
        /// <param name="value">検証する値を指定します。</param>
        /// <returns>検証結果を返します。</returns>
        protected abstract ValidationResult ValidateInternal(TValue value);

        /// <summary>
        /// パラメータ値のハッシュコードを作成します。
        /// </summary>
        /// <param name="algorithm">ハッシュアルゴリズムを指定します。</param>
        /// <param name="key">パラメータキーを指定します。</param>
        /// <param name="filter">対象パラメータのフィルタを指定します。</param>
        /// <returns>作成したハッシュ値を返します。</returns>
        protected virtual HashCode CreateHashCode(
            HashAlgorithm algorithm,
            string key,
            Func<IParameterValue, bool> filter)
        {
            Ensure.Argument.NotNull(algorithm);
            Ensure.Argument.NotNull(key);

            // キーのバイト列と値のバイト列を連結してハッシュ計算します。
            var bytes = Encoding.Unicode.GetBytes(key).Concat(this.GetBytes());
            return new HashCode(algorithm.ComputeHash(bytes.ToArray()));
        }

        /// <summary>
        /// パラメータ値のハッシュ値用の動的に変化するバイト列を取得します。
        /// </summary>
        /// <returns>ハッシュ値用バイト列の列挙子を返します。</returns>
        protected virtual IEnumerable<byte> GetDynamicBytesForHashCode()
        {
            return null;
        }

        /// <summary>
        /// パラメータ値のバイト列を取得します。
        /// </summary>
        /// <returns>バイト列の列挙子を返します。</returns>
        protected abstract IEnumerable<byte> GetBytes();

        /// <summary>
        /// パラメータの値が変更されると発生します。
        /// </summary>
        /// <param name="e">イベントパラメータを指定します。</param>
        protected virtual void OnValueChanged(ParameterEventArgs e)
        {
            this.hashCode = HashCode.Empty;

            if (null != ValueChanged)
            {
                ValueChanged(this, e);
            }
        }

        /// <summary>
        /// 値のイベントに対してリスナー登録します。
        /// </summary>
        /// <param name="value">値を指定します。</param>
        private void RegisterEventListener(TValue value)
        {
            if (value == null) { return; }

            if (value is IParameterProvider)
            {
                (value as IParameterProvider).Parameters.ParameterValueChanged += OnParameterDictionaryValueChanged;
                return;
            }

            if (value is INotifyPropertyChanged)
            {
                (value as INotifyPropertyChanged).PropertyChanged += OnValuePropertyChanged;
                return;
            }
        }

        /// <summary>
        /// 値のイベントに対してリスナー登録を解除します。
        /// </summary>
        /// <param name="value">値を指定します。</param>
        private void UnregisterEventListener(TValue value)
        {
            if (value == null) { return; }

            if (value is IParameterProvider)
            {
                (value as IParameterProvider).Parameters.ParameterValueChanged -= OnParameterDictionaryValueChanged;
                return;
            }

            if (value is INotifyPropertyChanged)
            {
                (value as INotifyPropertyChanged).PropertyChanged -= OnValuePropertyChanged;
                return;
            }
        }

        /// <summary>
        /// ソースの値が変更されると発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元を指定します。</param>
        /// <param name="e">イベントパラメータを指定します。</param>
        private void OnValuePropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            this.OnValueChanged(new ParameterEventArgs(e.PropertyName, this));
        }

        /// <summary>
        /// パラメータ辞書の値が変更されると発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元を指定します。</param>
        /// <param name="e">イベントパラメータを指定します。</param>
        private void OnParameterDictionaryValueChanged(object sender, ParameterEventArgs e)
        {
            this.OnValueChanged(new ParameterEventArgs(string.Empty, this));
        }
    }
}
