﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Core.Parameters
{
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using System.Text;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// string 型のパラメータ値（ファイルパス）を格納します。
    /// </summary>
    public class FilePathParameterValue : TextParameterValue
    {
        public delegate ValidationResult FilePathParamenterValueValidator(string value);
        public static FilePathParamenterValueValidator Validator { get; set; }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public FilePathParameterValue()
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="value">パラメータの値を指定します。</param>
        public FilePathParameterValue(string value)
            : base(value)
        {
        }

        /// <summary>
        /// パラメータ値を取得または設定します。
        /// </summary>
        public override string Value
        {
            get { return base.Value; }
            set
            {
                Ensure.Argument.NotNull(value);

                // パス区切り文字を置換
                base.Value = value.Replace('/', '\\');
            }
        }

        /// <summary>
        /// 指定した値を検証します。
        /// </summary>
        /// <param name="value">検証する値を指定します。</param>
        /// <returns>検証結果を返します。</returns>
        protected override ValidationResult ValidateInternal(string value)
        {
            ValidationResult result;

            if (Validator != null)
            {
                result = Validator(value);
            }
            else
            {
                result = FilePathValidator.ValidateFilePath(value);
            }

            return result;
        }

        /// <summary>
        /// パラメータ値のハッシュ値用の動的に変化するバイト列を取得します。
        /// </summary>
        /// <returns>ハッシュ値用バイト列の列挙子を返します。</returns>
        protected override IEnumerable<byte> GetDynamicBytesForHashCode()
        {
            if (!File.Exists(this.Value))
            {
                return null;
            }

            return Encoding.Unicode.GetBytes(File.GetLastWriteTime(this.Value).ToString());
        }
    }
}
