﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Core.Diagnostics
{
    using System;
    using System.Collections;
    using System.Collections.Generic;
    using System.Collections.Specialized;
    using System.Diagnostics;
    using System.Linq;
    using System.Threading;
    using ToolDevelopmentKit;

    /// <summary>
    /// コンソールプロセスを操作します。
    /// </summary>
    public class ConsoleProcess : IDisposable
    {
        private readonly Dictionary<string, string> environmentVariables = new Dictionary<string, string>();
        private readonly object processLock = new object();

        private Process process = null;

        private string currentCommand = null;
        private AutoResetEvent commandStartingEvent = new AutoResetEvent(false);

        public event EventHandler<CommandLineEventArgs> OutputLineReceived;
        public event EventHandler<CommandLineEventArgs> ErrorLineReceived;

        //-----------------------------------------------------------------

        /// <summary>
        /// 環境変数の辞書を取得します。
        /// </summary>
        public Dictionary<string, string> EnvironmentVariables
        {
            get { return this.environmentVariables; }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// プロセスを開始します。
        /// </summary>
        /// <param name="workDirectoryPath">作業ディレクトリを指定します。</param>
        /// <returns>
        /// プロセスの起動に成功した場合は true、既に起動中の場合や起動に失敗した場合は false を返します。
        /// </returns>
        public bool Start(string workDirectoryPath)
        {
            if (this.process != null)
            {
                return false;
            }

            try
            {
                this.process = new Process();

                this.process.StartInfo.FileName = System.Environment.GetEnvironmentVariable("ComSpec");
                this.process.StartInfo.Arguments = "/K \"\"";
                this.process.StartInfo.UseShellExecute = false;
                this.process.StartInfo.CreateNoWindow = true;
                this.process.StartInfo.WorkingDirectory = workDirectoryPath;
                this.process.StartInfo.RedirectStandardInput = true;
                this.process.StartInfo.RedirectStandardOutput = true;
                this.process.StartInfo.RedirectStandardError = true;

                foreach (var item in this.EnvironmentVariables)
                {
                    this.process.StartInfo.EnvironmentVariables.Add(item.Key, item.Value);
                }

                this.process.Exited += OnProcessExited;
                this.process.OutputDataReceived += (sender, e) => this.OutputLine(e.Data);
                this.process.ErrorDataReceived += (sender, e) => this.OutputErrorLine(e.Data);

                this.commandStartingEvent.Reset();

                if (!this.process.Start())
                {
                    return false;
                }

                this.process.BeginOutputReadLine();
                this.process.BeginErrorReadLine();
            }
            catch
            {
                if (process != null)
                {
                    process.Dispose();
                }
                throw;
            }

            return true;
        }

        /// <summary>
        /// プロセスを終了します。
        /// </summary>
        public void Exit()
        {
            lock (processLock)
            {
                if (this.process == null)
                {
                    return;
                }

                this.WriteCommandLine("exit", false);
                while (this.process.WaitForExit(1000) == false) { }

                Ensure.Argument.NotNull(this.process);

                this.process = null;
                this.currentCommand = null;
            }
        }

        /// <summary>
        /// プロセスをキャンセルします。
        /// </summary>
        public void Cancel()
        {
            try
            {
                this.commandStartingEvent.Set();

                this.process.CancelErrorRead();
                this.process.CancelOutputRead();

                this.process.Kill();
            }
            catch
            {
            }
        }

        /// <summary>
        /// コマンドラインを出力します。
        /// </summary>
        /// <param name="line">コマンドライン文字列を指定します。</param>
        /// <param name="isWaitDone">コマンドの実行が開始されるまで待機します。</param>
        public void WriteCommandLine(string line, bool isWaitStarting)
        {
            Ensure.Argument.NotNull(line);

            this.currentCommand = line;
            this.commandStartingEvent.Reset();

            this.process.StandardInput.WriteLine(line);

            if (isWaitStarting)
            {
                this.commandStartingEvent.WaitOne();
            }
        }

        /// <summary>
        /// 保有するリソース破棄します。
        /// </summary>
        public void Dispose()
        {
            if (this.process != null)
            {
                this.process.Dispose();
                this.process = null;
            }

            if (this.commandStartingEvent != null)
            {
                this.commandStartingEvent.Close();
                this.commandStartingEvent = null;
            }

            GC.SuppressFinalize(this);
        }

        protected virtual void OnOutputLineReceived(CommandLineEventArgs e)
        {
            Assertion.Argument.NotNull(e);

            if (this.OutputLineReceived != null)
            {
                this.OutputLineReceived(this, e);
            }
        }

        private void OnErrorLineReceived(CommandLineEventArgs e)
        {
            Assertion.Argument.NotNull(e);

            if (this.ErrorLineReceived != null)
            {
                this.ErrorLineReceived(this, e);
            }
        }

        private void OutputLine(string line)
        {
            if (line == null)
            {
                return;
            }

            // 終了コマンドは出力しません。
            if (line.EndsWith(">exit"))
            {
                return;
            }

            string outputLine = line;

            if (this.currentCommand != null && line.EndsWith(this.currentCommand))
            {
                outputLine = this.currentCommand;
                this.currentCommand = null;
                this.commandStartingEvent.Set();
            }

            this.OutputLineReceived(this, new CommandLineEventArgs(outputLine));
        }

        private void OutputErrorLine(string line)
        {
            if (line == null || line.Length == 0)
            {
                return;
            }

            this.OnErrorLineReceived(new CommandLineEventArgs(line));
        }

        private void OnProcessExited(object sender, EventArgs e)
        {
            this.process = null;
            this.currentCommand = null;
        }
    }
}
