﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion.NintendoWareReport
{
    using System.IO;
    using System.Linq;
    using System.Xml;
    using NintendoWare.SoundFoundation.Projects;
    using ToolDevelopmentKit;

    public class BankReportExporter
    {
        private HtmlReportExporter exporter;

        public string Generator { get; set; }

        public void Export(TextWriter writer, BankService bankService, ListOutput instrumentListOutput)
        {
            Ensure.Argument.NotNull(writer);
            Ensure.Argument.NotNull(bankService);
            Ensure.Argument.NotNull(instrumentListOutput);

            try
            {
                // TODO : ★名前が決まったらTitleを変更。バンクレポートにする？
                this.exporter = new HtmlReportExporter()
                {
                    Title = string.Format("{0} Instrument List", Path.GetFileName(bankService.BankDocument.Resource.Key)),
                    Generator = this.Generator,
                };

                using (XmlTextWriter xmlWriter = new XmlTextWriter(writer))
                {
                    xmlWriter.Formatting = Formatting.Indented;

                    xmlWriter.WriteStartElement("html");
                    this.exporter.WriteHtmlHeader(xmlWriter);

                    {
                        xmlWriter.WriteStartElement("body");

                        this.exporter.WriteTitle(xmlWriter);
                        this.exporter.WriteStartBodyDiv(xmlWriter);

                        {
                            this.ExportItemLists(xmlWriter, bankService, instrumentListOutput);

                            this.exporter.WriteFooter(xmlWriter, bankService.BankDocument.Resource.Key);
                        }

                        this.exporter.WriteEndBodyDiv(xmlWriter);
                        xmlWriter.WriteFullEndElement();    // body
                    }

                    xmlWriter.WriteFullEndElement();    // html
                    xmlWriter.Flush();
                }
            }
            finally
            {
                this.exporter = null;
            }
        }

        private void ExportItemLists(
            XmlTextWriter writer, BankService bankService, ListOutput instrumentListOutput)
        {
            Assertion.Argument.NotNull(writer);
            Assertion.Argument.NotNull(bankService);
            Assertion.Argument.NotNull(instrumentListOutput);

            if (!instrumentListOutput.IsOutput)
            {
                return;
            }

            this.ExportItemList<Instrument>(
               writer, new InstrumentListInfoProvider(new BankReportTraits(), bankService, instrumentListOutput),
               new InstrumentInfoProvider(), 0);
        }

        private void ExportItemList<TItem>(
            XmlTextWriter writer, IListInfoProvider<TItem> listInfoProvider,
            IListItemInfoProvider<TItem> itemInfoProvider, int listIndex)
            where TItem : Component
        {
            Assertion.Argument.NotNull(writer);
            Assertion.Argument.NotNull(listInfoProvider);
            Assertion.Argument.NotNull(itemInfoProvider);

            // アイテムがない場合は、テーブルを作成しません。
            if (listInfoProvider.Items.FirstOrDefault() == null)
            {
                return;
            }

            {
                writer.WriteStartElement("h3");
                this.exporter.WriteShowButton(writer, "display", listIndex.ToString(), listInfoProvider.Caption);
                writer.WriteFullEndElement();
            }

            {
                writer.WriteStartElement("div");
                writer.WriteAttributeString("id", "display" + listIndex.ToString());

                this.exporter.WriteItemList<TItem>(writer, listInfoProvider, itemInfoProvider);

                writer.WriteFullEndElement();
            }
        }
    }
}
