﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Binarization
{
    using System;
    using System.Runtime.InteropServices;

    /// <summary>
    /// 実機アドレスクラスです。（TDK.ObjectBinarizationから移植）
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Pack = 1)]
    public struct Address
    {
        public static readonly Address MaxValue = new Address(UInt32.MaxValue);
        public static readonly Address MinValue = new Address(0);

        /// <summary>
        /// 構造体のサイズです。
        /// </summary>
        public const uint Size = sizeof(uint);

        /// <summary>
        /// 値です。
        /// </summary>
        private ulong value;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="address">アドレス値として設定する値です。</param>
        public Address(ulong address)
        {
            this.value = address;
        }

        /// <summary>
        /// Address から ulong への暗黙のキャストです。
        /// </summary>
        /// <param name="address">キャストをおこなう Address 変数です。</param>
        /// <returns>アドレスの ulong 値を返します。</returns>
        public static implicit operator ulong(Address address)
        {
            return address.value;
        }

        /// <summary>
        /// ulong から Address への暗黙のキャストです。
        /// </summary>
        /// <param name="address">キャストをおこなう int 値です。</param>
        /// <returns>Address の値を返します。</returns>
        public static implicit operator Address(ulong address)
        {
            return new Address(address);
        }

        /// <summary>
        /// 加算演算子のオーバーロードです。
        /// </summary>
        /// <param name="address1">演算対象の値１です。</param>
        /// <param name="address2">演算対象の値２です。</param>
        /// <returns>加算結果の Address 変数です。</returns>
        public static Address operator +(Address address1, Address address2)
        {
            return new Address(address1.value + address2.value);
        }

        /// <summary>
        /// 減算演算子のオーバーロードです。
        /// </summary>
        /// <param name="address1">演算対象の値１です。</param>
        /// <param name="address2">演算対象の値２です。</param>
        /// <returns>減算結果の Address 変数です。</returns>
        public static Address operator -(Address address1, Address address2)
        {
            return new Address(address1.value - address2.value);
        }
    }

}
