﻿namespace Opal.Windows
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using System.Windows;
    using Opal.ComponentModel;
    using Opal.Utilities;

    /// <summary>
    /// ウィンドウマネージャークラスです。
    /// </summary>
    public sealed class WindowManager : SynchronizableObject, IWindowManager
    {
        private static readonly string ManagerKeyString = Enum.GetName(typeof(ManagerKey), ManagerKey.Window);
        private readonly Dictionary<Type, WindowMaker> cache =
            new Dictionary<Type, WindowMaker>();

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public WindowManager()
        {
        }

        /// <summary>
        /// マネージャーのキーを取得します。
        /// </summary>
        public string Key
        {
            get
            {
                return ManagerKeyString;
            }
        }

        /// <summary>
        /// ウィンドウメーカーを追加します。
        /// </summary>
        /// <param name="maker">追加するウィンドウメーカーです。</param>
        public void AddWindowMaker(WindowMaker maker)
        {
            Debug.Assert(maker != null);

            lock (this.SyncRoot)
            {
                this.cache.Add(maker.WindowType, maker);
            }
        }

        /// <summary>
        /// ウィンドウメーカーを取得します。
        /// </summary>
        /// <typeparam name="TWindow">取得するメーカーのウィンドウのテンプレートの型です。</typeparam>
        /// <returns>メーカーのインスタンスを返します。</returns>
        public WindowMaker GetWindowMaker<TWindow>()
        {
            lock (this.SyncRoot)
            {
                return this.cache[typeof(TWindow)];
            }
        }

        /// <summary>
        /// ウィンドウメーカーを取得します。
        /// </summary>
        /// <returns>メーカーのインスタンスを返します。</returns>
        public IEnumerable<WindowMaker> GetWindowMakers()
        {
            foreach (var maker in this.cache.Values)
            {
                yield return maker;
            }
        }

        /// <summary>
        /// 全てクリアします。
        /// </summary>
        public void ClearAll()
        {
            this.cache.Clear();
        }
    }
}
