﻿namespace Opal.ViewModels
{
    using System.Windows.Input;
    using Opal.Commands;
    using Opal.Services;
    using Opal.Storages;

    /// <summary>
    /// ツールビューモデルクラスです。
    /// </summary>
    public abstract class ToolViewModel : PaneViewModel, ITool
    {
        private readonly ICommand closeCommand;
        private bool isSelected = false;
        private bool isVisible = true;
        private string contentId = string.Empty;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        protected ToolViewModel()
        {
            this.IsVisible = true;
            this.closeCommand =
                new RelayCommand(param => this.IsVisible = false, param => { return true; });
        }

        /// <summary>
        /// 推奨の配置を取得します。
        /// </summary>
        public virtual PaneLocation PreferredLocation
        {
            get
            {
                return PaneLocation.Right;
            }
        }

        /// <summary>
        /// 推奨の幅を取得します。
        /// </summary>
        public virtual double PreferredWidth
        {
            get
            {
                return 200;
            }
        }

        /// <summary>
        /// 推奨の高さを取得します。
        /// </summary>
        public virtual double PreferredHeight
        {
            get
            {
                return 200;
            }
        }

        /// <summary>
        /// 閉じる処理のコマンドを取得します。
        /// </summary>
        public ICommand CloseCommand
        {
            get
            {
                return this.closeCommand;
            }
        }

        /// <summary>
        /// コンテントIDを取得設定します。
        /// </summary>
        public string ContentId
        {
            get
            {
                return this.contentId;
            }

            internal set
            {
                this.contentId = value;
            }
        }

        /// <summary>
        /// 選択状態を取得設定します。
        /// </summary>
        public bool IsSelected
        {
            get
            {
                return this.isSelected;
            }

            set
            {
                this.SetProperty(ref this.isSelected, value);
            }
        }

        /// <summary>
        /// 可視状態を取得設定します。
        /// </summary>
        public bool IsVisible
        {
            get
            {
                return this.isVisible;
            }

            set
            {
                this.SetProperty(ref this.isVisible, value);
            }
        }

        /// <summary>
        /// アクティブなドキュメントが変更された時に実行されます。
        /// </summary>
        /// <param name="sender">送信元のオブジェクトです。</param>
        /// <param name="e">イベント引数です。</param>
        internal void CallActiveDocumentChanged(object sender, ActiveDocumentChangedEventArgs e)
        {
            this.OnActiveDocumentChanged(e);
        }

        /// <summary>
        /// ストレージが変更された時に実行されます。
        /// </summary>
        /// <param name="sender">送信元のオブジェクトです。</param>
        /// <param name="e">イベント引数です。</param>
        internal void CallStorageChanged(object sender, StorageChangedEventArgs e)
        {
            this.OnStorageChanged(e);
        }

        /// <summary>
        /// アクティブなドキュメントが変更された時に実行される処理です。
        /// </summary>
        /// <param name="e">イベント引数です。</param>
        protected virtual void OnActiveDocumentChanged(ActiveDocumentChangedEventArgs e)
        {
        }

        /// <summary>
        /// ストレージが変更された時に実行される処理です。
        /// </summary>
        /// <param name="e">イベント引数です。</param>
        protected virtual void OnStorageChanged(StorageChangedEventArgs e)
        {
        }
    }
}
