﻿namespace Opal.ViewModels
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Linq;
    using System.Reflection;
    using System.Text;
    using System.Threading.Tasks;
    using Opal.ComponentModel;

    /// <summary>
    /// 値変更通知ビューモデルクラスです。
    /// </summary>
    public class ObservableValueViewModel : ViewModel, INotifyPropertyValueChanged
    {
        private readonly List<INotifyPropertyValueChanged> properties =
            new List<INotifyPropertyValueChanged>();

        private bool isValueChanged = false;

        /// <summary>
        /// プロパティ値変更時のイベントです。
        /// </summary>
        internal event EventHandler<NotifyPropertyValueChangedEventArgs> PropertyValueChanged;

        /// <summary>
        /// 値が変更されたか判定します。
        /// </summary>
        public bool IsValueChanged
        {
            get
            {
                return this.isValueChanged;
            }

            protected set
            {
                this.SetProperty(ref this.isValueChanged, value, () => this.UpdateState());
            }
        }

        /// <summary>
        /// 値が変更されたか状態を更新します。
        /// </summary>
        public void UpdateState()
        {
            if (this.PropertyValueChanged != null)
            {
                this.PropertyValueChanged(this, new NotifyPropertyValueChangedEventArgs(this.IsValueChanged));
            }
        }

        /// <summary>
        /// 値変更通知機能付のプロパティを追加します。
        /// </summary>
        /// <param name="property">追加するプロパティのインスタンスです。</param>
        protected void AddProperty(INotifyPropertyValueChanged property)
        {
            Debug.Assert(property != null);

            lock (this.SyncRoot)
            {
                if (this.properties.Contains(property))
                {
                    return;
                }

                this.properties.Add(property);
                var notifyProperty = property as ObservableValueViewModel;
                if (notifyProperty != null)
                {
                    notifyProperty.PropertyValueChanged += this.CallPropertyValueChanged;
                }
            }
        }

        /// <summary>
        /// オブジェクト破棄の内部処理です。継承した先で固有の処理を実装します。
        /// </summary>
        protected override void DisposeInternal()
        {
            lock (this.SyncRoot)
            {
                foreach (var property in this.properties)
                {
                    var notifyProperty = property as ObservableValueViewModel;
                    if (notifyProperty != null)
                    {
                        notifyProperty.PropertyValueChanged -= this.CallPropertyValueChanged;
                    }
                }
            }
        }

        private void CallPropertyValueChanged(object sender, NotifyPropertyValueChangedEventArgs e)
        {
            if (e.IsValueChanged)
            {
                this.IsValueChanged |= e.IsValueChanged;
            }
            else
            {
                if (this.properties.FindAll(param => param.IsValueChanged == false).Count == this.properties.Count)
                {
                    this.IsValueChanged = false;
                }
            }
        }
    }
}
