﻿namespace Opal.Utilities
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Globalization;
    using System.IO;
    using System.Text;

    /// <summary>
    /// 引数パーサーです。
    /// </summary>
    public class ArgumentParser
    {
        // 引数
        public readonly List<string> Arguments = new List<string>();

        // プログラムオプション辞書
        private Dictionary<string, string> programOptionNameDic = new Dictionary<string, string>();
        private Dictionary<string, OptionValueMode> programOptionValueDic = new Dictionary<string, OptionValueMode>();

        // ファイルオプション辞書
        private Dictionary<string, string> fileOptionNameDic = new Dictionary<string, string>();
        private Dictionary<string, OptionValueMode> fileOptionValueDic = new Dictionary<string, OptionValueMode>();

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ArgumentParser()
        {
            string[] args = Environment.GetCommandLineArgs();
            for (int i = 1; i < args.Length; ++i)
            {
                this.Arguments.Add(args[i]);
            }
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="args">解析する引数配列です。</param>
        public ArgumentParser(string[] args)
        {
            Debug.Assert(args != null);

            foreach (var arg in args)
            {
                this.Arguments.Add(arg);
            }
        }

        /// <summary>
        /// オプション値のモードです。
        /// </summary>
        public enum OptionValueMode
        {
            /// <summary>
            /// 必須オプションです。
            /// </summary>
            Require,

            /// <summary>
            /// 省略可能オプションです。
            /// </summary>
            Option,

            /// <summary>
            /// オプション無し。
            /// </summary>
            None,
        }

        /// <summary>
        /// 登録していないオプションを許容するかどうかです。
        /// true を指定した場合、登録されていないオプションが含まれた状態で Parse() を行っても
        /// 例外を送出しません。
        /// </summary>
        public bool IsTolerant
        {
            get;
            set;
        }

        /// <summary>
        /// プログラムオプションを登録します。
        /// </summary>
        /// <param name="name">名前です。</param>
        /// <param name="shortName">名前の省略形です。</param>
        /// <param name="valueMode">オプション値のモードです。</param>
        public void RegisterProgramOption(string name, string shortName, OptionValueMode valueMode)
        {
            Debug.Assert(name != null);
            Debug.Assert(shortName != null);
            Debug.Assert(name.StartsWith("--") && shortName.StartsWith("-"));

            this.programOptionNameDic.Add(shortName, name);
            this.programOptionValueDic.Add(name, valueMode);
        }

        /// <summary>
        /// ファイルオプションを登録します。
        /// </summary>
        /// <param name="name">名前です。</param>
        /// <param name="shortName">名前の省略形です。</param>
        /// <param name="valueMode">オプション値のモードです。</param>
        public void RegisterFileOption(string name, string shortName, OptionValueMode valueMode)
        {
            Debug.Assert(name != null);
            Debug.Assert(shortName != null);
            Debug.Assert(name.StartsWith("--") && shortName.StartsWith("-"));

            this.fileOptionNameDic.Add(shortName, name);
            this.fileOptionValueDic.Add(name, valueMode);
        }

        /// <summary>
        /// パースを行います。
        /// @ による複数 ProgramOption 対応は、ParseMulti() を別途作成します。
        /// </summary>
        /// <returns>結果を返します。</returns>
        public ProgramOption Parse()
        {
            // 引数リストの構築
            List<string> argList = new List<string>();

            foreach (string arg in this.Arguments)
            {
                if (arg.StartsWith("@"))
                {
                    // arg.Length == 1 でプログラムオプション分割に対応可能
                    this.ParseArgFile(arg.Substring(1), argList);
                }
                else
                {
                    argList.Add(arg);
                }
            }

            // 引数リストの解析
            return this.AnalyzeArgumentList(argList);
        }

        ////---------------------------------------------------------------------

        // 引数ファイルのパース
        // 行を跨ぐダブルクォーテーションエスケープは非サポート
        private void ParseArgFile(string argFilePath, List<string> argList)
        {
            char[] delimiter = new char[] { ' ', '\t', '\r', '\n' };

            // システムのロケールに対応するには、起動時にロケールを退避する必要がある
            string[] lines = File.ReadAllLines(
                argFilePath,
                Encoding.GetEncoding(CultureInfo.CurrentCulture.TextInfo.ANSICodePage));

            foreach (string originalLine in lines)
            {
                // トリムしておく
                string line = originalLine.Trim();

                // コメントの読み飛ばし
                if (line.StartsWith("#"))
                {
                    continue;
                }

                while (line.Length != 0)
                {
                    if (line.StartsWith("-"))
                    {
                        // オプション
                        int quotIndex = line.IndexOf('"');
                        int delIndex = line.IndexOfAny(delimiter);
                        if (delIndex == -1)
                        {
                            if (quotIndex == -1)
                            {
                                argList.Add(line);
                                line = string.Empty;
                            }
                            else
                            {
                                argList.Add(line.Replace("\"", string.Empty));
                                line = string.Empty;
                            }
                        }
                        else if ((quotIndex != -1) && (quotIndex < delIndex))
                        {
                            // ダブルクォーテーションをエスケープしてオプション取り出し
                            int index = line.IndexOf('"', quotIndex + 1);
                            if (index == -1)
                            {
                                ExceptionUtility.Throw(
                                    string.Format(Resources.Errors.ArgumentParserDoubleQuotNotClose, originalLine));
                            }

                            argList.Add(line.Substring(0, index).Replace("\"", string.Empty));
                            line = line.Substring(index + 1).TrimStart();
                        }
                        else
                        {
                            // オプション取り出し
                            argList.Add(line.Substring(0, delIndex));
                            line = line.Substring(delIndex + 1).TrimStart();
                        }
                    }
                    else
                    {
                        // パス
                        if (line.StartsWith("\""))
                        {
                            // ダブルクォーテーションをエスケープしてパス取り出し
                            line = line.Substring(1);
                            int index = line.IndexOf('"');
                            if (index == -1)
                            {
                                ExceptionUtility.Throw(
                                    string.Format(Resources.Errors.ArgumentParserDoubleQuotNotClose, originalLine));
                            }

                            argList.Add(line.Substring(0, index));
                            line = line.Substring(index + 1).TrimStart();
                        }
                        else
                        {
                            // パス取り出し
                            int index = line.IndexOfAny(delimiter);
                            if (index == -1)
                            {
                                argList.Add(line);
                                line = string.Empty;
                            }
                            else
                            {
                                argList.Add(line.Substring(0, index));
                                line = line.Substring(index + 1).TrimStart();
                            }
                        }
                    }
                }
            }
        }

        // 引数リストの解析
        private ProgramOption AnalyzeArgumentList(List<string> argList)
        {
            ProgramOption programOption = new ProgramOption();
            FileOption fileOption = null;
            bool programMode = true;

            foreach (string arg in argList)
            {
                if (programMode)
                {
                    // プログラムオプションの解析
                    if (arg.StartsWith("-"))
                    {
                        var option = this.AnalyzeArgumentOption(arg, true);
                        if (option != null)
                        {
                            programOption.Options.Add(option);
                        }

                        continue;
                    }
                    else
                    {
                        programMode = false;
                    }
                }

                // ファイルオプションの解析
                if (arg.StartsWith("-"))
                {
                    var option = this.AnalyzeArgumentOption(arg, false);
                    if (option != null)
                    {
                        fileOption.Options.Add(option);
                    }
                }
                else
                {
                    fileOption = new FileOption(arg);
                    programOption.FileOptions.Add(fileOption);
                }
            }

            return programOption;
        }

        // 引数オプションの解析
        private ArgumentOption AnalyzeArgumentOption(string arg, bool isProgramOption)
        {
            string optName = arg;
            string optValue = null;

            // 値の取り出し
            int index = arg.IndexOf('=');
            if (index != -1)
            {
                optName = arg.Substring(0, index);
                optValue = arg.Substring(index + 1);
            }

            // ショートネームを通常名に変換
            if (!optName.StartsWith("--"))
            {
                if (isProgramOption)
                {
                    if (!this.programOptionNameDic.ContainsKey(optName))
                    {
                        if (this.IsTolerant)
                        {
                            return null;
                        }

                        ExceptionUtility.Throw(
                            string.Format(Resources.Errors.ArgumentParserUnknownProgramOption, arg));
                    }

                    optName = this.programOptionNameDic[optName];
                }
                else
                {
                    if (!this.fileOptionNameDic.ContainsKey(optName))
                    {
                        if (this.IsTolerant)
                        {
                            return null;
                        }

                        ExceptionUtility.Throw(
                            string.Format(Resources.Errors.ArgumentParserUnknownFileOption, arg));
                    }

                    optName = this.fileOptionNameDic[optName];
                }
            }

            // 値のモード取得
            OptionValueMode valueMode;
            if (isProgramOption)
            {
                if (!this.programOptionValueDic.ContainsKey(optName))
                {
                    if (this.IsTolerant)
                    {
                        return null;
                    }

                    ExceptionUtility.Throw(
                        string.Format(Resources.Errors.ArgumentParserUnknownProgramOption, arg));
                }

                valueMode = this.programOptionValueDic[optName];
            }
            else
            {
                if (!this.fileOptionValueDic.ContainsKey(optName))
                {
                    if (this.IsTolerant)
                    {
                        return null;
                    }

                    ExceptionUtility.Throw(
                        string.Format(Resources.Errors.ArgumentParserUnknownFileOption, arg));
                }

                valueMode = this.fileOptionValueDic[optName];
            }

            // 値のエラーチェック
            if ((valueMode == OptionValueMode.Require) && (optValue == null))
            {
                ExceptionUtility.Throw(
                    string.Format(Resources.Errors.ArgumentParserRequireOptionValue, arg));
            }

            if ((valueMode == OptionValueMode.None) && (optValue != null))
            {
                ExceptionUtility.Throw(
                    string.Format(Resources.Errors.ArgumentParserNoneOptionValue, arg));
            }

            ArgumentOption argOpt = new ArgumentOption(arg, optName);
            if (optValue != null)
            {
                argOpt.Value = optValue;
            }

            return argOpt;
        }
    }
}
