﻿namespace Opal.Threading
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using Opal.Utilities;

    /// <summary>
    /// ジョブ管理クラスです。
    /// </summary>
    public sealed class JobManager : IJobManager
    {
        private static readonly string ManagerKeyString;

        private readonly object syncRoot = new object();
        private readonly List<Job> jobs = new List<Job>();

        static JobManager()
        {
            ManagerKeyString = Enum.GetName(typeof(ManagerKey), ManagerKey.Job);
        }

        /// <summary>
        /// マネージャーのキーを取得します。
        /// </summary>
        public string Key
        {
            get
            {
                return ManagerKeyString;
            }
        }

        /// <summary>
        /// ジョブを追加します。
        /// </summary>
        /// <param name="job">追加するジョブです。</param>
        public void AddJob(Job job)
        {
            lock (this.syncRoot)
            {
                Debug.Assert(this.jobs.FirstOrDefault(x => x.Name == job.Name) == null);

                this.jobs.Add(job);
            }
        }

        /// <summary>
        /// ジョブを削除します。
        /// </summary>
        /// <param name="job">削除するジョブです。</param>
        public void RemoveJob(Job job)
        {
            lock (this.syncRoot)
            {
                this.jobs.Remove(job);
            }
        }

        /// <summary>
        /// ジョブを取得します。
        /// </summary>
        /// <typeparam name="TJob">取得するジョブの型です。</typeparam>
        /// <param name="name">取得するジョブの名前です。</param>
        /// <returns>取得できたジョブを返します。</returns>
        public TJob GetJob<TJob>(string name) where TJob : Job
        {
            lock (this.syncRoot)
            {
                return this.jobs.FirstOrDefault(x => x is TJob && x.Name == name) as TJob;
            }
        }

        /// <summary>
        /// 全てのジョブを停止します。
        /// </summary>
        public void StopAllJobs()
        {
            lock (this.syncRoot)
            {
                foreach (var job in this.jobs)
                {
                    job.Stop();
                }
            }
        }
    }
}
