﻿namespace Opal.Operations
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.Diagnostics;
    using System.Linq.Expressions;
    using System.Reflection;

    /// <summary>
    /// キーが値に埋め込まれているコレクションアイテムのプロパティ編集オペレーションです。
    /// </summary>
    /// <typeparam name="TKey">コレクションアイテムキーのテンプレート型です。</typeparam>
    /// <typeparam name="TItem">コレクションアイテムのテンプレート型です。</typeparam>
    /// <typeparam name="TEditData">編集対象データのテンプレート型です。</typeparam>
    public sealed class KeyedCollectionItemPropertyEditOperation<TKey, TItem, TEditData>
        : KeyedCollectionItemOperation<TKey, TItem> where TEditData : class
    {
        private readonly PropertyInfo propertyInfo;
        private Tuple<TItem, PropertyInfo> targetAndProperty = null;
        private TItem targetItem;
        private TEditData data;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="targetCollection">編集対象を含むコレクションです。</param>
        /// <param name="key">編集対象のキーです。</param>
        /// <param name="propertyName">編集対象名です。</param>
        /// <param name="data">設定するデータです。</param>
        public KeyedCollectionItemPropertyEditOperation(
            KeyedCollection<TKey, TItem> targetCollection,
            TKey key,
            string propertyName,
            TEditData data)
            : this(targetCollection, key, typeof(TItem).GetProperty(propertyName), data)
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="targetCollection">編集対象を含むコレクションです。</param>
        /// <param name="key">編集対象のキーです。</param>
        /// <param name="propertyInfo">編集対象の PropertyInfo です。</param>
        /// <param name="data">設定するデータです。</param>
        public KeyedCollectionItemPropertyEditOperation(
            KeyedCollection<TKey, TItem> targetCollection,
            TKey key,
            PropertyInfo propertyInfo,
            TEditData data)
            : base(targetCollection, key)
        {
            Debug.Assert(propertyInfo != null);
            Debug.Assert(data != null);

            this.propertyInfo = propertyInfo;
            this.data = data;
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="targetCollection">編集対象を含むコレクションです。</param>
        /// <param name="key">編集対象のキーです。</param>
        /// <param name="propertyExpression">対象プロパティを示す式です。</param>
        /// <param name="data">設定するデータです。</param>
        public KeyedCollectionItemPropertyEditOperation(
            KeyedCollection<TKey, TItem> targetCollection,
            TKey key,
            Expression<Func<TItem, TEditData>> propertyExpression,
            TEditData data)
            : base(targetCollection, key)
        {
            Debug.Assert(propertyExpression != null);
            Debug.Assert(data != null);

            var memberExpression = propertyExpression.Body as MemberExpression;
            Debug.Assert(memberExpression != null);

            this.propertyInfo = typeof(TItem).GetProperty(memberExpression.Member.Name);
            this.data = data;
        }

        /// <summary>
        /// Undo を実行します。
        /// </summary>
        /// <param name="target">編集対象です。</param>
        /// <param name="key">対象のアイテムのキーです。</param>
        protected override void Undo(KeyedCollection<TKey, TItem> target, TKey key)
        {
            if (this.targetAndProperty == null)
            {
                return;
            }

            TEditData temp = (TEditData)this.targetAndProperty.Item2.GetValue(this.targetItem);
            this.targetAndProperty.Item2.SetValue(this.targetItem, this.data, null);
            this.data = temp;
        }

        /// <summary>
        /// Redo を実行します。
        /// </summary>
        /// <param name="target">編集対象です。</param>
        /// <param name="key">対象のアイテムのキーです。</param>
        protected override void Redo(KeyedCollection<TKey, TItem> targetCollection, TKey key)
        {
            try
            {
                this.targetItem = targetCollection[key];
            }
            catch (KeyNotFoundException)
            {
                // 対象が見つからない場合は何もしないこととする。
                return;
            }

            this.targetAndProperty = Tuple.Create(this.targetItem, this.propertyInfo);

            TEditData temp = (TEditData)this.targetAndProperty.Item2.GetValue(this.targetItem);
            this.targetAndProperty.Item2.SetValue(this.targetItem, this.data, null);
            this.data = temp;
        }
    }
}
