﻿namespace Opal.Logs
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.Diagnostics;
    using System.Linq;
    using System.Runtime.CompilerServices;
    using System.Text;
    using System.Threading.Tasks;
    using Opal.Utilities;

    /// <summary>
    /// ロガークラスです。
    /// </summary>
    public sealed class Logger
    {
        private readonly object syncRoot = new object();
        private readonly List<Log> logs = new List<Log>();

        /// <summary>
        /// デバッグ用のログを登録します。
        /// </summary>
        /// <param name="message">登録するメッセージです。</param>
        /// <param name="fileName">ファイル名です。記述した場所を自動的に入れるので設定不要です。</param>
        /// <param name="fileLine">ファイル行です。記述した場所を自動的に入れるので設定不要です。</param>
        /// <param name="memberName">メンバー名です。記述した場所を自動的に入れるので設定不要です。</param>
        [Conditional("DEBUG")]
        public void Debug(string message, [CallerFilePath] string fileName = null, [CallerLineNumber] int fileLine = 0, [CallerMemberName] string memberName = null)
        {
            System.Diagnostics.Debug.Assert(message != null);
            lock (this.syncRoot)
            {
                Log log = new Log(LogLevel.Debug, message, fileName, fileLine, memberName);
                this.logs.Add(log);
            }
        }

        /// <summary>
        /// デバッグ用のログを登録します。
        /// </summary>
        /// <param name="message">登録するメッセージです。</param>
        /// <param name="exception">登録する例外です。</param>
        /// <param name="fileName">ファイル名です。記述した場所を自動的に入れるので設定不要です。</param>
        /// <param name="fileLine">ファイル行です。記述した場所を自動的に入れるので設定不要です。</param>
        /// <param name="memberName">メンバー名です。記述した場所を自動的に入れるので設定不要です。</param>
        [Conditional("DEBUG")]
        public void Debug(string message, Exception exception, [CallerFilePath] string fileName = null, [CallerLineNumber] int fileLine = 0, [CallerMemberName] string memberName = null)
        {
            System.Diagnostics.Debug.Assert(message != null);
            System.Diagnostics.Debug.Assert(exception != null);
            lock (this.syncRoot)
            {
                Log log = new Log(LogLevel.Debug, message, exception, null, fileName, fileLine, memberName);
                this.logs.Add(log);
            }
        }

        /// <summary>
        /// 警告用のログを登録します。
        /// </summary>
        /// <param name="message">登録するメッセージです。</param>
        /// <param name="fileName">ファイル名です。記述した場所を自動的に入れるので設定不要です。</param>
        /// <param name="fileLine">ファイル行です。記述した場所を自動的に入れるので設定不要です。</param>
        /// <param name="memberName">メンバー名です。記述した場所を自動的に入れるので設定不要です。</param>
        public void Warning(string message, [CallerFilePath] string fileName = null, [CallerLineNumber] int fileLine = 0, [CallerMemberName] string memberName = null)
        {
            System.Diagnostics.Debug.Assert(message != null);
            lock (this.syncRoot)
            {
                Log log = new Log(LogLevel.Warning, message, fileName, fileLine, memberName);
                this.logs.Add(log);
            }
        }

        /// <summary>
        /// 警告用のログを登録します。
        /// </summary>
        /// <param name="message">登録するメッセージです。</param>
        /// <param name="exception">登録する例外です。</param>
        /// <param name="associatedAction">ログに関連付ける処理です。</param>
        /// <param name="fileName">ファイル名です。記述した場所を自動的に入れるので設定不要です。</param>
        /// <param name="fileLine">ファイル行です。記述した場所を自動的に入れるので設定不要です。</param>
        /// <param name="memberName">メンバー名です。記述した場所を自動的に入れるので設定不要です。</param>
        public void Warning(string message, Exception exception, Action associatedAction, [CallerFilePath] string fileName = null, [CallerLineNumber] int fileLine = 0, [CallerMemberName] string memberName = null)
        {
            System.Diagnostics.Debug.Assert(message != null);
            lock (this.syncRoot)
            {
                Log log = new Log(LogLevel.Warning, message, exception, associatedAction, fileName, fileLine, memberName);
                this.logs.Add(log);
            }
        }

        /// <summary>
        /// エラー用のログを登録します。
        /// </summary>
        /// <param name="message">登録するメッセージです。</param>
        /// <param name="fileName">ファイル名です。記述した場所を自動的に入れるので設定不要です。</param>
        /// <param name="fileLine">ファイル行です。記述した場所を自動的に入れるので設定不要です。</param>
        /// <param name="memberName">メンバー名です。記述した場所を自動的に入れるので設定不要です。</param>
        public void Error(string message, [CallerFilePath] string fileName = null, [CallerLineNumber] int fileLine = 0, [CallerMemberName] string memberName = null)
        {
            System.Diagnostics.Debug.Assert(message != null);
            lock (this.syncRoot)
            {
                Log log = new Log(LogLevel.Error, message, fileName, fileLine, memberName);
                this.logs.Add(log);
            }
        }

        /// <summary>
        /// エラー用のログを登録します。
        /// </summary>
        /// <param name="message">登録するメッセージです。</param>
        /// <param name="exception">登録する例外です。</param>
        /// <param name="associatedAction">ログに関連付ける処理です。</param>
        /// <param name="fileName">ファイル名です。記述した場所を自動的に入れるので設定不要です。</param>
        /// <param name="fileLine">ファイル行です。記述した場所を自動的に入れるので設定不要です。</param>
        /// <param name="memberName">メンバー名です。記述した場所を自動的に入れるので設定不要です。</param>
        public void Error(string message, Exception exception, Action associatedAction, [CallerFilePath] string fileName = null, [CallerLineNumber] int fileLine = 0, [CallerMemberName] string memberName = null)
        {
            System.Diagnostics.Debug.Assert(message != null);
            lock (this.syncRoot)
            {
                Log log = new Log(LogLevel.Error, message, exception, associatedAction, fileName, fileLine, memberName);
                this.logs.Add(log);
            }
        }

        /// <summary>
        /// ユーザ通知用メッセージログを登録します。
        /// </summary>
        /// <param name="message">登録するメッセージです。</param>
        /// <param name="fileName">ファイル名です。記述した場所を自動的に入れるので設定不要です。</param>
        /// <param name="fileLine">ファイル行です。記述した場所を自動的に入れるので設定不要です。</param>
        /// <param name="memberName">メンバー名です。記述した場所を自動的に入れるので設定不要です。</param>
        public void UserMessage(string message, [CallerFilePath] string fileName = null, [CallerLineNumber] int fileLine = 0, [CallerMemberName] string memberName = null)
        {
            System.Diagnostics.Debug.Assert(message != null);
            lock (this.syncRoot)
            {
                Log log = new Log(LogLevel.UserMessage, message, fileName, fileLine, memberName);
                this.logs.Add(log);
            }
        }

        /// <summary>
        /// 情報用のログを登録します。
        /// </summary>
        /// <param name="message">登録するメッセージです。</param>
        /// <param name="fileName">ファイル名です。記述した場所を自動的に入れるので設定不要です。</param>
        /// <param name="fileLine">ファイル行です。記述した場所を自動的に入れるので設定不要です。</param>
        /// <param name="memberName">メンバー名です。記述した場所を自動的に入れるので設定不要です。</param>
        public void Info(string message, [CallerFilePath] string fileName = null, [CallerLineNumber] int fileLine = 0, [CallerMemberName] string memberName = null)
        {
            System.Diagnostics.Debug.Assert(message != null);
            lock (this.syncRoot)
            {
                Log log = new Log(LogLevel.Application, message, fileName, fileLine, memberName);
                this.logs.Add(log);
            }
        }

        /// <summary>
        /// 情報用のログを登録します。
        /// </summary>
        /// <param name="message">登録するメッセージです。</param>
        /// <param name="exception">登録する例外です。</param>
        /// <param name="fileName">ファイル名です。記述した場所を自動的に入れるので設定不要です。</param>
        /// <param name="fileLine">ファイル行です。記述した場所を自動的に入れるので設定不要です。</param>
        /// <param name="memberName">メンバー名です。記述した場所を自動的に入れるので設定不要です。</param>
        public void Info(string message, Exception exception, [CallerFilePath] string fileName = null, [CallerLineNumber] int fileLine = 0, [CallerMemberName] string memberName = null)
        {
            System.Diagnostics.Debug.Assert(message != null);
            System.Diagnostics.Debug.Assert(exception != null);
            lock (this.syncRoot)
            {
                Log log = new Log(LogLevel.Application, message, exception, null, fileName, fileLine, memberName);
                this.logs.Add(log);
            }
        }

        /// <summary>
        /// 状態をクリアします。
        /// </summary>
        public void Clear()
        {
            lock (this.syncRoot)
            {
                this.logs.Clear();
            }
        }

        /// <summary>
        /// ログを取得します。
        /// </summary>
        /// <returns>ログのインスタンスを返します。</returns>
        internal ReadOnlyCollection<Log> GetLogs()
        {
            lock (this.syncRoot)
            {
                return new ReadOnlyCollection<Log>(this.logs);
            }
        }
    }
}
