﻿namespace Opal.Commands
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using System.Windows.Input;

    /// <summary>
    /// ViewModelとViewを中継するコマンドです。
    /// </summary>
    public class RelayCommand : ICommand
    {
        private readonly Action<object> execute;
        private readonly Func<object, bool> canExecute;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="execute">コマンドが実行される時に呼ばれる処理です。</param>
        public RelayCommand(Action<object> execute)
            : this(execute, null)
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="execute">コマンドが実行される時に呼ばれる処理です。</param>
        /// <param name="canExecute">コマンドが実行できるか判定する時に呼ばれる処理です。</param>
        public RelayCommand(Action<object> execute, Func<object, bool> canExecute)
        {
            Debug.Assert(execute != null);

            this.execute = execute;
            this.canExecute = canExecute;
        }

        /// <summary>
        /// 実行可能か判定時に呼ばれるイベントです。
        /// </summary>
        public event EventHandler CanExecuteChanged
        {
            add
            {
                CommandManager.RequerySuggested += value;
            }

            remove
            {
                CommandManager.RequerySuggested -= value;
            }
        }

        /// <summary>
        /// 実行可能か判定します。
        /// </summary>
        /// <param name="parameter">判定時のパラメータです。</param>
        /// <returns>実行可能なら true を返します。</returns>
        public bool CanExecute(object parameter)
        {
            return this.canExecute == null ? true : this.canExecute(parameter);
        }

        /// <summary>
        /// コマンドを実行します。
        /// </summary>
        /// <param name="parameter">実行時のパラメータです。</param>
        public void Execute(object parameter)
        {
            this.execute(parameter);
        }
    }
}
