﻿// ========================================================================
// <copyright file="CUIToolUtility.cs" company="Nintendo">
//      Copyright 2013 Nintendo.  All rights reserved.
// </copyright>
//
// These coded instructions, statements, and computer programs contain
// proprietary information of Nintendo of America Inc. and/or Nintendo
// Company Ltd., and are protected by Federal copyright law.  They may
// not be disclosed to third parties or copied or duplicated in any form,
// in whole or in part, without the prior written consent of Nintendo.
// ========================================================================

namespace Opal.Tentatives.Utilities
{
    using System;
    using System.Collections;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Threading;
    using Opal.Logs;

    /// <summary>
    /// コマンドラインツール用のユーティリティです。
    /// </summary>
    public static class CUIToolUtility
    {
        private const string ExtraInfoKey = "ExtraInfo";

        /// <summary>
        /// エラー発生時にポーズするかを設定します。
        /// </summary>
        public static bool EnableErrorPause { get; set; }

        /// <summary>
        /// ログマネージャです。
        /// </summary>
        public static ILogManager LogManager { get; set; }

        ////---------------------------------------------------------------------

        /// <summary>
        /// 処理を実行します。
        /// </summary>
        /// <param name="runMethod">実行するメソッドです。</param>
        public static void Run(Action runMethod)
        {
            Thread.GetDomain().UnhandledException += CUIToolUtility.CatchUnhandledException;

            try
            {
                runMethod();
            }
            catch (Exception exception)
            {
                // 例外の表示
                CUIToolUtility.PrintException(exception);
            }

            if ((Environment.ExitCode == 1) && CUIToolUtility.EnableErrorPause)
            {
                Console.ReadKey();
            }
        }

        /// <summary>
        /// ログを出力します。
        /// </summary>
        /// <param name="logger">ロガーです。</param>
        public static void WriteLog(Logger logger)
        {
            Debug.Assert(logger != null);

            if (LogManager != null)
            {
                LogManager.PushLogger(logger);
                LogManager.FlushAll();
            }
        }

        /// <summary>
        /// 例外の表示時に併せて表示する付加情報を追加します。
        /// </summary>
        /// <param name="exp">付加情報を追加する例外です。</param>
        /// <param name="message">付加情報です。</param>
        public static void AddExtraInfoToException(Exception exp, string message)
        {
            exp.Data.Add(ExtraInfoKey, message);
        }

        ////---------------------------------------------------------------------

        // 未処理例外のキャッチ
        private static void CatchUnhandledException(object sender, UnhandledExceptionEventArgs args)
        {
            CUIToolUtility.PrintException(args.ExceptionObject as Exception);
            if (CUIToolUtility.EnableErrorPause)
            {
                Console.ReadKey();
            }

            Environment.Exit(1);
        }

        // 例外の表示
        private static void PrintException(Exception exp)
        {
            ExceptionListException exceptionList = exp as ExceptionListException;
            if (exceptionList != null)
            {
                foreach (Exception exception in exceptionList.Exceptions)
                {
                    PrintException(exception);
                }
            }
            else
            {
                while (exp != null)
                {
                    Logger logger = new Logger();
                    if (exp.Message != null)
                    {
                        logger.Error(exp.Message);
                    }

                    foreach (DictionaryEntry extraInfo in exp.Data)
                    {
                        if (extraInfo.Key.ToString() == ExtraInfoKey)
                        {
                            string extraInfoString = extraInfo.Value.ToString();
                            if (extraInfoString != null)
                            {
                                logger.Error(extraInfoString);
                            }
                        }
                    }

#if DEBUG
                    if (exp.StackTrace != null)
                    {
                        logger.Error(exp.StackTrace);
                    }
#endif
                    CUIToolUtility.WriteLog(logger);

                    exp = exp.InnerException;
                }
            }
        }

        ////---------------------------------------------------------------------

        /// <summary>
        /// 例外リストを持つ例外です。
        /// </summary>
        public class ExceptionListException : Exception
        {
            /// <summary>
            /// 例外リストです。
            /// </summary>
            public readonly List<Exception> Exceptions = new List<Exception>();
        }
    }
}
