﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public abstract class Stream : ObservableEntity<streamType>, IDeepCopyable<Stream>, IDeepCopyFrom<Stream>
    {
        private int @column = 1;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public Stream()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Stream(Stream source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Stream(streamType source)
        {
            this.@column = source.@column;
        }

        public int Column
        {
            get
            {
                return this.@column;
            }

            set
            {
                if (this.@column == value)
                {
                    return;
                }

                this.SetProperty(ref this.@column, value, () => this.CalcCRC());
            }
        }

        public abstract int Count{ get; }

        public abstract stream_typeType Type{ get; }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override streamType CreateSerializableData()
        {
            var writeData = new streamType();
            writeData.@column = this.Column;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        Stream IDeepCopyable<Stream>.DeepCopy()
        {
            return this.DeepCopy();
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        private protected abstract Stream DeepCopy();

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(Stream source)
        {
            this.@column = source.@column;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Column)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }
    }
}
