﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class Srt2d : ObservableEntity<String>, IDeepCopyable<Srt2d>, IDeepCopyFrom<Srt2d>
    {
        private float @scaleX = 1;
        private float @scaleY = 1;
        private float @rotate = 0;
        private float @translateX = 0;
        private float @translateY = 0;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public Srt2d()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Srt2d(Srt2d source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Srt2d(String source)
        {
            string[] valueTexts = StringUtility.SplitValueListText(source);
            Ensure.Argument.AreEqual(5, valueTexts.Length, "Invalid input text format");
            this.ScaleX = float.Parse(valueTexts[0]);
            this.ScaleY = float.Parse(valueTexts[1]);
            this.Rotate = float.Parse(valueTexts[2]);
            this.TranslateX = float.Parse(valueTexts[3]);
            this.TranslateY = float.Parse(valueTexts[4]);
        }

        public float ScaleX
        {
            get
            {
                return this.@scaleX;
            }

            set
            {
                if (this.@scaleX == value)
                {
                    return;
                }

                this.SetProperty(ref this.@scaleX, value, () => this.CalcCRC());
            }
        }

        public float ScaleY
        {
            get
            {
                return this.@scaleY;
            }

            set
            {
                if (this.@scaleY == value)
                {
                    return;
                }

                this.SetProperty(ref this.@scaleY, value, () => this.CalcCRC());
            }
        }

        public float Rotate
        {
            get
            {
                return this.@rotate;
            }

            set
            {
                if (this.@rotate == value)
                {
                    return;
                }

                this.SetProperty(ref this.@rotate, value, () => this.CalcCRC());
            }
        }

        public float TranslateX
        {
            get
            {
                return this.@translateX;
            }

            set
            {
                if (this.@translateX == value)
                {
                    return;
                }

                this.SetProperty(ref this.@translateX, value, () => this.CalcCRC());
            }
        }

        public float TranslateY
        {
            get
            {
                return this.@translateY;
            }

            set
            {
                if (this.@translateY == value)
                {
                    return;
                }

                this.SetProperty(ref this.@translateY, value, () => this.CalcCRC());
            }
        }

        public override string ToString()
        {
            return this.CreateSerializableData();
        }

        public static Srt2d Parse(string source)
        {
            return new Srt2d(source);
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override String CreateSerializableData()
        {
            var writeData = string.Empty;
            writeData += this.ScaleX.ToString() + " ";
            writeData += this.ScaleY.ToString() + " ";
            writeData += this.Rotate.ToString() + " ";
            writeData += this.TranslateX.ToString() + " ";
            writeData += this.TranslateY.ToString() + " ";
            writeData = writeData.TrimEnd();
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        Srt2d IDeepCopyable<Srt2d>.DeepCopy()
        {
            return new Srt2d(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(Srt2d source)
        {
            this.@scaleX = source.@scaleX;
            this.@scaleY = source.@scaleY;
            this.@rotate = source.@rotate;
            this.@translateX = source.@translateX;
            this.@translateY = source.@translateY;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.ScaleX)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.ScaleY)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Rotate)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.TranslateX)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.TranslateY)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }
    }
}
