﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class Source : ObservableEntity<sourceType>, IDeepCopyable<Source>, IDeepCopyFrom<Source>
    {
        private int @plugIndex;
        private string @block = string.Empty;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public Source()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Source(Source source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Source(sourceType source)
        {
            this.@plugIndex = source.@plug_index;
            this.@block = source.@block;
        }

        public int PlugIndex
        {
            get
            {
                return this.@plugIndex;
            }

            set
            {
                if (this.@plugIndex == value)
                {
                    return;
                }

                this.SetProperty(ref this.@plugIndex, value, () => this.CalcCRC());
            }
        }

        public string Block
        {
            get
            {
                return this.@block;
            }

            set
            {
                if (this.@block == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@block, value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override sourceType CreateSerializableData()
        {
            var writeData = new sourceType();
            writeData.@plug_index = this.PlugIndex;
            writeData.@block = this.Block;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        Source IDeepCopyable<Source>.DeepCopy()
        {
            return new Source(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(Source source)
        {
            this.@plugIndex = source.@plugIndex;
            this.@block = source.@block;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.PlugIndex)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Block)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }
    }
}
