﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class SkeletalAnim : RootEntity<skeletal_animType>, IDeepCopyable<SkeletalAnim>, IDeepCopyFrom<SkeletalAnim>, IToolData
    {
        private readonly ObservableList<ProcessLog> @processLogs = new ObservableList<ProcessLog>();
        private readonly SkeletalAnimInfo @skeletalAnimInfo = new SkeletalAnimInfo();
        private readonly ObservableList<BoneAnim> @boneAnims = new ObservableList<BoneAnim>();
        private readonly ObservableList<UserData> @userDatas = new ObservableList<UserData>();
        private readonly ObservableList<Stream> @streams = new ObservableList<Stream>();
        private readonly Comment @comment = new Comment();
        private readonly ToolData @toolData = new ToolData();
        private readonly UserToolData @userToolData = new UserToolData();
        private skeletal_anim_versionType @version;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        internal SkeletalAnim()
        {
            this.@processLogs.CollectionChanged += OnCollectionChanged<ProcessLog>;
            this.@skeletalAnimInfo.PropertyChanged += this.OnPropertyChanged;
            this.@boneAnims.CollectionChanged += OnCollectionChanged<BoneAnim>;
            this.@userDatas.CollectionChanged += OnCollectionChanged<UserData>;
            this.@streams.CollectionChanged += OnCollectionChanged<Stream>;
            this.@comment.PropertyChanged += this.OnPropertyChanged;
            this.@toolData.PropertyChanged += this.OnPropertyChanged;
            this.@userToolData.PropertyChanged += this.OnPropertyChanged;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public SkeletalAnim(SkeletalAnim source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public SkeletalAnim(skeletal_animType source)
        {
            this.@processLogs.CollectionChanged += OnCollectionChanged<ProcessLog>;
            this.@boneAnims.CollectionChanged += OnCollectionChanged<BoneAnim>;
            this.@userDatas.CollectionChanged += OnCollectionChanged<UserData>;
            this.@streams.CollectionChanged += OnCollectionChanged<Stream>;
            if (source.@process_log_array != null)
            {
                this.@processLogs.Clear();
                foreach (var elem in source.@process_log_array.Items)
                {
                    this.@processLogs.Add(new ProcessLog(elem));
                }
            }
            if (source.@skeletal_anim_info != null)
            {
                this.@skeletalAnimInfo = new SkeletalAnimInfo(source.@skeletal_anim_info);
            }
            else
            {
                this.@skeletalAnimInfo = new SkeletalAnimInfo();
            }
            if (source.@bone_anim_array != null)
            {
                this.@boneAnims.Clear();
                foreach (var elem in source.@bone_anim_array.Items)
                {
                    this.@boneAnims.Add(new BoneAnim(elem));
                }
            }
            if (source.@user_data_array != null)
            {
                this.@userDatas.Clear();
                foreach (var elem in source.@user_data_array.Items)
                {
                    this.@userDatas.Add(DataModelConverter.Convert(elem));
                }
            }
            if (source.@stream_array != null)
            {
                this.@streams.Clear();
                foreach (var elem in source.@stream_array.Items)
                {
                    this.@streams.Add(DataModelConverter.Convert(elem));
                }
            }
            if (source.@comment != null)
            {
                this.@comment = new Comment(source.@comment);
            }
            else
            {
                this.@comment = new Comment();
            }
            if (source.@tool_data != null)
            {
                this.@toolData = new ToolData(source.@tool_data);
            }
            else
            {
                this.@toolData = new ToolData();
            }
            if (source.@user_tool_data != null)
            {
                this.@userToolData = new UserToolData(source.@user_tool_data);
            }
            else
            {
                this.@userToolData = new UserToolData();
            }
            this.@version = source.@version;
            this.@skeletalAnimInfo.PropertyChanged += this.OnPropertyChanged;
            this.@comment.PropertyChanged += this.OnPropertyChanged;
            this.@toolData.PropertyChanged += this.OnPropertyChanged;
            this.@userToolData.PropertyChanged += this.OnPropertyChanged;
            this.ResolveInternalReferences();
        }

        public ObservableList<ProcessLog> ProcessLogs
        {
            get
            {
                return this.@processLogs;
            }
        }

        public SkeletalAnimInfo SkeletalAnimInfo
        {
            get
            {
                return this.@skeletalAnimInfo;
            }
        }

        public ObservableList<BoneAnim> BoneAnims
        {
            get
            {
                return this.@boneAnims;
            }
        }

        public ObservableList<UserData> UserDatas
        {
            get
            {
                return this.@userDatas;
            }
        }

        public override ObservableList<Stream> Streams
        {
            get
            {
                return this.@streams;
            }
        }

        public Comment Comment
        {
            get
            {
                return this.@comment;
            }
        }

        public ToolData ToolData
        {
            get
            {
                return this.@toolData;
            }
        }

        public UserToolData UserToolData
        {
            get
            {
                return this.@userToolData;
            }
        }

        public skeletal_anim_versionType Version
        {
            get
            {
                return this.@version;
            }

            set
            {
                if (this.@version == value)
                {
                    return;
                }

                this.SetProperty(ref this.@version, value, () => this.CalcCRC());
            }
        }

        public override void ResolveInternalReferences()
        {
            // インデックスで紐づけをしているデータに実データの割り当て
            if (this.Streams.Count > 0)
            {
                foreach (var anim in this.BoneAnims)
                {
                    foreach (var target in anim.BoneAnimTargets.Values)
                    {
                        if (target.Curve != null)
                        {
                            target.Curve.Stream = this.Streams[(target.Curve as IStreamIndex).StreamIndex];
                        }
                    }
                }
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override skeletal_animType CreateSerializableData()
        {
            Ensure.Operation.Range(this.BoneAnims.Count, 1, int.MaxValue);
            var writeData = new skeletal_animType();
            if (this.@ProcessLogs.Count > 0)
            {
                writeData.@process_log_array = new process_log_arrayType();
                writeData.@process_log_array.Items = this.@processLogs.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@process_log_array.Items.Length == 0)
                {
                    writeData.@process_log_array = null;
                }
            }
            writeData.@skeletal_anim_info = this.@skeletalAnimInfo.CreateSerializableData() as skeletal_anim_infoType;
            if (this.@BoneAnims.Count > 0)
            {
                writeData.@bone_anim_array = new bone_anim_arrayType();
                writeData.@bone_anim_array.Items = this.@boneAnims.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@bone_anim_array.Items.Length == 0)
                {
                    writeData.@bone_anim_array = null;
                }
            }
            if (this.@UserDatas.Count > 0)
            {
                writeData.@user_data_array = new user_data_arrayType();
                writeData.@user_data_array.Items = this.@userDatas.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@user_data_array.Items.Length == 0)
                {
                    writeData.@user_data_array = null;
                }
            }
            if (this.IsStreamSerializationEnabled)
            {
                if (this.@Streams.Count > 0)
                {
                    writeData.@stream_array = new stream_arrayType();
                    writeData.@stream_array.Items = this.@streams.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                    if (writeData.@stream_array.Items.Length == 0)
                    {
                        writeData.@stream_array = null;
                    }
                }
            }
            writeData.@comment = this.@comment.CreateSerializableData() as commentType;
            writeData.@tool_data = this.@toolData.CreateSerializableData() as tool_dataType;
            writeData.@user_tool_data = this.@userToolData.CreateSerializableData() as user_tool_dataType;
            writeData.@version = this.Version;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        SkeletalAnim IDeepCopyable<SkeletalAnim>.DeepCopy()
        {
            return new SkeletalAnim(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(SkeletalAnim source)
        {
            CopyElements(source.@processLogs, this.@processLogs);
            this.@skeletalAnimInfo.DeepCopyFrom(source.@skeletalAnimInfo);
            CopyElements(source.@boneAnims, this.@boneAnims);
            CopyElements(source.@userDatas, this.@userDatas);
            CopyElements(source.@streams, this.@streams);
            this.@comment.DeepCopyFrom(source.@comment);
            this.@toolData.DeepCopyFrom(source.@toolData);
            this.@userToolData.DeepCopyFrom(source.@userToolData);
            this.@version = source.@version;
            this.ResolveInternalReferences();
        }

        private protected override IRootEntity DeepCopy()
        {
            return new SkeletalAnim(this);
        }

        private protected override void DeepCopyFrom(IRootEntity source)
        {
            Nintendo.ToolFoundation.Contracts.Ensure.Argument.True(source is SkeletalAnim);
            this.DeepCopyFrom(source as SkeletalAnim);
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            foreach (var elem in this.ProcessLogs)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.SkeletalAnimInfo.HashValue)));
            foreach (var elem in this.BoneAnims)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.UserDatas)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.Streams)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Comment.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.ToolData.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.UserToolData.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Version)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            foreach (var elem in this.@processLogs)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            this.@skeletalAnimInfo.AutoCalc = this.AutoCalc;
            foreach (var elem in this.@boneAnims)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@userDatas)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@streams)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            this.@comment.AutoCalc = this.AutoCalc;
            this.@toolData.AutoCalc = this.AutoCalc;
            this.@userToolData.AutoCalc = this.AutoCalc;
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            foreach (var elem in this.@processLogs)
            {
                elem.Reset();
            }
            this.@skeletalAnimInfo.Reset();
            foreach (var elem in this.@boneAnims)
            {
                elem.Reset();
            }
            foreach (var elem in this.@userDatas)
            {
                elem.Reset();
            }
            foreach (var elem in this.@streams)
            {
                elem.Reset();
            }
            this.@comment.Reset();
            this.@toolData.Reset();
            this.@userToolData.Reset();
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            foreach (var elem in this.@processLogs)
            {
                elem.Refresh();
            }
            this.@skeletalAnimInfo.Refresh();
            foreach (var elem in this.@boneAnims)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@userDatas)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@streams)
            {
                elem.Refresh();
            }
            this.@comment.Refresh();
            this.@toolData.Refresh();
            this.@userToolData.Refresh();
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
