﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class ShaderSrc : ObservableEntity<shader_srcType>, IChildEntity, IStreamIndex, IDeepCopyable<ShaderSrc>, IDeepCopyFrom<ShaderSrc>
    {
        private string @includePath = string.Empty;
        private string @path = string.Empty;
        private int @streamIndex;
        private ShaderDefinition parent = null;
        private StreamWstring @stream = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ShaderSrc()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public ShaderSrc(ShaderSrc source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public ShaderSrc(shader_srcType source)
        {
            this.@includePath = source.@include_path;
            this.@path = source.@path;
            this.@streamIndex = source.@stream_index;
        }

        public string IncludePath
        {
            get
            {
                return this.@includePath;
            }

            set
            {
                if (this.@includePath == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@includePath, value, () => this.CalcCRC());
            }
        }

        public string Path
        {
            get
            {
                return this.@path;
            }

            set
            {
                if (this.@path == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@path, value, () => this.CalcCRC());
            }
        }

        int IStreamIndex.StreamIndex
        {
            get
            {
                if (this.Stream != null)
                {
                    // 親が見つからない場合はインデックスを求められないので -1 を返す
                    IRootEntity rootEntity = Utility.FindRootEntity((this as IChildEntity).Parent);
                    if (rootEntity == null)
                    {
                        return -1;
                    }

                    var sourceProp = rootEntity.GetType().GetProperty("Streams").GetValue(rootEntity) as ObservableList<Stream>;
                    return sourceProp.IndexOf(this.Stream);
                }
                return this.@streamIndex;
            }
        }

        IEntity IChildEntity.Parent
        {
            get
            {
                return this.parent as IEntity;
            }

            set
            {
                if (this.parent == value)
                {
                    return;
                }

                Ensure.Argument.True(value != null ? value is ShaderDefinition : true);
                this.SetProperty(ref this.parent, value as ShaderDefinition, () => this.CalcCRC());
            }
        }

        public ShaderDefinition Parent
        {
            get
            {
                return this.parent;
            }
        }

        Type IChildEntity.ParentType
        {
            get
            {
                return typeof(ShaderDefinition);
            }
        }

        public StreamWstring Stream
        {
            get
            {
                return this.@stream;
            }

            set
            {
                if (this.@stream == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@stream, value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override shader_srcType CreateSerializableData()
        {
            var writeData = new shader_srcType();
            writeData.@include_path = this.IncludePath;
            writeData.@path = this.Path;
            {
                IRootEntity rootEntity = Utility.FindRootEntity((this as IChildEntity).Parent);
                Ensure.Operation.NotNull(rootEntity);
                var sourceProp = rootEntity.GetType().GetProperty("Streams").GetValue(rootEntity) as ObservableList<Stream>;
                // 読み取り高速化のために不要なデータを読み飛ばす場合があるので、参照先が存在しない場合を許容します
                Ensure.Operation.True((sourceProp.Count == 0) || (Stream != null));
                writeData.stream_index = (this as IStreamIndex).StreamIndex;
            }
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        ShaderSrc IDeepCopyable<ShaderSrc>.DeepCopy()
        {
            return new ShaderSrc(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(ShaderSrc source)
        {
            this.@includePath = source.@includePath;
            this.@path = source.@path;
            this.@streamIndex = source.@streamIndex;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.IncludePath)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Path)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32((this as IStreamIndex).StreamIndex)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }
    }
}
