﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public abstract class ShaderParam : ObservableEntity<shader_paramType>, IDeepCopyable<ShaderParam>, IDeepCopyFrom<ShaderParam>
    {
        private string @id = string.Empty;
        private string @originalHint = string.Empty;
        private string @depend = string.Empty;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ShaderParam()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public ShaderParam(ShaderParam source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public ShaderParam(shader_paramType source)
        {
            this.@id = source.@id;
            this.@originalHint = source.@original_hint;
            this.@depend = source.@depend;
        }

        public string Id
        {
            get
            {
                return this.@id;
            }

            set
            {
                if (this.@id == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@id, value, () => this.CalcCRC());
            }
        }

        public string OriginalHint
        {
            get
            {
                return this.@originalHint;
            }

            set
            {
                if (this.@originalHint == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@originalHint, value, () => this.CalcCRC());
            }
        }

        public string Depend
        {
            get
            {
                return this.@depend;
            }

            set
            {
                if (this.@depend == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@depend, value, () => this.CalcCRC());
            }
        }

        public abstract shader_param_typeType Type { get; }

        private protected abstract void SetToDefaultInternal(UniformVar definition);

        private protected void SetToDefault(UniformVar definition)
        {
            this.Depend = definition.Depend;
            this.SetToDefaultInternal(definition);
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override shader_paramType CreateSerializableData()
        {
            var writeData = new shader_paramType();
            writeData.@id = this.Id;
            writeData.@original_hint = this.OriginalHint;
            writeData.@depend = this.Depend;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        ShaderParam IDeepCopyable<ShaderParam>.DeepCopy()
        {
            return this.DeepCopy();
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        private protected abstract ShaderParam DeepCopy();

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(ShaderParam source)
        {
            this.@id = source.@id;
            this.@originalHint = source.@originalHint;
            this.@depend = source.@depend;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Id)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.OriginalHint)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Depend)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }
    }
}
