﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class PerMaterialAnim : ObservableEntity<per_material_animType>, IChildEntity, IDeepCopyable<PerMaterialAnim>, IDeepCopyFrom<PerMaterialAnim>, IDisposable
    {
        private readonly ObservableList<ParamAnim> @paramAnims = new ObservableList<ParamAnim>();
        private readonly ObservableList<PatternAnim> @patternAnims = new ObservableList<PatternAnim>();
        private MaterialVisibilityAnim @materialVisibilityAnim = null;
        private string @matName = string.Empty;
        private MaterialAnim parent = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public PerMaterialAnim()
        {
            this.@paramAnims.CollectionChanged += OnCollectionChanged<ParamAnim>;
            this.@patternAnims.CollectionChanged += OnCollectionChanged<PatternAnim>;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public PerMaterialAnim(PerMaterialAnim source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public PerMaterialAnim(per_material_animType source)
        {
            this.@paramAnims.CollectionChanged += OnCollectionChanged<ParamAnim>;
            this.@patternAnims.CollectionChanged += OnCollectionChanged<PatternAnim>;
            if (source.@shader_param_anim_array != null)
            {
                this.@paramAnims.Clear();
                foreach (var elem in source.@shader_param_anim_array.Items)
                {
                    this.@paramAnims.Add(new ParamAnim(elem));
                }
            }
            if (source.@tex_pattern_anim_array != null)
            {
                this.@patternAnims.Clear();
                foreach (var elem in source.@tex_pattern_anim_array.Items)
                {
                    this.@patternAnims.Add(new PatternAnim(elem));
                }
            }
            if (source.@material_visibility_anim != null)
            {
                this.MaterialVisibilityAnim = new MaterialVisibilityAnim(source.@material_visibility_anim);
            }
            else
            {
                this.MaterialVisibilityAnim = null;
            }
            this.@matName = source.@mat_name;
        }

        public void Dispose()
        {
            if (this.@materialVisibilityAnim != null)
            {
                this.@materialVisibilityAnim.PropertyChanged -= this.OnPropertyChanged;
            }
        }

        public ObservableList<ParamAnim> ParamAnims
        {
            get
            {
                return this.@paramAnims;
            }
        }

        public ObservableList<PatternAnim> PatternAnims
        {
            get
            {
                return this.@patternAnims;
            }
        }

        public MaterialVisibilityAnim MaterialVisibilityAnim
        {
            get
            {
                return this.@materialVisibilityAnim;
            }

            set
            {
                if (this.@materialVisibilityAnim == value)
                {
                    return;
                }

                Ensure.Argument.True(value != null ? typeof(MaterialVisibilityAnim).IsAssignableFrom(value.GetType()) : true);
                if (this.@materialVisibilityAnim != null)
                {
                    (this.@materialVisibilityAnim as IChildEntity).Parent = null;
                }

                if (value != null)
                {
                    (value as IChildEntity).Parent = this;
                }

                if (this.@materialVisibilityAnim != null)
                {
                    this.@materialVisibilityAnim.PropertyChanged -= this.OnPropertyChanged;
                }

                if (value != null)
                {
                    value.PropertyChanged += this.OnPropertyChanged;
                }

                this.SetProperty(ref this.@materialVisibilityAnim, value, () => this.CalcCRC());
            }
        }

        public string MatName
        {
            get
            {
                return this.@matName;
            }

            set
            {
                if (this.@matName == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@matName, value, () => this.CalcCRC());
            }
        }

        IEntity IChildEntity.Parent
        {
            get
            {
                return this.parent as IEntity;
            }

            set
            {
                if (this.parent == value)
                {
                    return;
                }

                Ensure.Argument.True(value != null ? value is MaterialAnim : true);
                this.SetProperty(ref this.parent, value as MaterialAnim, () => this.CalcCRC());
            }
        }

        public MaterialAnim Parent
        {
            get
            {
                return this.parent;
            }
        }

        Type IChildEntity.ParentType
        {
            get
            {
                return typeof(MaterialAnim);
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override per_material_animType CreateSerializableData()
        {
            var writeData = new per_material_animType();
            if (this.@ParamAnims.Count > 0)
            {
                writeData.@shader_param_anim_array = new shader_param_anim_arrayType();
                writeData.@shader_param_anim_array.Items = this.@paramAnims.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@shader_param_anim_array.Items.Length == 0)
                {
                    writeData.@shader_param_anim_array = null;
                }
            }
            if (this.@PatternAnims.Count > 0)
            {
                writeData.@tex_pattern_anim_array = new tex_pattern_anim_arrayType();
                writeData.@tex_pattern_anim_array.Items = this.@patternAnims.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@tex_pattern_anim_array.Items.Length == 0)
                {
                    writeData.@tex_pattern_anim_array = null;
                }
            }
            if (this.@materialVisibilityAnim != null)
            {
                writeData.@material_visibility_anim = this.MaterialVisibilityAnim.CreateSerializableData() as material_visibility_animType;
            }
            writeData.@mat_name = this.MatName;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        PerMaterialAnim IDeepCopyable<PerMaterialAnim>.DeepCopy()
        {
            return new PerMaterialAnim(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(PerMaterialAnim source)
        {
            CopyElements(source.@paramAnims, this.@paramAnims);
            CopyElements(source.@patternAnims, this.@patternAnims);
            if (source.@materialVisibilityAnim == null)
            {
                this.MaterialVisibilityAnim = null;
            }
            else
            {
                this.MaterialVisibilityAnim.DeepCopyFrom(source.@materialVisibilityAnim);
            }
            this.@matName = source.@matName;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            foreach (var elem in this.ParamAnims)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.PatternAnims)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            if (this.MaterialVisibilityAnim != null)
            {
                buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.MaterialVisibilityAnim.HashValue)));
            }
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.MatName)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            foreach (var elem in this.@paramAnims)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@patternAnims)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            if (this.@materialVisibilityAnim != null)
            {
                this.@materialVisibilityAnim.AutoCalc = this.AutoCalc;
            }
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            foreach (var elem in this.@paramAnims)
            {
                elem.Reset();
            }
            foreach (var elem in this.@patternAnims)
            {
                elem.Reset();
            }
            if (this.@materialVisibilityAnim != null)
            {
                this.@materialVisibilityAnim.Reset();
            }
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            foreach (var elem in this.@paramAnims)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@patternAnims)
            {
                elem.Refresh();
            }
            if (this.@materialVisibilityAnim != null)
            {
                this.@materialVisibilityAnim.Refresh();
            }
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
