﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class LodOffset : ObservableEntity<lod_offsetType>, IDeepCopyable<LodOffset>, IDeepCopyFrom<LodOffset>
    {
        private readonly ObservableList<float> values = new ObservableList<float>();

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public LodOffset()
        {
            this.values.CollectionChanged += OnValueCollectionChanged;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public LodOffset(LodOffset source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public LodOffset(lod_offsetType source)
        {
            this.values.CollectionChanged += OnValueCollectionChanged;
            this.values.Clear();
            this.values.Add(StringUtility.ConvertTextToValueArray<float>(source.@Value));
        }

        public ObservableList<float> Values
        {
            get
            {
                return this.values;
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override lod_offsetType CreateSerializableData()
        {
            var writeData = new lod_offsetType();
            if (this.Values.Count == 0)
            {
                return null;
            }

            writeData.Value = StringUtility.ConvertValueArrayToText(this.Values);
            writeData.count = this.Values.Count;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        LodOffset IDeepCopyable<LodOffset>.DeepCopy()
        {
            return new LodOffset(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(LodOffset source)
        {
            CopyValues(source.values, this.values);
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            foreach (var elem in this.Values)
            {
                buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(elem)));
            }
            return crc.ComputeHashUInt32(buffers.ToArray());
        }
    }
}
