﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;
namespace Nintendo.G3dTool.Entities{
    public class Float3 : ObservableEntity<string>, IVector, IEquatable<Float3>
    {
        private readonly float[] values = new float[3];

        public Float3()
        {
        }

        public Float3(
            float elem0,
            float elem1,
            float elem2)
            : this()
        {
            this[0] = elem0;
            this[1] = elem1;
            this[2] = elem2;
        }

        internal float[] Values
        {
            get
            {
                return this.values;
            }
        }

        public float this[int index]
        {
            get
            {
                Nintendo.ToolFoundation.Contracts.Ensure.Argument.Range(index, 0, 3);
                return this.values[index];
            }

            set
            {
                Nintendo.ToolFoundation.Contracts.Ensure.Argument.Range(index, 0, 3);
                this.SetProperty(ref this.values[index], value, () => this.CalcCRC());
            }
        }


        public int Dimension
        {
            get
            {
                return 3;
            }
        }

        /// <summary>
        /// 行列とスカラーを乗算します。
        /// </summary>
        /// <param name="mtx">演算の左の行列です。</param>
        /// <param name="scalar">演算の右のスカラーです。</param>
        /// <returns>演算結果です。</returns>
        public static Float3 operator *(Float3 vec, float scalar)
        {
            var result = new Float3();
            for (int i = 0; i < vec.Dimension; ++i)
            {
                result[i] = vec[i] * scalar;
            }

            return result;
        }

        public float X
        {
            get
            {
                return this.values[0];
            }

            set
            {
                this.SetProperty(ref this.values[0], value, () => this.CalcCRC());
            }
        }

        public float Y
        {
            get
            {
                return this.values[1];
            }

            set
            {
                this.SetProperty(ref this.values[1], value, () => this.CalcCRC());
            }
        }

        public float Z
        {
            get
            {
                return this.values[2];
            }

            set
            {
                this.SetProperty(ref this.values[2], value, () => this.CalcCRC());
            }
        }

        public void SetToZero()
        {
            for (int elemIndex = 0; elemIndex < 3; ++elemIndex)
            {
                this[elemIndex] = 0;
            }
        }

        /// <summary>
        /// 比較演算子
        /// </summary>
        /// <param name="obj"></param>
        /// <returns>一致していれば true、そうでなければ false を返します。</returns>
        public bool Equals(Float3 obj)
        {
            if (obj == null)
            {
                return false;
            }

            bool isEqual = true;
            for (int elemIndex = 0; elemIndex < 3; ++elemIndex)
            {
                isEqual &= (obj[elemIndex] == this[elemIndex]);
            }

            return isEqual;
        }

        public override string CreateSerializableData()
        {
            return this.ToString();
        }

        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            return crc.ComputeHashUInt32(this.ToString());
        }

        public override string ToString()
        {
            string result = string.Empty;
            for (int elemIndex = 0; elemIndex < 3; ++elemIndex)
            {
                result += Convert.ToDouble(this[elemIndex]).ToString() + " ";
            }

            return result.TrimEnd();
        }

        public void DeepCopyFrom(Float3 source)
        {
            for (int elemIndex = 0; elemIndex < 3; ++elemIndex)
            {
                this[elemIndex] = source.Values[elemIndex];
            }
        }

        public static Float3 Parse(string text)
        {
            string[] valueTexts = StringUtility.SplitValueListText(text);
            Float3 result = new Float3();
            for (int elemIndex = 0; elemIndex < 3; ++elemIndex)
            {
                result[elemIndex] = float.Parse(valueTexts[elemIndex]);
            }

            return result;
        }
    }
}
