﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class CombinerShader : RootEntity<combiner_shaderType>, IDeepCopyable<CombinerShader>, IDeepCopyFrom<CombinerShader>, IToolData, IDisposable
    {
        private readonly ObservableList<ProcessLog> @processLogs = new ObservableList<ProcessLog>();
        private readonly ObservableList<UserData> @userDatas = new ObservableList<UserData>();
        private readonly Comment @comment = new Comment();
        private readonly ToolData @toolData = new ToolData();
        private readonly UserToolData @userToolData = new UserToolData();
        private readonly ObservableList<Stream> @streams = new ObservableList<Stream>();
        private Material @material = null;
        private readonly ObservableList<Block> @blocks = new ObservableList<Block>();
        private readonly ObservableList<Connection> @connections = new ObservableList<Connection>();
        private readonly ObservableList<Uniform> @uniforms = new ObservableList<Uniform>();
        private combiner_shader_versionType @version;
        private int @x;
        private int @y;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        internal CombinerShader()
        {
            this.@processLogs.CollectionChanged += OnCollectionChanged<ProcessLog>;
            this.@userDatas.CollectionChanged += OnCollectionChanged<UserData>;
            this.@comment.PropertyChanged += this.OnPropertyChanged;
            this.@toolData.PropertyChanged += this.OnPropertyChanged;
            this.@userToolData.PropertyChanged += this.OnPropertyChanged;
            this.@streams.CollectionChanged += OnCollectionChanged<Stream>;
            this.@blocks.CollectionChanged += OnCollectionChanged<Block>;
            this.@connections.CollectionChanged += OnCollectionChanged<Connection>;
            this.@uniforms.CollectionChanged += OnCollectionChanged<Uniform>;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public CombinerShader(CombinerShader source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public CombinerShader(combiner_shaderType source)
        {
            this.@processLogs.CollectionChanged += OnCollectionChanged<ProcessLog>;
            this.@userDatas.CollectionChanged += OnCollectionChanged<UserData>;
            this.@streams.CollectionChanged += OnCollectionChanged<Stream>;
            this.@blocks.CollectionChanged += OnCollectionChanged<Block>;
            this.@connections.CollectionChanged += OnCollectionChanged<Connection>;
            this.@uniforms.CollectionChanged += OnCollectionChanged<Uniform>;
            if (source.@process_log_array != null)
            {
                this.@processLogs.Clear();
                foreach (var elem in source.@process_log_array.Items)
                {
                    this.@processLogs.Add(new ProcessLog(elem));
                }
            }
            if (source.@user_data_array != null)
            {
                this.@userDatas.Clear();
                foreach (var elem in source.@user_data_array.Items)
                {
                    this.@userDatas.Add(DataModelConverter.Convert(elem));
                }
            }
            if (source.@comment != null)
            {
                this.@comment = new Comment(source.@comment);
            }
            else
            {
                this.@comment = new Comment();
            }
            if (source.@tool_data != null)
            {
                this.@toolData = new ToolData(source.@tool_data);
            }
            else
            {
                this.@toolData = new ToolData();
            }
            if (source.@user_tool_data != null)
            {
                this.@userToolData = new UserToolData(source.@user_tool_data);
            }
            else
            {
                this.@userToolData = new UserToolData();
            }
            if (source.@stream_array != null)
            {
                this.@streams.Clear();
                foreach (var elem in source.@stream_array.Items)
                {
                    this.@streams.Add(DataModelConverter.Convert(elem));
                }
            }
            if (source.@material != null)
            {
                this.Material = new Material(source.@material);
            }
            else
            {
                this.Material = null;
            }
            if (source.@block_array != null)
            {
                this.@blocks.Clear();
                foreach (var elem in source.@block_array.Items)
                {
                    this.@blocks.Add(new Block(elem));
                }
            }
            if (source.@connection_array != null)
            {
                this.@connections.Clear();
                foreach (var elem in source.@connection_array.Items)
                {
                    this.@connections.Add(new Connection(elem));
                }
            }
            if (source.@uniform_array != null)
            {
                this.@uniforms.Clear();
                foreach (var elem in source.@uniform_array.Items)
                {
                    this.@uniforms.Add(new Uniform(elem));
                }
            }
            this.@version = source.@version;
            this.@x = source.@x;
            this.@y = source.@y;
            this.@comment.PropertyChanged += this.OnPropertyChanged;
            this.@toolData.PropertyChanged += this.OnPropertyChanged;
            this.@userToolData.PropertyChanged += this.OnPropertyChanged;
            this.ResolveInternalReferences();
        }

        public void Dispose()
        {
            if (this.@material != null)
            {
                this.@material.PropertyChanged -= this.OnPropertyChanged;
            }
        }

        public ObservableList<ProcessLog> ProcessLogs
        {
            get
            {
                return this.@processLogs;
            }
        }

        public ObservableList<UserData> UserDatas
        {
            get
            {
                return this.@userDatas;
            }
        }

        public Comment Comment
        {
            get
            {
                return this.@comment;
            }
        }

        public ToolData ToolData
        {
            get
            {
                return this.@toolData;
            }
        }

        public UserToolData UserToolData
        {
            get
            {
                return this.@userToolData;
            }
        }

        public override ObservableList<Stream> Streams
        {
            get
            {
                return this.@streams;
            }
        }

        public Material Material
        {
            get
            {
                return this.@material;
            }

            set
            {
                if (this.@material == value)
                {
                    return;
                }

                if (this.@material != null)
                {
                    this.@material.PropertyChanged -= this.OnPropertyChanged;
                }

                if (value != null)
                {
                    value.PropertyChanged += this.OnPropertyChanged;
                }

                this.SetProperty(ref this.@material, value, () => this.CalcCRC());
            }
        }

        public ObservableList<Block> Blocks
        {
            get
            {
                return this.@blocks;
            }
        }

        public ObservableList<Connection> Connections
        {
            get
            {
                return this.@connections;
            }
        }

        public ObservableList<Uniform> Uniforms
        {
            get
            {
                return this.@uniforms;
            }
        }

        public combiner_shader_versionType Version
        {
            get
            {
                return this.@version;
            }

            set
            {
                if (this.@version == value)
                {
                    return;
                }

                this.SetProperty(ref this.@version, value, () => this.CalcCRC());
            }
        }

        public int X
        {
            get
            {
                return this.@x;
            }

            set
            {
                if (this.@x == value)
                {
                    return;
                }

                this.SetProperty(ref this.@x, value, () => this.CalcCRC());
            }
        }

        public int Y
        {
            get
            {
                return this.@y;
            }

            set
            {
                if (this.@y == value)
                {
                    return;
                }

                this.SetProperty(ref this.@y, value, () => this.CalcCRC());
            }
        }

        public override void ResolveInternalReferences()
        {
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override combiner_shaderType CreateSerializableData()
        {
            var writeData = new combiner_shaderType();
            if (this.@ProcessLogs.Count > 0)
            {
                writeData.@process_log_array = new process_log_arrayType();
                writeData.@process_log_array.Items = this.@processLogs.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@process_log_array.Items.Length == 0)
                {
                    writeData.@process_log_array = null;
                }
            }
            if (this.@UserDatas.Count > 0)
            {
                writeData.@user_data_array = new user_data_arrayType();
                writeData.@user_data_array.Items = this.@userDatas.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@user_data_array.Items.Length == 0)
                {
                    writeData.@user_data_array = null;
                }
            }
            writeData.@comment = this.@comment.CreateSerializableData() as commentType;
            writeData.@tool_data = this.@toolData.CreateSerializableData() as tool_dataType;
            writeData.@user_tool_data = this.@userToolData.CreateSerializableData() as user_tool_dataType;
            if (this.IsStreamSerializationEnabled)
            {
                if (this.@Streams.Count > 0)
                {
                    writeData.@stream_array = new stream_arrayType();
                    writeData.@stream_array.Items = this.@streams.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                    if (writeData.@stream_array.Items.Length == 0)
                    {
                        writeData.@stream_array = null;
                    }
                }
            }
            if (this.@material != null)
            {
                writeData.@material = this.Material.CreateSerializableData() as materialType;
            }
            if (this.@Blocks.Count > 0)
            {
                writeData.@block_array = new block_arrayType();
                writeData.@block_array.Items = this.@blocks.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@block_array.Items.Length == 0)
                {
                    writeData.@block_array = null;
                }
            }
            if (this.@Connections.Count > 0)
            {
                writeData.@connection_array = new connection_arrayType();
                writeData.@connection_array.Items = this.@connections.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@connection_array.Items.Length == 0)
                {
                    writeData.@connection_array = null;
                }
            }
            if (this.@Uniforms.Count > 0)
            {
                writeData.@uniform_array = new uniform_arrayType();
                writeData.@uniform_array.Items = this.@uniforms.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@uniform_array.Items.Length == 0)
                {
                    writeData.@uniform_array = null;
                }
            }
            writeData.@version = this.Version;
            writeData.@x = this.X;
            writeData.@y = this.Y;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        CombinerShader IDeepCopyable<CombinerShader>.DeepCopy()
        {
            return new CombinerShader(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(CombinerShader source)
        {
            CopyElements(source.@processLogs, this.@processLogs);
            CopyElements(source.@userDatas, this.@userDatas);
            this.@comment.DeepCopyFrom(source.@comment);
            this.@toolData.DeepCopyFrom(source.@toolData);
            this.@userToolData.DeepCopyFrom(source.@userToolData);
            CopyElements(source.@streams, this.@streams);
            if (this.@material != null)
            {
                this.@material.DeepCopyFrom(source.@material);
            }
            CopyElements(source.@blocks, this.@blocks);
            CopyElements(source.@connections, this.@connections);
            CopyElements(source.@uniforms, this.@uniforms);
            this.@version = source.@version;
            this.@x = source.@x;
            this.@y = source.@y;
            this.ResolveInternalReferences();
        }

        private protected override IRootEntity DeepCopy()
        {
            return new CombinerShader(this);
        }

        private protected override void DeepCopyFrom(IRootEntity source)
        {
            Nintendo.ToolFoundation.Contracts.Ensure.Argument.True(source is CombinerShader);
            this.DeepCopyFrom(source as CombinerShader);
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            foreach (var elem in this.ProcessLogs)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.UserDatas)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Comment.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.ToolData.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.UserToolData.HashValue)));
            foreach (var elem in this.Streams)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            if (this.Material != null)
            {
                buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Material.HashValue)));
            }
            foreach (var elem in this.Blocks)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.Connections)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.Uniforms)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Version)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.X)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Y)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            foreach (var elem in this.@processLogs)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@userDatas)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            this.@comment.AutoCalc = this.AutoCalc;
            this.@toolData.AutoCalc = this.AutoCalc;
            this.@userToolData.AutoCalc = this.AutoCalc;
            foreach (var elem in this.@streams)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            if (this.@material != null)
            {
                this.@material.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@blocks)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@connections)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@uniforms)
            {
                elem.AutoCalc = this.AutoCalc;
            }
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            foreach (var elem in this.@processLogs)
            {
                elem.Reset();
            }
            foreach (var elem in this.@userDatas)
            {
                elem.Reset();
            }
            this.@comment.Reset();
            this.@toolData.Reset();
            this.@userToolData.Reset();
            foreach (var elem in this.@streams)
            {
                elem.Reset();
            }
            if (this.@material != null)
            {
                this.@material.Reset();
            }
            foreach (var elem in this.@blocks)
            {
                elem.Reset();
            }
            foreach (var elem in this.@connections)
            {
                elem.Reset();
            }
            foreach (var elem in this.@uniforms)
            {
                elem.Reset();
            }
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            foreach (var elem in this.@processLogs)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@userDatas)
            {
                elem.Refresh();
            }
            this.@comment.Refresh();
            this.@toolData.Refresh();
            this.@userToolData.Refresh();
            foreach (var elem in this.@streams)
            {
                elem.Refresh();
            }
            if (this.@material != null)
            {
                this.@material.Refresh();
            }
            foreach (var elem in this.@blocks)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@connections)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@uniforms)
            {
                elem.Refresh();
            }
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
