﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class Block : ObservableEntity<blockType>, IDeepCopyable<Block>, IDeepCopyFrom<Block>
    {
        private readonly Const @const = new Const();
        private string @id = string.Empty;
        private string @guid = string.Empty;
        private float @x;
        private float @y;
        private int @w;
        private int @h;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public Block()
        {
            this.@const.PropertyChanged += this.OnPropertyChanged;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Block(Block source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Block(blockType source)
        {
            if (source.@const != null)
            {
                this.@const = new Const(source.@const);
            }
            else
            {
                this.@const = new Const();
            }
            this.@id = source.@id;
            this.@guid = source.@guid;
            this.@x = source.@x;
            this.@y = source.@y;
            this.@w = source.@w;
            this.@h = source.@h;
            this.@const.PropertyChanged += this.OnPropertyChanged;
        }

        public Const Const
        {
            get
            {
                return this.@const;
            }
        }

        public string Id
        {
            get
            {
                return this.@id;
            }

            set
            {
                if (this.@id == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@id, value, () => this.CalcCRC());
            }
        }

        public string Guid
        {
            get
            {
                return this.@guid;
            }

            set
            {
                if (this.@guid == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@guid, value, () => this.CalcCRC());
            }
        }

        public float X
        {
            get
            {
                return this.@x;
            }

            set
            {
                if (this.@x == value)
                {
                    return;
                }

                this.SetProperty(ref this.@x, value, () => this.CalcCRC());
            }
        }

        public float Y
        {
            get
            {
                return this.@y;
            }

            set
            {
                if (this.@y == value)
                {
                    return;
                }

                this.SetProperty(ref this.@y, value, () => this.CalcCRC());
            }
        }

        public int W
        {
            get
            {
                return this.@w;
            }

            set
            {
                if (this.@w == value)
                {
                    return;
                }

                this.SetProperty(ref this.@w, value, () => this.CalcCRC());
            }
        }

        public int H
        {
            get
            {
                return this.@h;
            }

            set
            {
                if (this.@h == value)
                {
                    return;
                }

                this.SetProperty(ref this.@h, value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override blockType CreateSerializableData()
        {
            var writeData = new blockType();
            writeData.@const = this.@const.CreateSerializableData() as constType;
            writeData.@id = this.Id;
            writeData.@guid = this.Guid;
            writeData.@x = this.X;
            writeData.@y = this.Y;
            writeData.@w = this.W;
            writeData.@h = this.H;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        Block IDeepCopyable<Block>.DeepCopy()
        {
            return new Block(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(Block source)
        {
            this.@const.DeepCopyFrom(source.@const);
            this.@id = source.@id;
            this.@guid = source.@guid;
            this.@x = source.@x;
            this.@y = source.@y;
            this.@w = source.@w;
            this.@h = source.@h;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Const.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Id)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Guid)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.X)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Y)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.W)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.H)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            this.@const.AutoCalc = this.AutoCalc;
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            this.@const.Reset();
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            this.@const.Refresh();
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
