﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef __ShellExtension_FileData_H__
#define __ShellExtension_FileData_H__

//==============================================================================
//
// Forward declaration
//
//==============================================================================
class CShellExtensionManager;
class CNWTexture;

//==============================================================================
//
// CShellExtensionFileData
//
// - This class contains data used by Explorer for one particular file.
//   Such as Texture, Icon, Bitmap etc
//   This maybe shared by multiple service ( eg. Icon, Preview ) and be used from
//   different thread. Therefore some operations like loading texture, are using
//   CriticalSection to prevent collapsing data and dead lock.
//
//==============================================================================
class CShellExtensionFileData
{
    friend class CShellExtensionManager;

public:
    CShellExtensionFileData( CShellExtensionManager *pManager,
                         const WCHAR* szFileName );
    virtual ~CShellExtensionFileData();

    //--------------------------------------------------------------------------
    // Add Reference
    //
    // - szCaller is to identify who is calling ( for debugging )
    //--------------------------------------------------------------------------
    void AddRef( const WCHAR *szCaller );

    //--------------------------------------------------------------------------
    // Release ( when ref count reaches zero, remove from Manager and free memory )
    //
    // - szCaller is to identify who is calling Release ( for debugging )
    //--------------------------------------------------------------------------
    void Release( const WCHAR *szCaller );

    //--------------------------------------------------------------------------
    // RefCount
    //--------------------------------------------------------------------------
    LONG GetRefCount() const { return m_refCount; };

    //--------------------------------------------------------------------------
    // FileName
    //--------------------------------------------------------------------------
    const std::wstring& GetFileName() const { return m_fileName; }

    //--------------------------------------------------------------------------
    // Texture
    //--------------------------------------------------------------------------
    CNWTexture*       GetTexture();

    //--------------------------------------------------------------------------
    // Load ImageData from file
    //
    // - If bLoadPreview is true, create preview image for icon
    //   If bLoadPreview is false, just load information ( format, size etc )
    //--------------------------------------------------------------------------
    bool UpdateImage( bool bLoadPreview );
    void DestroyImage();

    //--------------------------------------------------------------------------
    // Get or Create an icon for requested size
    // If bWithInfo is set to true, icon will contain information text
    // ( eg. texture size, format etc )
    //--------------------------------------------------------------------------
    HICON GetIcon( int iconWidth, int iconHeight, bool bWithInfo );

    //--------------------------------------------------------------------------
    // Get or Create an small icon for requested size
    //--------------------------------------------------------------------------
    HICON GetSmallIcon( int iconWidth, int iconHeight );

    //--------------------------------------------------------------------------
    // Get or Create an bmp for requested size
    // If bWithInfo is set to true, icon will contain information text
    // ( eg. texture size, format etc )
    //--------------------------------------------------------------------------
    HBITMAP GetBitmap( int bmpWidth, int bmpHeight, bool bWithInfo );

    //--------------------------------------------------------------------------
    // Get critical section for this file data
    //--------------------------------------------------------------------------
    CNWCriticalSection* GetCriticalSection() { return &m_CS; };

private :
    CShellExtensionManager *m_pManager;

    volatile LONG m_refCount;

    std::wstring  m_fileName;

    //--------------------------------------------------------------------------
    // Critical Section
    //--------------------------------------------------------------------------
    CNWCriticalSection m_CS;

    //--------------------------------------------------------------------------
    // Texture
    //--------------------------------------------------------------------------
    CNWTexture *m_pTexture;

    //--------------------------------------------------------------------------
    // Render Image
    //--------------------------------------------------------------------------
    void RenderImage( Gdiplus::Bitmap *pDestBitmap,
                      int width,
                      int height,
                      bool bSmall = false)const;
    void RenderImageWithInfo( Gdiplus::Bitmap *pDestBitmap,
                              int width,
                              int height )const;
    void RenderCaption( Gdiplus::Bitmap *pDestBitmap,
                        int width,
                        int height )const;
    void RenderCaptionWithInfo( Gdiplus::Bitmap *pDestBitmap,
                                int width,
                                int height )const;
}; // End of CShellExtensionFileData

#endif // End of __ShellExtension_FileData_H__
