﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ShellExtension_PCH.h"
#include "ShellExtension_Resource.h"
#include "ShellExtension_Utility.h"
#include "ShellExtension_SyncObject.h"
#include "ShellExtension_Texture.h"
#include "ShellExtension_FileData.h"
#include "ShellExtension_Manager.h"

//==============================================================================
//
// Implementation of CShellExtensionFileData
//
//==============================================================================
CShellExtensionFileData::CShellExtensionFileData( CShellExtensionManager *pManager,
                                          const WCHAR* szFileName ) :
    m_pManager(pManager),
    m_refCount(0),
    m_fileName(szFileName),
    m_pTexture(NULL)
{
} // End of Constructor for CShellExtensionFileData


//------------------------------------------------------------------------------
CShellExtensionFileData::~CShellExtensionFileData()
{
    DestroyImage();
} // End of Destructor for CShellExtensionFileData


//------------------------------------------------------------------------------
// Add Reference
//
// - szCaller is to identify who is calling ( for debugging )
//------------------------------------------------------------------------------
void CShellExtensionFileData::AddRef( const WCHAR *szCaller )
{
    NW_USE_VAR(szCaller);
    ::InterlockedIncrement(&m_refCount);

    // For debugging
    //NW_LOG(L"CShellExtensionFileData::AddRef --> %s %s 0x%p %d", szCaller, m_fileName.c_str(), this, (int)m_refCount);
} // End of AddRef for CShellExtensionFileData


//------------------------------------------------------------------------------
// Release ( when ref count reaches zero, remove from Manager and free memory )
//
// - szCaller is to identify who is calling Release ( for debugging )
//------------------------------------------------------------------------------
void CShellExtensionFileData::Release( const WCHAR *szCaller )
{
    NW_USE_VAR(szCaller);
    CShellExtensionManager::Instance()->GetFileDataCS()->Lock();

    ::InterlockedDecrement(&m_refCount);

    // For debugging
    //NW_LOG(L"CShellExtensionFileData::Release --> %s %s 0x%p %d", szCaller, m_fileName.c_str(), this, (int)m_refCount);

    if (m_refCount==0)
    {
        // Remove this file data from manager
        if (m_pManager!=NULL)
        {
            m_pManager->RemoveFileData(this);
            m_pManager = NULL;
        } // End if

        delete this;
    } // End if

    CShellExtensionManager::Instance()->GetFileDataCS()->Unlock();
} // End of Release for CShellExtensionFileData


//------------------------------------------------------------------------------
// Destroy Bitmap
//------------------------------------------------------------------------------
void CShellExtensionFileData::DestroyImage()
{
    CNWSingleLock lock(&m_CS);

    if (m_pTexture!=NULL)
    {
        delete m_pTexture;
        m_pTexture = NULL;
    } // End if
} // End of DestroyImage for CShellExtensionFileData


//------------------------------------------------------------------------------
// Update ImageData from file
//
// - If bLoadPreview is true, create preview image for icon
//   If bLoadPreview is false, just load information ( format, size etc )
//------------------------------------------------------------------------------
bool CShellExtensionFileData::UpdateImage( bool bLoadPreview )
{
    CNWSingleLock lock(&m_CS);

    m_pTexture = new CNWTexture();
    if (m_pTexture->Load(m_fileName.c_str(),bLoadPreview)==false)
    {
        DestroyImage();
        return false;
    } // End if

    return true;
} // End of UpdateImage for CShellExtensionFileData


//------------------------------------------------------------------------------
// Texture
//------------------------------------------------------------------------------
CNWTexture* CShellExtensionFileData::GetTexture()
{
    return m_pTexture;
} // End of GetTexture for CShellExtensionFileData


//------------------------------------------------------------------------------
// Render Image to destination bitmap
//------------------------------------------------------------------------------
void CShellExtensionFileData::RenderImage( Gdiplus::Bitmap *pDestBitmap,
                                       int width,
                                       int height,
                                       bool bSmall) const
{
    std::unique_ptr<Gdiplus::Graphics> pGraphics(Gdiplus::Graphics::FromImage(pDestBitmap));
    Gdiplus::SolidBrush blackBrush(Gdiplus::Color(255,204,204,204));
    Gdiplus::SolidBrush whiteBrush(Gdiplus::Color(255,255,255,255));

    if (m_pTexture->HasAlpha())
        pGraphics->SetCompositingMode(Gdiplus::CompositingModeSourceOver);
    else
        pGraphics->SetCompositingMode(Gdiplus::CompositingModeSourceCopy);

    pGraphics->SetInterpolationMode(Gdiplus::InterpolationModeBilinear);

    // Now draw actual image
    int textureWidth = 0, textureHeight = 0;
    switch(m_pTexture->GetTextureType())
    {
    case NW_IMG_TYPE_2D:
        textureWidth = m_pTexture->GetWidth();
        textureHeight = m_pTexture->GetHeight();
        break;
    default:
        textureWidth = m_pTexture->GetPreviewWidth();
        textureHeight = m_pTexture->GetPreviewHeight();
        break;
    }

    //Fit texture image to icon image
    float drawIconImgRatio = ((float)width  / (float)textureWidth < (float)height / (float)textureHeight)
        ? (float)width  / (float)textureWidth : (float)height / (float)textureHeight;

    int drawImgWd = (int)(textureWidth * drawIconImgRatio);
    int drawImgHt = (int)(textureHeight * drawIconImgRatio);

    if (drawImgWd<1)
        drawImgWd = 1;

    if (drawImgHt<1)
        drawImgHt = 1;

    // In case image has no size ( not loaded? )
    if (m_pTexture->HasPreviewImage()==false)
    {
        drawImgWd = 0;
        drawImgHt = 0;
    } // End if

    int drawImgX = ( width  / 2 ) - ( drawImgWd / 2 );
    int drawImgY = ( height / 2 ) - ( drawImgHt / 2 );

    Gdiplus::RectF drawRect((float)drawImgX,(float)drawImgY,(float)drawImgWd,(float)drawImgHt);
    pGraphics->SetClip(drawRect);

    // Draw background checker board if texture has alpha
    if (m_pTexture->HasAlpha())
    {
        int iTileSizeX = max(1,(width   / 8));
        int iTileSizeY = max(1,(height  / 8));

        int x = 0;
        int y = 0;
        int tileIndexX = 0;
        int tileIndexY = 0;

        while (y<height)
        {
            x = 0;
            tileIndexX = 0;
            while (x<width)
            {
                Gdiplus::SolidBrush *pBrush;
                if ((tileIndexX+tileIndexY)&0x01)
                    pBrush = &whiteBrush;
                else
                    pBrush = &blackBrush;

                pGraphics->FillRectangle(pBrush,Gdiplus::Rect(x,y,iTileSizeX,iTileSizeY));
                x += iTileSizeX;
                tileIndexX++;
            } // End while

            y += iTileSizeY;
            tileIndexY++;
        } // End while
    } // End if
    else
    {
        pGraphics->Clear(Gdiplus::Color(0,0,0,0));
    } // End else

    float drawPreviewImgRatio = ((float)m_pTexture->GetPreviewWidth()  / (float)m_pTexture->GetWidth()
        < (float)m_pTexture->GetPreviewHeight() / (float)m_pTexture->GetHeight())
        ? (float)m_pTexture->GetPreviewWidth()  / (float)m_pTexture->GetWidth()
        : (float)m_pTexture->GetPreviewHeight() / (float)(float)m_pTexture->GetHeight();

    // Draw Image with blending
    float textureX = (m_pTexture->GetPreviewWidth() - m_pTexture->GetWidth() * drawPreviewImgRatio)/2.f;
    float textureY = (m_pTexture->GetPreviewHeight() - m_pTexture->GetHeight() * drawPreviewImgRatio)/2.f;

    //あとで変数名修正の必要あり
    float textureWidth2,textureHeight2;
    switch(m_pTexture->GetTextureType())
    {
    case NW_IMG_TYPE_1D_ARRAY:
        textureX = 0.f;
        textureY = 0.f;
        textureWidth2 = (float)m_pTexture->GetPreviewWidth();
        textureHeight2 = (float)m_pTexture->GetPreviewHeight();
        break;
    default:
        textureWidth2 = (float)m_pTexture->GetWidth();
        textureHeight2 = (float)m_pTexture->GetHeight();
        break;
    }
    if(!bSmall)
    {
#ifdef NW_FOR_PSD
        // PSDは現在Color+Alphaに対応していないため、とりあえずColorを使用
        pGraphics->DrawImage(m_pTexture->GetPreviewColorBitmap(),drawRect,
#else
       // GetPreviewBitmapでColor+Alpha画像を取得
        pGraphics->DrawImage(m_pTexture->GetPreviewBitmap(),drawRect,
#endif //NW_FOR_FTX
                             textureX, textureY,
                             textureWidth2 * drawPreviewImgRatio,
                             textureHeight2 * drawPreviewImgRatio,
                             Gdiplus::UnitPixel);
    }
    else
    {
#ifdef NW_FOR_PSD
        // PSDは現在Color+Alphaに対応していないため、とりあえずColorを使用
        pGraphics->DrawImage(m_pTexture->GetPreviewColorSmallBitmap(),drawRect,
#else
        // GetPreviewSmallBitmapでColor+Alpha画像を取得
        pGraphics->DrawImage(m_pTexture->GetPreviewSmallBitmap(),drawRect,
#endif //NW_FOR_FTX
                         textureX, textureY,
                         (float)m_pTexture->GetPreviewWidth(),(float)m_pTexture->GetPreviewHeight(),
                         Gdiplus::UnitPixel);
    }
    pGraphics->Flush(Gdiplus::FlushIntentionSync);
} // End of RenderImage for CShellExtensionFileData


//------------------------------------------------------------------------------
// Render Image to destination bitmap with extra info like
// texture format, size, mipmap count etc
//------------------------------------------------------------------------------
void CShellExtensionFileData::RenderImageWithInfo( Gdiplus::Bitmap *pDestBitmap,
                                               int width,
                                               int height )const
{
    std::unique_ptr<Gdiplus::Graphics> pGraphics(Gdiplus::Graphics::FromImage(pDestBitmap));
    Gdiplus::SolidBrush grayBrush(Gdiplus::Color(255,204,204,204));

    pGraphics->SetCompositingMode(Gdiplus::CompositingModeSourceCopy);
    pGraphics->SetInterpolationMode(Gdiplus::InterpolationModeNearestNeighbor);
    pGraphics->SetCompositingQuality(Gdiplus::CompositingQualityHighQuality);

    pGraphics->Clear(Gdiplus::Color(255,192,192,192));

    float dpyYNormal = 96.0f;
    float dpyY = pGraphics->GetDpiY();
    float textScaleRatioY = dpyYNormal/ dpyY;

    float halfWd = (float)width  / (float)2.0f;
    float halfHt = (float)height / (float)2.0f;

    // Now draw actual image
    float drawImgX  = 0.0f;
    float drawImgY  = 0.0f;
    float drawImgWd = halfWd;
    float drawImgHt = halfHt;

    // In case image has no size ( not loaded? )
    if (m_pTexture->HasPreviewImage()==false)
    {
        drawImgWd = 0;
        drawImgHt = 0;
    } // End if

    Gdiplus::Pen disabledPen(Gdiplus::Color(255,96,96,96),2.0f);
    Gdiplus::Pen blackPen(Gdiplus::Color(255,0,0,0));

    float textFontHt = textScaleRatioY * halfHt / 6.0f;
    Gdiplus::Font font(L"Courier", textFontHt);

    Gdiplus::SolidBrush textBrush(Gdiplus::Color(255,0,0,0));
    Gdiplus::SolidBrush mipmapWarningBrush(Gdiplus::Color(255,96,0,0));

    // Draw RGB
    Gdiplus::ImageAttributes imgAttr;
    Gdiplus::ColorMatrix colorMat =
    {
        1.f,0.f,0.f,0.f,0.f,
        0.f,1.f,0.f,0.f,0.f,
        0.f,0.f,1.f,0.f,0.f,
        0.f,0.f,0.f,0.f,0.f,
        0.f,0.f,0.f,1.f,0.f,
    };

    imgAttr.SetColorMatrix(&colorMat);

    if ( (m_pTexture->GetFormat()!=NW_IMG_FMT_A4) &&
         (m_pTexture->GetFormat()!=NW_IMG_FMT_A8) )
    {
        Gdiplus::RectF drawRect(drawImgX,drawImgY,drawImgWd,drawImgHt);
        pGraphics->DrawImage(m_pTexture->GetPreviewColorBitmap(),
                             drawRect,
                             -0.5f,-0.5f,
                             (float)m_pTexture->GetPreviewWidth(),(float)m_pTexture->GetPreviewHeight(),
                             Gdiplus::UnitPixel,&imgAttr);
    } // End if
    else
    {
        Gdiplus::RectF drawRect(drawImgX+1,drawImgY+1,drawImgWd-2,drawImgHt-2);

        pGraphics->DrawLine(&disabledPen,drawRect.GetLeft(),drawRect.GetTop(),drawRect.GetRight(),drawRect.GetBottom());
        pGraphics->DrawLine(&disabledPen,drawRect.GetRight(),drawRect.GetTop(),drawRect.GetLeft(),drawRect.GetBottom());

        pGraphics->DrawRectangle(&disabledPen,drawRect);
    } // End else

    // Draw Alpha
    if (m_pTexture->HasAlpha())
    {
        Gdiplus::RectF drawRect(drawImgX+halfWd,drawImgY,drawImgWd,drawImgHt);
        pGraphics->DrawImage(m_pTexture->GetPreviewAlphaBitmap(),drawRect,
                             -0.5f,-0.5f,
                             (float)m_pTexture->GetPreviewWidth(),(float)m_pTexture->GetPreviewHeight(),
                             Gdiplus::UnitPixel,&imgAttr);
    } // End if
    else
    {
        Gdiplus::RectF drawRect(drawImgX+halfWd,drawImgY+1,drawImgWd-2,drawImgHt-2);

        pGraphics->DrawLine(&disabledPen,drawRect.GetLeft(),drawRect.GetTop(),drawRect.GetRight(),drawRect.GetBottom());
        pGraphics->DrawLine(&disabledPen,drawRect.GetRight(),drawRect.GetTop(),drawRect.GetLeft(),drawRect.GetBottom());
        pGraphics->DrawRectangle(&disabledPen,drawRect);
    } // End else

    // Draw Information
    float textHt = ((float)halfHt-2.0f) / 5.0f;
    float textWd = (float)width;
    float rightMargin = 4.0f;
    float topMargin   = 1.0f;

    Gdiplus::StringFormat strFormat(0);

    strFormat.SetAlignment(Gdiplus::StringAlignmentFar);
    strFormat.SetLineAlignment(Gdiplus::StringAlignmentCenter);

    // it seems text rendering do not support CompositingModeSourceCopy mode
    pGraphics->SetCompositingMode(Gdiplus::CompositingModeSourceOver);

    WCHAR szTempStr[256];
    WCHAR szTempStr2[256];

    { // Comment
        std::wstring commentText(m_pTexture->GetComment());
        if (!commentText.empty())
        {
            Gdiplus::SolidBrush blackBrush(Gdiplus::Color(255,30,30,30));
            Gdiplus::SolidBrush whiteBrush(Gdiplus::Color(255,245,245,245));

            Gdiplus::Font commentFont(L"Courier", textScaleRatioY * (float)height / 12.0f);

            Gdiplus::StringFormat commentFormat(0);
            commentFormat.SetAlignment(Gdiplus::StringAlignmentCenter);
            commentFormat.SetLineAlignment(Gdiplus::StringAlignmentNear);

            Gdiplus::RectF shadowRect(2.0f, 2.0f, (float)width, halfHt);
            pGraphics->DrawString(commentText.c_str(),(int)commentText.length(),&commentFont,
                                  shadowRect,&commentFormat,&blackBrush);

            Gdiplus::RectF textRect(0.0f, 0.0f, (float)width, halfHt);
            pGraphics->DrawString(commentText.c_str(),(int)commentText.length(),&commentFont,
                                  textRect,&commentFormat,&whiteBrush);
        } // End if
    } // End

    { // Format
        const WCHAR* szFormatName = NWGetTextureFormatNameW(m_pTexture->GetFormat());
        Gdiplus::RectF textRect(0.0f,halfHt+topMargin,textWd-rightMargin,textHt);

        Gdiplus::Font *pFontToUse = &font;

        Gdiplus::Font smallerFormatFont(L"Courier", textFontHt*0.75f);

        if (wcslen(szFormatName)>14)
        {
            pFontToUse = &smallerFormatFont;
        } // End if

        if ( ( (m_pTexture->GetFormat()==NW_IMG_FMT_ETC1) ||
               (m_pTexture->GetFormat()==NW_IMG_FMT_ETCA) ) &&
             (m_pTexture->GetCompressTypeText().size()>0) )
        {
            _snwprintf_s(szTempStr,255,_TRUNCATE,L"%s(%s)", szFormatName, m_pTexture->GetCompressTypeTextShort().c_str());
            pGraphics->DrawString(szTempStr,(int)wcslen(szTempStr),pFontToUse,
                                  textRect,&strFormat,&textBrush);
        } // End if
        else
        {
            pGraphics->DrawString(szFormatName,(int)wcslen(szFormatName),pFontToUse,
                                  textRect,&strFormat,&textBrush);
        } // End else
    } // End

    // Dimension
    switch(m_pTexture->GetTextureType())
    {
    default:
        _snwprintf_s(szTempStr,255,_TRUNCATE,L"%d x %d", m_pTexture->GetWidth(), m_pTexture->GetHeight());
        break;
    case NW_IMG_TYPE_1D_ARRAY:
    case NW_IMG_TYPE_2D_ARRAY:
    case NW_IMG_TYPE_CUBE_ARRAY:
        _snwprintf_s(szTempStr,255,_TRUNCATE,L"%d x %d x %d", m_pTexture->GetWidth(), m_pTexture->GetHeight(), m_pTexture->GetCount());
        break;
    }

    // Format
    {
        Gdiplus::RectF textRect(0.0f, halfHt + topMargin + textHt*1.0f, textWd - rightMargin, textHt);
        pGraphics->DrawString(szTempStr, (int)wcslen(szTempStr), &font,
            textRect, &strFormat, &textBrush);
    }

    // Mipmaps
    if (m_pTexture->HasMipmaps())
    {
        _snwprintf_s(szTempStr,255,_TRUNCATE,L"Mipmap : %d ", m_pTexture->GetNumMipmaps());

        Gdiplus::RectF textRect(0.0f,halfHt+topMargin+textHt*2.0f,textWd-rightMargin,textHt);
        if (m_pTexture->IsUsingAllMipmaps())
        {
            pGraphics->DrawString(szTempStr,(int)wcslen(szTempStr),&font,
                                    textRect,&strFormat,&textBrush);
        } // End if
        else
        {
            pGraphics->DrawString(szTempStr,(int)wcslen(szTempStr),&font,
                                    textRect,&strFormat,&mipmapWarningBrush);
        } // End if
    } // End if

    { // ByteSize
        NW4FormatNumberStringWithComma(szTempStr,256,m_pTexture->GetTextureDataSize());
        _snwprintf_s(szTempStr2,255,_TRUNCATE,L"%s B", szTempStr);

        Gdiplus::RectF textRect(0.0f,halfHt+topMargin+textHt*3.0f,textWd-rightMargin,textHt);
        pGraphics->DrawString(szTempStr2,(int)wcslen(szTempStr2),&font,
                              textRect,&strFormat,&textBrush);
    } // End

    // Component Selector
    if ( m_pTexture->IsForNW4F() && m_pTexture->HasCompSel() )
    {
        const NW_COMP_SEL &compSel = m_pTexture->GetCompSel();
        std::wstring compSelStr = NWCompSelToString(compSel);

        Gdiplus::RectF textRect(0.0f,halfHt+topMargin+textHt*4.0f,textWd-rightMargin,textHt);
        pGraphics->DrawString(compSelStr.c_str(),(int)compSelStr.length(),&font,
                                textRect,&strFormat,&textBrush);
    } // End if

    pGraphics->Flush(Gdiplus::FlushIntentionSync);
} // End of RenderImageWithInfo for CShellExtensionFileData


//------------------------------------------------------------------------------
// Render Caption to destination bitmap
//------------------------------------------------------------------------------
void CShellExtensionFileData::RenderCaption( Gdiplus::Bitmap *pDestBitmap,
                                         int width,
                                         int height ) const
{
    std::unique_ptr<Gdiplus::Graphics> pGraphics(Gdiplus::Graphics::FromImage(pDestBitmap));

    pGraphics->SetCompositingMode(Gdiplus::CompositingModeSourceOver);
    pGraphics->SetInterpolationMode(Gdiplus::InterpolationModeNearestNeighbor);
    pGraphics->SetCompositingQuality(Gdiplus::CompositingQualityHighQuality);

    pGraphics->Clear(Gdiplus::Color(255,192,192,192));

    float dpyYNormal = 96.0f;
    float dpyY = pGraphics->GetDpiY();
    float textScaleRatioY = dpyYNormal/ dpyY;

    float textFontHt = textScaleRatioY * (float)height / 5.0f;
    Gdiplus::Font font(L"Courier", textFontHt);
    Gdiplus::SolidBrush textBrush(Gdiplus::Color(255,0,0,0));
    Gdiplus::SolidBrush mipmapWarningBrush(Gdiplus::Color(255,96,0,0));

    // Draw Information
    float halfHt = (float)height / 2.0f;
    float textHt = ((float)height + 2.0f) / 3.0f;
    float textWd = (float)width;
    float leftMargin = (float)(width - 4.0f) / 10.0f;

    Gdiplus::StringFormat strFormat(0);

    strFormat.SetAlignment(Gdiplus::StringAlignmentNear);
    strFormat.SetLineAlignment(Gdiplus::StringAlignmentNear);

    { // Caption
        const WCHAR* szCaption = L"No";
        Gdiplus::RectF textRect1(leftMargin,halfHt-textHt,textWd-leftMargin,textHt);
        pGraphics->DrawString(szCaption,(int)wcslen(szCaption),&font,
                              textRect1,&strFormat,&textBrush);

        szCaption = L"Image";

        Gdiplus::RectF textRect3(leftMargin,halfHt,textWd-leftMargin,textHt);
        pGraphics->DrawString(szCaption,(int)wcslen(szCaption),&font,
                              textRect3,&strFormat,&textBrush);
    } // End

    pGraphics->Flush(Gdiplus::FlushIntentionSync);
} // End of RenderCaption for CShellExtensionFileData


//------------------------------------------------------------------------------
// Render Caption to destination bitmap with extra info like
// texture format, size, mipmap count etc
//------------------------------------------------------------------------------
void CShellExtensionFileData::RenderCaptionWithInfo( Gdiplus::Bitmap *pDestBitmap,
                                                 int width,
                                                 int height ) const
{
    std::unique_ptr<Gdiplus::Graphics> pGraphics(Gdiplus::Graphics::FromImage(pDestBitmap));

    pGraphics->SetCompositingMode(Gdiplus::CompositingModeSourceOver);
    pGraphics->SetInterpolationMode(Gdiplus::InterpolationModeNearestNeighbor);
    pGraphics->SetCompositingQuality(Gdiplus::CompositingQualityHighQuality);

    pGraphics->Clear(Gdiplus::Color(255,192,192,192));

    float dpyYNormal = 96.0f;
    float dpyY = pGraphics->GetDpiY();
    float textScaleRatioY = dpyYNormal/ dpyY;

    float halfWd = (float)width  / (float)2.0f;
    float halfHt = (float)height / (float)2.0f;

    // Now draw actual image
    float drawImgX  = 0.0f;
    float drawImgY  = 0.0f;
    float drawImgWd = halfWd;
    float drawImgHt = halfHt;

    // If not squared
    if (m_pTexture->GetPreviewWidth()!=m_pTexture->GetPreviewHeight())
    {
        float drawImgWdRatio = (float)halfWd / (float)m_pTexture->GetPreviewWidth();
        float drawImgHtRatio = (float)halfHt / (float)m_pTexture->GetPreviewHeight();

        if (drawImgWdRatio<drawImgHtRatio)
        {
            // Match size scaling with width
            drawImgWd = ((float)halfWd);
            drawImgHt = ((float)m_pTexture->GetPreviewHeight()*drawImgWdRatio);
        } // End if
        else
        {
            // Match size scaling with height
            drawImgWd = ((float)m_pTexture->GetPreviewWidth() *drawImgHtRatio);
            drawImgHt = ((float)halfHt);
        } // End if

        if (drawImgWd<1)
            drawImgWd = 1;

        if (drawImgHt<1)
            drawImgHt = 1;

        drawImgX = ( halfWd / 2.0f ) - ( drawImgWd / 2.0f );
        drawImgY = ( halfHt / 2.0f ) - ( drawImgHt / 2.0f );
    } // End if

    // In case image has no size ( not loaded? )
    if (m_pTexture->HasPreviewImage()==false)
    {
        drawImgWd = 0;
        drawImgHt = 0;
    } // End if

    Gdiplus::Pen disabledPen(Gdiplus::Color(255,160,160,160),2.0f);
    Gdiplus::Pen blackPen(Gdiplus::Color(255,0,0,0));

    float textFontHt = textScaleRatioY * halfHt / 6.0f;
    Gdiplus::Font font(L"Courier", textFontHt);
    Gdiplus::SolidBrush textBrush(Gdiplus::Color(255,0,0,0));
    Gdiplus::SolidBrush mipmapWarningBrush(Gdiplus::Color(255,96,0,0));

    // Draw RGB
    {
        Gdiplus::RectF drawRect(drawImgX+1,drawImgY+1,drawImgWd-2,drawImgHt-2);

        pGraphics->DrawLine(&disabledPen,drawRect.GetLeft(),drawRect.GetTop(),drawRect.GetRight(),drawRect.GetBottom());
        pGraphics->DrawLine(&disabledPen,drawRect.GetRight(),drawRect.GetTop(),drawRect.GetLeft(),drawRect.GetBottom());

        pGraphics->DrawRectangle(&disabledPen,drawRect);
    }

    // Draw Alpha
    {
        Gdiplus::RectF drawRect(drawImgX+halfWd,drawImgY+1,drawImgWd-2,drawImgHt-2);

        pGraphics->DrawLine(&disabledPen,drawRect.GetLeft(),drawRect.GetTop(),drawRect.GetRight(),drawRect.GetBottom());
        pGraphics->DrawLine(&disabledPen,drawRect.GetRight(),drawRect.GetTop(),drawRect.GetLeft(),drawRect.GetBottom());
        pGraphics->DrawRectangle(&disabledPen,drawRect);
    }

    // Draw Caption
    {
        // Draw Information
        float capTextHt     = ((float)halfHt - 5.0f) / 4.2f;
        float capTextWd     = (float)width;
        float capTopMargin  = (float)height / 9.0f;
        float capLeftMargin = (float)width / 4.5f;

        Gdiplus::Font captionFont(L"Courier", textScaleRatioY * (float)height / 13.0f);

        Gdiplus::StringFormat capStrFormat(0);

        capStrFormat.SetAlignment(Gdiplus::StringAlignmentNear);
        capStrFormat.SetLineAlignment(Gdiplus::StringAlignmentNear);

        const WCHAR* szCaption = L"No";
        Gdiplus::RectF textRect1(capLeftMargin,capTopMargin,capTextWd-capLeftMargin,capTextHt);
        pGraphics->DrawString(szCaption,(int)wcslen(szCaption),&captionFont,
                              textRect1,&capStrFormat,&textBrush);

        szCaption = L"Thumbnail";
        Gdiplus::RectF textRect2(capLeftMargin,capTopMargin+capTextHt*1.0f,capTextWd-capLeftMargin,capTextHt);
        pGraphics->DrawString(szCaption,(int)wcslen(szCaption),&captionFont,
                              textRect2,&capStrFormat,&textBrush);

        szCaption = L"Image";
        Gdiplus::RectF textRect3(capLeftMargin,capTopMargin+capTextHt*2.0f,capTextWd-capLeftMargin,capTextHt);
        pGraphics->DrawString(szCaption,(int)wcslen(szCaption),&captionFont,
                              textRect3,&capStrFormat,&textBrush);
    } // End

    // Draw Information
    float textHt      = ((float)halfHt-2.0f) / 5.0f;
    float textWd      = (float)width;
    float rightMargin = 4.0f;
    float topMargin   = 1.0f;

    Gdiplus::StringFormat strFormat(0);

    strFormat.SetAlignment(Gdiplus::StringAlignmentFar);
    strFormat.SetLineAlignment(Gdiplus::StringAlignmentCenter);

    WCHAR szTempStr[256];
    WCHAR szTempStr2[256];

    { // Comment
        std::wstring commentText(m_pTexture->GetComment());
        if (!commentText.empty())
        {
            Gdiplus::SolidBrush blackBrush(Gdiplus::Color(255,30,30,30));
            Gdiplus::SolidBrush whiteBrush(Gdiplus::Color(255,245,245,245));

            Gdiplus::Font commentFont(L"Courier", textScaleRatioY * (float)height / 12.0f);

            Gdiplus::StringFormat commentFormat(0);
            commentFormat.SetAlignment(Gdiplus::StringAlignmentCenter);
            commentFormat.SetLineAlignment(Gdiplus::StringAlignmentNear);

            Gdiplus::RectF shadowRect(2.0f, 2.0f, (float)width, halfHt);
            pGraphics->DrawString(commentText.c_str(),(int)commentText.length(),&commentFont,
                shadowRect,&commentFormat,&blackBrush);

            Gdiplus::RectF textRect(0.0f, 0.0f, (float)width, halfHt);
            pGraphics->DrawString(commentText.c_str(),(int)commentText.length(),&commentFont,
                textRect,&commentFormat,&whiteBrush);
        } // End if
    } // End

    { // Format
        const WCHAR* szFormatName = NWGetTextureFormatNameW(m_pTexture->GetFormat());

        Gdiplus::Font *pFontToUse = &font;

        Gdiplus::Font smallerFormatFont(L"Courier", textFontHt*0.75f);

        if (wcslen(szFormatName)>14)
        {
            pFontToUse = &smallerFormatFont;
        } // End if

        Gdiplus::RectF textRect(0.0f,halfHt+topMargin,textWd-rightMargin,textHt);
        pGraphics->DrawString(szFormatName,(int)wcslen(szFormatName),pFontToUse,
                              textRect,&strFormat,&textBrush);
    } // End

    // Dimension
    switch(m_pTexture->GetTextureType())
    {
    default:
        _snwprintf_s(szTempStr,255,_TRUNCATE,L"%d x %d", m_pTexture->GetWidth(), m_pTexture->GetHeight());
        break;
    case NW_IMG_TYPE_1D_ARRAY:
    case NW_IMG_TYPE_2D_ARRAY:
    case NW_IMG_TYPE_CUBE_ARRAY:
        _snwprintf_s(szTempStr,255,_TRUNCATE,L"%d x %d x %d", m_pTexture->GetWidth(), m_pTexture->GetHeight(), m_pTexture->GetCount());
        break;
    }

    // Format
    {
        Gdiplus::RectF textRect(0.0f, halfHt + topMargin + textHt*1.0f, textWd - rightMargin, textHt);
        pGraphics->DrawString(szTempStr, (int)wcslen(szTempStr), &font,
            textRect, &strFormat, &textBrush);
    }


    // Mipmaps
    if (m_pTexture->HasMipmaps())
    {
        _snwprintf_s(szTempStr,255,_TRUNCATE,L"Mipmap : %d ", m_pTexture->GetNumMipmaps());

        Gdiplus::RectF textRect(0.0f,halfHt+topMargin+textHt*2.0f,textWd-rightMargin,textHt);
        if (m_pTexture->IsUsingAllMipmaps())
        {
            pGraphics->DrawString(szTempStr,(int)wcslen(szTempStr),&font,
                                    textRect,&strFormat,&textBrush);
        } // End if
        else
        {
            pGraphics->DrawString(szTempStr,(int)wcslen(szTempStr),&font,
                                    textRect,&strFormat,&mipmapWarningBrush);
        } // End if
    } // End if

    { // ByteSize
        NW4FormatNumberStringWithComma(szTempStr,256,m_pTexture->GetTextureDataSize());
        _snwprintf_s(szTempStr2,255,_TRUNCATE,L"%s B", szTempStr);

        Gdiplus::RectF textRect(0.0f,halfHt+topMargin+textHt*3.0f,textWd-rightMargin,textHt);
        pGraphics->DrawString(szTempStr2,(int)wcslen(szTempStr2),&font,
                              textRect,&strFormat,&textBrush);
    } // End

    // Component Selector
    if ( m_pTexture->IsForNW4F() && m_pTexture->HasCompSel() )
    {
        const NW_COMP_SEL &compSel = m_pTexture->GetCompSel();
        std::wstring compSelStr = NWCompSelToString(compSel);

        Gdiplus::RectF textRect(0.0f,halfHt+topMargin+textHt*4.0f,textWd-rightMargin,textHt);
        pGraphics->DrawString(compSelStr.c_str(),(int)compSelStr.length(),&font,
                                textRect,&strFormat,&textBrush);
    } // End if

    pGraphics->Flush(Gdiplus::FlushIntentionSync);
} // End of RenderCaptionWithInfo for CShellExtensionFileData

namespace
{
    void RenderBitmap(Gdiplus::Bitmap* target, Gdiplus::Bitmap* source)
    {
        Gdiplus::Graphics g(target);
        g.SetInterpolationMode(Gdiplus::InterpolationModeHighQuality);
        float ratio = (std::min)(target->GetWidth() / (float)source->GetWidth(), target->GetHeight() / (float)source->GetHeight());
        float width = source->GetWidth() * ratio;
        float height = source->GetHeight() * ratio;
        Gdiplus::RectF rect((target->GetWidth() - width) / 2, (target->GetHeight() - height), width, height);
        g.DrawImage(source, rect);
    }
}

//------------------------------------------------------------------------------
// Get or Create an icon for requested size
// If bWithInfo is set to true, icon will contain information text
// ( eg. texture size, format etc )
//------------------------------------------------------------------------------
HICON CShellExtensionFileData::GetIcon( int iconWidth,
                                    int iconHeight,
                                    bool bWithInfo )
{
    CNWSingleLock lock(&m_CS);

    if (m_pTexture==NULL)
        return NULL;


    Gdiplus::Bitmap* previewBitmap = m_pTexture->GetPreviewBitmap();

    Gdiplus::Bitmap destBitmap(iconWidth,iconHeight,
        (previewBitmap != NULL) ? previewBitmap->GetPixelFormat() : PixelFormat24bppRGB);
    if (m_pTexture->OnlyShowImage())
    {
        if (previewBitmap == NULL)
        {
            return false;
        }

        RenderBitmap(&destBitmap, previewBitmap);
    }
    else if (bWithInfo)
    {
        if (m_pTexture->GetPreviewBitmap() != NULL)
            RenderImageWithInfo(&destBitmap,iconWidth,iconHeight);
        else
            RenderCaptionWithInfo(&destBitmap,iconWidth,iconHeight);
    }
    else
    {
        if (m_pTexture->GetPreviewBitmap() != NULL)
            RenderImage(&destBitmap,iconWidth,iconHeight);
        else
            RenderCaption(&destBitmap,iconWidth,iconHeight);
    }

    HICON hIcon = NULL;
    destBitmap.GetHICON(&hIcon);

    if (hIcon==NULL)
        return NULL;

    return hIcon;
} // End of GetIcon for CShellExtensionFileData

//--------------------------------------------------------------------------
// Get or Create an small icon for requested size
//--------------------------------------------------------------------------
HICON CShellExtensionFileData::GetSmallIcon( int iconWidth, int iconHeight )
{
    CNWSingleLock lock(&m_CS);

    if (m_pTexture==NULL)
        return NULL;

    Gdiplus::Bitmap* previewSmallBitmap = m_pTexture->GetPreviewSmallBitmap();
    if(previewSmallBitmap == NULL)
        return GetIcon(iconWidth,iconHeight,false);

    if (m_pTexture->OnlyShowImage())
    {
        HICON hIcon = NULL;
        previewSmallBitmap->GetHICON(&hIcon);
        return hIcon;
    }

    Gdiplus::Bitmap destBitmap(iconWidth,iconHeight,
        (previewSmallBitmap != NULL) ? previewSmallBitmap->GetPixelFormat() : PixelFormat24bppRGB);

    if (previewSmallBitmap != NULL)
        RenderImage(&destBitmap,iconWidth,iconHeight,true);
    else
        RenderCaption(&destBitmap,iconWidth,iconHeight);

    HICON hIcon = NULL;
    destBitmap.GetHICON(&hIcon);

    if (hIcon==NULL)
        return NULL;

    return hIcon;
} // End of GetSmallIcon for CShellExtensionFileData


//------------------------------------------------------------------------------
// Get or Create an bitmap for requested size
// If bWithInfo is set to true, icon will contain information text
// ( eg. texture size, format etc )
//------------------------------------------------------------------------------
HBITMAP CShellExtensionFileData::GetBitmap( int bitmapWidth,
                                        int bitmapHeight,
                                        bool bWithInfo )
{
    CNWSingleLock lock(&m_CS);

    if (m_pTexture==NULL)
        return NULL;

    Gdiplus::Bitmap* previewBitmap = m_pTexture->GetPreviewBitmap();

    if (m_pTexture->OnlyShowImage())
    {
        HBITMAP hBitmap = NULL;
        Gdiplus::Color color((Gdiplus::ARGB)Gdiplus::Color::White);
        previewBitmap->GetHBITMAP(color, &hBitmap);
        return hBitmap;
    }

    Gdiplus::Bitmap destBitmap(bitmapWidth,bitmapHeight,
        (previewBitmap != NULL) ? previewBitmap->GetPixelFormat() : PixelFormat24bppRGB);
    if (bWithInfo)
    {
        if (m_pTexture->GetPreviewBitmap() != NULL)
            RenderImageWithInfo(&destBitmap,bitmapWidth,bitmapHeight);
        else
            RenderCaptionWithInfo(&destBitmap,bitmapWidth,bitmapHeight);
    }
    else
    {
        if (m_pTexture->GetPreviewBitmap() != NULL)
            RenderImage(&destBitmap,bitmapWidth,bitmapHeight);
        else
            RenderCaption(&destBitmap,bitmapWidth,bitmapHeight);
    }

    HBITMAP hBitmap = NULL;
    destBitmap.GetHBITMAP(Gdiplus::Color(0,0,0,0),&hBitmap);

    if (hBitmap==NULL)
        return NULL;

    return hBitmap;
} // End of GetBitmap for CShellExtensionFileData


