﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ShellExtension_PCH.h"

#include "ShellExtension_Resource.h"
#include "ShellExtension_Type.h"
#include "ShellExtension_SyncObject.h"
#include "ShellExtension_Texture.h"
#include "ShellExtension_FileData.h"
#include "ShellExtension_Manager.h"
#include "ShellExtension_Utility.h"

#include "Service/Icon/ShellExtension_Icon.h"

//==============================================================================
//
// CShellExtensionIcon Implementation
//
//==============================================================================
CShellExtensionIcon::CShellExtensionIcon() :
    m_pFileData(NULL)
{
    NWShellLockModule();
} // End of Constructor for CShellExtensionIcon


//------------------------------------------------------------------------------
CShellExtensionIcon::~CShellExtensionIcon()
{
    if (m_pFileData!=NULL)
    {
        m_pFileData->Release(L"ExtractIcon");
        m_pFileData = NULL;
    } // End if

    NWShellUnlockModule();
} // End of Destructor for CShellExtensionIcon


//------------------------------------------------------------------------------
// Load
//------------------------------------------------------------------------------
STDMETHODIMP CShellExtensionIcon::Load( LPCOLESTR wszFile, DWORD dwMode )
{
    if (m_pFileData!=NULL)
    {
        m_pFileData->Release(L"ExtractIcon::Load");
        m_pFileData = NULL;
    } // End if

    NW_USE_VAR(dwMode);

    USES_CONVERSION;
    const WCHAR* szFileName = W2CT(wszFile);
    m_pFileData = CShellExtensionManager::Instance()->GetFileData(szFileName,L"ExtractIcon");
    if (m_pFileData==NULL)
        return E_FAIL;

    return S_OK;
} // End of Load for CShellExtensionIcon


//------------------------------------------------------------------------------
// GetIconLocation
//------------------------------------------------------------------------------
STDMETHODIMP CShellExtensionIcon::GetIconLocation( UINT uFlags, LPTSTR szIconFile, UINT cchMax,
                                               int* piIndex, UINT* pwFlags )
{
    if (szIconFile!=NULL)
    {
        wcscpy_s(szIconFile,cchMax,CShellExtensionManager::Instance()->GetModuleName().c_str());
    } // End if

    if (piIndex!=NULL)
    {
        *piIndex = 0;
    } // End if

    if (pwFlags!=NULL)
    {
        *pwFlags = GIL_DONTCACHE | GIL_NOTFILENAME;
    } // End if

    if (uFlags&GIL_ASYNC)
        return E_PENDING;

    return S_OK;
} // End of GetIconLocation for CShellExtensionIcon


//------------------------------------------------------------------------------
// Extract
//------------------------------------------------------------------------------
STDMETHODIMP CShellExtensionIcon::Extract( LPCTSTR pszFile, UINT nIconIndex, HICON* phiconLarge,
                                       HICON* phiconSmall, UINT nIconSize )
{
    NW_USE_VAR(pszFile);
    NW_USE_VAR(nIconIndex);
    NW_USE_VAR(nIconSize);

    if (m_pFileData==NULL)
        return E_FAIL;

    CNWSingleLock lock(m_pFileData->GetCriticalSection());

    if ( (m_pFileData->GetTexture()==NULL) ||
         (m_pFileData->GetTexture()->GetPreviewBitmap()==NULL) )
    {
        if (m_pFileData->UpdateImage(true)==false)
            return S_FALSE;
    } // End if

    if (phiconLarge!=NULL)
    {
        int iconSize = (int)LOWORD(nIconSize);
        if (iconSize>=256)
        {
            *phiconLarge = m_pFileData->GetIcon(iconSize,iconSize,true);
        } // End if
        else if (iconSize <= 32)
        {
            *phiconLarge = m_pFileData->GetSmallIcon(iconSize,iconSize);
        } // End else if
        else
        {
            *phiconLarge = m_pFileData->GetIcon(iconSize,iconSize,false);
        } // End else
    } // End if

    if (phiconSmall!=NULL)
    {
        int iconSize = (int)HIWORD(nIconSize);
        if (iconSize>=256)
        {
            *phiconSmall = m_pFileData->GetIcon(iconSize,iconSize,true);
        } // End if
        else if (iconSize<=32)
        {
            *phiconSmall = m_pFileData->GetSmallIcon(iconSize,iconSize);
        } // End else if
        else
        {
            *phiconSmall = m_pFileData->GetIcon(iconSize,iconSize,false);
        } // End else
    } // End if

    return S_OK;
} // End of Extract for CShellExtensionIcon


//------------------------------------------------------------------------------
// Register
//------------------------------------------------------------------------------
bool CShellExtensionIcon::RegisterService( REGISTRATION_PARAM &param )
{
    WCHAR szGUIDStr[128];
    WCHAR szValStr[128];

    _snwprintf_s(szGUIDStr,127,_TRUNCATE,L"{%s}", NWCreateUUIDString(CLSID_ShellExtensionIcon).c_str());

    //--------------------------------------------------------------------------
    // CLSID
    //--------------------------------------------------------------------------
    CRegKey CLSIDKey;
    if (CLSIDKey.Open( HKEY_CLASSES_ROOT,
                       _T("CLSID"),
                       KEY_SET_VALUE )==ERROR_SUCCESS)
    {
        //----------------------------------------------------------------------
        // My Interface ID
        //----------------------------------------------------------------------
        _snwprintf_s(szValStr,127,_TRUNCATE,_T("NintendoWare %s Icon Shell Extension"), param.fileTypeName.c_str() );

        CRegKey GUIDKey;
        if (GUIDKey.Create(CLSIDKey.m_hKey,szGUIDStr)==ERROR_SUCCESS)
        {
            GUIDKey.SetStringValue(NULL,szValStr);

            //------------------------------------------------------------------
            // Inproc server
            //------------------------------------------------------------------
            CRegKey InpProc32Key;
            if (InpProc32Key.Create(GUIDKey.m_hKey,L"InprocServer32")==ERROR_SUCCESS)
            {
                InpProc32Key.SetStringValue(NULL,param.moduleName.c_str());
                InpProc32Key.SetStringValue(L"ThreadingModel", L"Apartment");
                InpProc32Key.Close();
            } // End if

            GUIDKey.Close();
        } // End if

        CLSIDKey.Close();
    } // End if

    //--------------------------------------------------------------------------
    // ProgID
    //--------------------------------------------------------------------------
    int i;
    for (i=0;i<(int)param.supportedExtensions.size();i++)
    {
        CRegKey progIDKey;
        if (progIDKey.Open( HKEY_CLASSES_ROOT,
                            param.progIDNames[i].c_str(),
                            KEY_SET_VALUE )==ERROR_SUCCESS)
        {
            //------------------------------------------------------------------
            // Default Icon
            //------------------------------------------------------------------
            CRegKey DefIconKey;
            if (DefIconKey.Create(progIDKey.m_hKey,L"DefaultIcon")==ERROR_SUCCESS)
            {
                DefIconKey.SetStringValue(NULL,L"%1");
                DefIconKey.Close();
            } // End if

            //------------------------------------------------------------------
            // ShellEx Icon
            //------------------------------------------------------------------
            CRegKey shellExKey;
            if (shellExKey.Open(progIDKey.m_hKey,
                                L"ShellEx",
                                KEY_SET_VALUE)==ERROR_SUCCESS)
            {
                //--------------------------------------------------------------
                // ShellEx Icon
                //--------------------------------------------------------------
                CRegKey handlerKey;
                if (handlerKey.Create(shellExKey.m_hKey,L"IconHandler")==ERROR_SUCCESS)
                {
                    handlerKey.SetStringValue(NULL,szGUIDStr);
                    handlerKey.Close();
                } // End if

                shellExKey.Close();
            } // End if

            progIDKey.Close();
        } // End if
    } // End for

    return true;
} // End of RegisterService for CShellExtensionIcon


//------------------------------------------------------------------------------
// Unregister
//------------------------------------------------------------------------------
bool CShellExtensionIcon::UnregisterService( REGISTRATION_PARAM &param )
{
    NW_USE_VAR(param.supportedExtensions.size());

    WCHAR szGUIDStr[128];
    _snwprintf_s(szGUIDStr,127,_TRUNCATE,L"{%s}", NWCreateUUIDString(CLSID_ShellExtensionIcon).c_str());

    //--------------------------------------------------------------------------
    // CLSID
    //--------------------------------------------------------------------------
    CRegKey CLSIDKey;
    if (CLSIDKey.Open ( HKEY_CLASSES_ROOT,
                        _T("CLSID"),
                        KEY_SET_VALUE )==ERROR_SUCCESS)
    {
        //----------------------------------------------------------------------
        // My Interface ID
        //----------------------------------------------------------------------
        CLSIDKey.RecurseDeleteKey(szGUIDStr);

        CLSIDKey.Close();
    } // End if

    return true;
} // End of UnregisterService for CShellExtensionIcon
