﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef __ShellExtension_TGALoader_H__
#define __ShellExtension_TGALoader_H__

//==============================================================================
//
// NW TGALoader
//
// - Derived from ImageLoader to load .tga files
//
//==============================================================================
class CNWTGALoader : public CNWImageLoader
{
public :
    CNWTGALoader();
    virtual ~CNWTGALoader();

    //--------------------------------------------------------------------------
    // Load from image file onto pTexture
    //
    // - If bLoadPreview is true, create preview image for icon
    //   If bLoadPreview is false, just load information ( format, size etc )
    //--------------------------------------------------------------------------
    virtual bool Load( CNWTexture *pTexture,
                       const WCHAR *szFilePath,
                       bool bLoadPreview );

private : // TGA file related types
    struct TGAHeader
    {
        unsigned char  idSize;          // size of ID field that follows 18 byte header (0 usually)
        unsigned char  colourMapType;   // type of colour map 0=none, 1=has palette
        unsigned char  imageType;       // type of image 0=none,1=indexed,2=rgb,3=grey,+8=rle packed

        short          colourMapStart;  // first colour map entry in palette
        short          colourMapLength; // number of colours in palette
        unsigned char  colourMapBits;   // number of bits per palette entry 15,16,24,32

        short          xStart;          // image x origin
        short          yStart;          // image y origin
        short          width;           // image width in pixels
        short          height;          // image height in pixels
        unsigned char  bits;            // image bits per pixel 8,16,24,32
        unsigned char  descriptor;      // image descriptor bits (vh flip bits)
    }; // End of TGAHeader

    struct NWTGADataBlockHeader
    {
        char dataBlockID[8];
        unsigned int dataBlockSize;
    }; // End of NWTGADataBlockHeader

    enum TGA_IMAGE_TYPE
    {
        TGA_IMAGE_TYPE_NONE                      =  0,
        TGA_IMAGE_TYPE_UNCOMPRESSED_COLORMAP     =  1,
        TGA_IMAGE_TYPE_UNCOMPRESSED_RGB          =  2,
        TGA_IMAGE_TYPE_UNCOMPRESSED_BW           =  3,
        TGA_IMAGE_TYPE_RLE_COLORMAP              =  9,
        TGA_IMAGE_TYPE_RLE_RGB                   = 10,
        TGA_IMAGE_TYPE_COMPRESSED_BW             = 11,
        TGA_IMAGE_TYPE_COMPRESSED_COLORMAP       = 32,
        TGA_IMAGE_TYPE_COMPRESSED_COLORMAP_4PASS = 33
    }; // End of TGA_IMAGE_TYPE

private : // Loading Code
    //--------------------------------------------------------------------------
    // Read TGA Header
    //
    // -
    //--------------------------------------------------------------------------
    bool ReadTGAHeader( CNWTexture *pTexture,
                        unsigned char *pTGABuffer, TGAHeader &fileHeader,
                        int fileSize, int &iCurBufferPos );
    bool ReadTGAIdField( CNWTexture *pTexture,
                         unsigned char *pTGABuffer, const TGAHeader &fileHeader,
                         int fileSize, int &iCurBufferPos,
                         int *pAdditionalInfoOfset,
                         NW_TGA_Type &TGACustomType );
    bool ReadNWTGADataBlockHeader( CNWTexture *pTexture,
                                   unsigned char *pTGABuffer, const TGAHeader &fileHeader,
                                   int fileSize, int &iCurBufferPos,
                                   NWTGADataBlockHeader &dataBlockHeader );
    bool ReadNW4CTGA( CNWTexture *pTexture,
                      unsigned char *pTGABuffer, const TGAHeader &fileHeader,
                      int fileSize, int &iCurBufferPos,
                      bool bLoadPreview );
    bool ReadNW4RTGA( CNWTexture *pTexture,
                      unsigned char *pTGABuffer, const TGAHeader &fileHeader,
                      int fileSize, int &iCurBufferPos,
                      bool bLoadPreview );
    bool ReadNormalTGA( CNWTexture *pTexture,
                        unsigned char *pTGABuffer, const TGAHeader &fileHeader,
                        int fileSize, int &iCurBufferPos,
                        bool bLoadPreview );
    bool ReadNormalTGA_UncompressedColormap( CNWTexture *pTexture,
                                             unsigned char *pTGABuffer, const TGAHeader &fileHeader,
                                             int fileSize, int &iCurBufferPos );
    bool ReadNormalTGA_UncompressedBW( CNWTexture *pTexture,
                                       unsigned char *pTGABuffer, const TGAHeader &fileHeader,
                                       int fileSize, int &iCurBufferPos );
    bool ReadNormalTGA_RLEBW( CNWTexture *pTexture,
                              unsigned char *pTGABuffer, const TGAHeader &fileHeader,
                              int fileSize, int &iCurBufferPos );
    bool ReadNormalTGA_UncompressedRGB( CNWTexture *pTexture,
                                        unsigned char *pTGABuffer, const TGAHeader &fileHeader,
                                        int fileSize, int &iCurBufferPos );
    bool ReadNormalTGA_RLERGB( CNWTexture *pTexture,
                               unsigned char *pTGABuffer, const TGAHeader &fileHeader,
                               int fileSize, int &iCurBufferPos);
}; // End of CNWTGALoader

#endif // End of __ShellExtension_TGALoader_H__

