﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef __ShellExtension_CTexLoader_H__
#define __ShellExtension_CTexLoader_H__

#include <memory>
//==============================================================================
//
// NW CTexLoader
//
// - Derived from ImageLoader to load .ctex files
//
//==============================================================================
class CNWCTexLoader : public CNWImageLoader
{
public :
    CNWCTexLoader();
    virtual ~CNWCTexLoader();

    //--------------------------------------------------------------------------
    // Load from image file onto pTexture
    //
    // - If bLoadPreview is true, create preview image for icon
    //   If bLoadPreview is false, just load information ( format, size etc )
    //--------------------------------------------------------------------------
    virtual bool Load( CNWTexture *pTexture,
                       const WCHAR *szFilePath,
                       bool bLoadPreview );

    //--------------------------------------------------------------------------
    // Convert ctex binary data to Preview-possible image ( linear format )
    // Made to be static so that it can be called from outside
    //--------------------------------------------------------------------------
    static bool ConvertToPreviewImage( CNWTexture *pTexture,
                                       const unsigned char* srcBuf,
                                       int srcBufSize,
                                       const unsigned short * palette,
                                       NW_CUBE_FACE cubeFace,
                                       Gdiplus::BitmapData &bmpData );

    //--------------------------------------------------------------------------
    // Convert 4x4 non-compressed block to linear PC displayable format
    //--------------------------------------------------------------------------
    static bool ConvertToPreviewImageBlock_4x4( CNWTexture *pTexture,
                                                const unsigned char* pSrcBuffer,
                                                unsigned char *pDestBuffer,
                                                const unsigned short * palette,
                                                int blockIndex,
                                                int numBlocksX,
                                                int numBlocksY,
                                                bool bFlipUV,
                                                Gdiplus::BitmapData &bmpData );

    //--------------------------------------------------------------------------
    // Convert 8x4 non-compressed block to linear PC displayable format
    //--------------------------------------------------------------------------
    static bool ConvertToPreviewImageBlock_8x4( CNWTexture *pTexture,
                                                const unsigned char* pSrcBuffer,
                                                unsigned char *pDestBuffer,
                                                const unsigned short * palette,
                                                int blockIndex,
                                                int numBlocksX,
                                                int numBlocksY,
                                                bool bFlipUV,
                                                Gdiplus::BitmapData &bmpData );

    //--------------------------------------------------------------------------
    // Convert 8x8 non-compressed block to linear PC displayable format
    //--------------------------------------------------------------------------
    static bool ConvertToPreviewImageBlock_8x8( CNWTexture *pTexture,
                                                const unsigned char* pSrcBuffer,
                                                unsigned char *pDestBuffer,
                                                const unsigned short * palette,
                                                int blockIndex,
                                                int numBlocksX,
                                                int numBlocksY,
                                                bool bFlipUV,
                                                Gdiplus::BitmapData &bmpData );

    //--------------------------------------------------------------------------
    // Convert 8x8 etc1 compressed block to linear PC displayable format
    // If bHasAlpha = true, format is etc1_a4
    //--------------------------------------------------------------------------
    static bool ConvertToPreviewImageBlock_4x4_Etc1( CNWTexture *pTexture,
                                                     const unsigned char* pSrcBuffer,
                                                     unsigned char *pDestBuffer,
                                                     int blockIndex,
                                                     int numBlocksX,
                                                     int numBlocksY,
                                                     bool bFlipUV,
                                                     bool bHasAlpha,
                                                     Gdiplus::BitmapData &bmpData );

private :
    //--------------------------------------------------------------------------
    // Load ImageTexture
    //--------------------------------------------------------------------------
    bool LoadImageTexture( CNWTexture *pTexture, MSXML2::IXMLDOMNodePtr textureNode );

    //--------------------------------------------------------------------------
    // Load CubeTexture
    //--------------------------------------------------------------------------
    bool LoadCubeTexture( CNWTexture *pTexture, MSXML2::IXMLDOMNodePtr textureNode );
    bool LoadCubeFace( CNWTexture *pTexture,
                       MSXML2::IXMLDOMNodePtr imagesNode,
                       const char* szImageCtrName,
                       NW_CUBE_FACE cubeFace,
                       Gdiplus::BitmapData &bmpData );

    //--------------------------------------------------------------------------
    // Decode text into images
    //--------------------------------------------------------------------------
    std::unique_ptr<unsigned char[]> DecodeBase64( const unsigned char* srcBuf, int srcSize,
                                 int& dstSize);

    //--------------------------------------------------------------------------
    // Decode one integer value
    //--------------------------------------------------------------------------
    int GetBase64Value( int c );

    //--------------------------------------------------------------------------
    // Get compress type text
    //--------------------------------------------------------------------------
    std::wstring ReadCompressTypeText( MSXML2::IXMLDOMNodePtr contentNode );

private : // Utility for converting blocked to linear offset
    struct PixelOffset
    {
        PixelOffset() {};

        PixelOffset( int _x, int _y ) :
            x(_x),
            y(_y)
        {
        }; // End of Constructor

        int x;
        int y;
    }; // End of PixelOffset

    static PixelOffset s_blockToLinearPixelOffset[8*8];       // Table to convert blocked image to linear offset
    static PixelOffset s_blockToLinearPixelOffsetFlipUV[8*8]; // Table to convert blocked image to linear offset for flipped case

public :
    static void InitBlockToLinearTable(); // Called from Manager
}; // End of CNWCTexLoader

#endif // End of __ShellExtension_CTexLoader_H__

