﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Runtime.InteropServices;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;

namespace NintendoWare.ExtensionManager
{
    //==========================================================================
    /// <summary>
    /// Lock App Dialog
    /// </summary>
    //==========================================================================
    public partial class LockAppsDialog : Form
    {
        //----------------------------------------------------------------------
        /// <summary>
        /// Constructor
        /// </summary>
        //----------------------------------------------------------------------
        public LockAppsDialog(IEnumerable<LockAppPair> lockApps)
        {
            InitializeComponent();

            UpdateList(lockApps);
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Button click
        /// </summary>
        //----------------------------------------------------------------------
        private void btnOk_Click(object sender, EventArgs e)
        {
            Close();
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Update List
        /// </summary>
        //----------------------------------------------------------------------
        private void UpdateList(IEnumerable<LockAppPair> lockApps)
        {
            MakeWindowCaptions();

            Debug.Assert(windowCaptions_ != null);

            lvwApp.BeginUpdate();
            {
                lvwApp.Items.Clear();

                foreach (var lockApp in lockApps)
                {

                    if ( (lockApp.LockingPath.EndsWith("explorer.exe") == true) ||
                         (lockApp.LockingPath.EndsWith("prevhost.exe") == true) )
                    {
                        continue;
                    }

                    string windowCaption = string.Empty;
                    windowCaptions_.TryGetValue(lockApp.LockingPath, out windowCaption);

                    lvwApp.Items.Add(
                        new ListViewItem(
                            new []
                            {
                                windowCaption,
                                Path.GetFileName(lockApp.LockingPath),
                                Path.GetFileName(lockApp.LockedPath)
                            }
                        )
                    );
                }
            }
            lvwApp.EndUpdate();
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Refresh button click
        /// </summary>
        //----------------------------------------------------------------------
        private void btnRefresh_Click(object sender, EventArgs e)
        {
            Cursor = Cursors.WaitCursor;

            UpdateList(MainApp.Instance.RegistrationManager.LockApps);

            Cursor = Cursors.Default;
        }

        [DllImport("user32")]
        private static extern bool EnumWindows(WNDENUMPROC lpEnumFunc, IntPtr lParam);

        private delegate bool WNDENUMPROC(IntPtr hWnd, IntPtr lParam);

        [DllImport("user32")]
        private static extern bool IsWindowVisible(IntPtr hWnd);

        [DllImport("user32", CharSet = CharSet.Auto)]
        private static extern int GetWindowText(IntPtr hWnd, StringBuilder lpString, int nMaxCount);

        [DllImport("user32")]
        private static extern int GetWindowThreadProcessId(IntPtr hWnd, out int lpdwProcessId);

        private static Dictionary<string, string> windowCaptions_;

        //----------------------------------------------------------------------
        /// <summary>
        /// Make Window Captions
        /// </summary>
        //----------------------------------------------------------------------
        private void MakeWindowCaptions()
        {
            windowCaptions_ = new Dictionary<string, string>();

            EnumWindows(EnumerateWindow, IntPtr.Zero);
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Enumerate Window
        /// </summary>
        //----------------------------------------------------------------------
        private bool EnumerateWindow(IntPtr hWnd, IntPtr lParam)
        {
            if (IsWindowVisible(hWnd))
            {
                // ウィンドウのキャプションを取得・表示
                var caption = new StringBuilder(1024);
                GetWindowText(hWnd, caption, caption.Capacity);

                int processId;
                GetWindowThreadProcessId(hWnd, out processId);

                var process = Process.GetProcessById(processId);

                windowCaptions_[process.MainModule.FileName] = caption.ToString();
            }

            return true;
        }
    }
}
