﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Diagnostics;
using System.Runtime.InteropServices;

namespace NintendoWare.ExtensionManager
{
    public class LockAppPair
    {
        public string LockedPath { get; set; }
        public string LockingPath { get; set; }
    }

    public class ProcessUtility
    {
        private List<Tuple<string, string>> dosDriveNames = new List<Tuple<string, string>>();

        public ProcessUtility()
        {
            for (char c = 'A'; c <='Z'; c++)
            {
                var builder = new StringBuilder(260 + 1);
                var length = QueryDosDevice(c + ":", builder, builder.Capacity);
                if (length != 0)
                {
                    dosDriveNames.Add(new Tuple<string, string>(builder.ToString(), c + ":"));
                }
            }
        }

        private string NormalizeFilepathString(string path)
        {
            foreach (var item in dosDriveNames)
            {
                path = path.Replace(item.Item1, item.Item2);
            }

            return path;
        }

        public IEnumerable<LockAppPair> GetBlockingModulePaths(Dictionary<string, string> moduleNames)
        {
            var files = new List<string>();
            var currentProcess = Process.GetCurrentProcess();
            var processes = Process.GetProcesses();

            HashSet<string> blockingModules = new HashSet<string>();


            foreach (var process in processes)
            {
                if (process.Id == currentProcess.Id)
                {
                    continue;
                }

                // Process.GetModules と似ているが、こちらは32 ビットプロセスも取得できる。

                IntPtr hProcess = OpenProcess(ProcessAccessFlags.QueryInformation | ProcessAccessFlags.VirtualMemoryRead, false, process.Id);
                if (hProcess != IntPtr.Zero)
                {
                    var exePath = new StringBuilder(260 + 1);
                    GetProcessImageFileName(hProcess, exePath, exePath.Capacity);
                    var fileName = exePath.ToString();

                    if (!blockingModules.Contains(fileName))
                    {
                        uint dwNeeded = 1;
                        IntPtr[] Modules = new IntPtr[dwNeeded];
                        while (EnumProcessModulesEx(hProcess, Modules, (uint)(IntPtr.Size * Modules.Length), out dwNeeded, ListModulesAll))
                        {
                            if (dwNeeded <= Modules.Length)
                            {
                                foreach (var module in Modules.Take((int)dwNeeded).Distinct())
                                {
                                    var szFile = new StringBuilder(260 + 1);

                                    GetModuleFileNameEx(hProcess, module, szFile, szFile.Capacity);

                                    string path;
                                    if (moduleNames.TryGetValue(szFile.ToString().ToLowerInvariant(), out path))
                                    {
                                        blockingModules.Add(fileName);
                                        yield return new LockAppPair()
                                        {
                                            LockingPath = NormalizeFilepathString(fileName),
                                            LockedPath = path,
                                        };
                                    }
                                }
                                break;
                            }
                            else
                            {
                                // だいぶ大きめにしておく
                                dwNeeded *= 2;
                                Modules = new IntPtr[dwNeeded];
                            }
                        }
                    }

                    CloseHandle(hProcess);
                }
            }
        }

        [Flags]
        private enum ProcessAccessFlags : uint
        {
            All = 0x001F0FFF,
            Terminate = 0x00000001,
            CreateThread = 0x00000002,
            VirtualMemoryOperation = 0x00000008,
            VirtualMemoryRead = 0x00000010,
            VirtualMemoryWrite = 0x00000020,
            DuplicateHandle = 0x00000040,
            CreateProcess = 0x000000080,
            SetQuota = 0x00000100,
            SetInformation = 0x00000200,
            QueryInformation = 0x00000400,
            QueryLimitedInformation = 0x00001000,
            Synchronize = 0x00100000
        }

        [DllImport("kernel32.dll", SetLastError = true)]
        private static extern IntPtr OpenProcess(ProcessAccessFlags processAccess, bool bInheritHandle, int processId);

        private const uint ListModules32bit = 1;
        private const uint ListModules64bit = 2;
        private const uint ListModulesAll = ListModules32bit | ListModules64bit;

        [DllImport("psapi.dll", CallingConvention = CallingConvention.StdCall)]
        private static extern bool EnumProcessModulesEx(IntPtr hProcess, [MarshalAs(UnmanagedType.LPArray)] [In][Out] IntPtr[] lphModule, uint cb, out uint lpcbNeeded, uint dwFilterFlag);

        [DllImport("kernel32.dll", SetLastError = true)]
        [return: MarshalAs(UnmanagedType.Bool)]
        static extern bool CloseHandle(IntPtr hObject);

        [DllImport("psapi.dll")]
        static extern uint GetModuleFileNameEx(IntPtr hProcess, IntPtr hModule, [Out] StringBuilder lpBaseName, [In] [MarshalAs(UnmanagedType.U4)] int nSize);

        [DllImport("kernel32.dll")]
        static extern uint QueryDosDevice(string lpDeviceName, StringBuilder lpTargetPath, int ucchMax);

        [DllImport("psapi.dll")]
        static extern uint GetProcessImageFileName(IntPtr hProcess, [Out] StringBuilder lpImageFileName, [In] [MarshalAs(UnmanagedType.U4)] int nSize);
    }
}
