﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Threading;

namespace NW4F.LayoutBinaryConverter
{
    public class Program
    {
        /// <summary>
        /// cygwin の仕様変更によって、1.7.10 以降の bash からツールが実行された場合、tmp と TMP など大文字・小文字違いの同名
        /// 環境変数が定義され、XmlSerializer内部で例外が発生してしまう問題があります。
        /// 対策として、小文字の環境変数を削除します。
        /// </summary>
        static void RemoveMultipleEnvironmentVariableWorkaround_(string envName)
        {
            // 環境変数を取得する。
            string envVal = Environment.GetEnvironmentVariable(envName);
            if (string.IsNullOrEmpty(envVal))
            {
                return;
            }

            // 手始めに一つ削除して、その後再度取得してみる。
            Environment.SetEnvironmentVariable(envName, null);
            string envVal2 = Environment.GetEnvironmentVariable(envName);
            if (!string.IsNullOrEmpty(envVal2))
            {
                // 2つ目が取得できた場合は、多重定義されている状況。
                // POSIX スタイルの環境変数を削除するようにする。
                if (envVal2.StartsWith("/"))
                {
                    Environment.SetEnvironmentVariable(envName, envVal);
                }
                return;
            }
            else
            {
                // 多重定義がなかったので、削除した環境変数を元に元しておく。
                Environment.SetEnvironmentVariable(envName, envVal);
            }
        }

        /// <summary>
        /// コンバート
        /// </summary>
        /// <param name="args">引数配列</param>
        public static BinaryConverterErrorType Convert(string[] args)
        {
            try
            {
                var converter = new BinaryConverter();
                return converter.Convert(args);
            }
#if ! DEBUG
            catch (Exception ex)   // 補足されなかった.NET例外
            {
                Report.Err.WriteLine(ex.ToString());
                return BinaryConverterErrorType.InternalError; // 例外を補足出来なかった
            }
#endif
            finally
            {
                Report.Stop();
            }
        }

        /// <summary>
        /// 引数ファイル内の文字列チェック
        /// "#"はコメントアウトとして無視します。
        /// </summary>
        /// <param name="str">チェックする文字列</param>
        static bool CheckArgFileStr(string str)
        {
            const string commentOutChar = "#";

            // 先頭と末尾の空白を削除しておく
            var ckStr = str.Trim();

            if (ckStr == "")
            {
                return false;
            }

            return ckStr[0] != commentOutChar[0];
        }

        /// <summary>
        /// メイン関数
        /// </summary>
        static int Main(string[] args)
        {
            // カルチャの設定
            SetCulture();

            //--------------------------------------------
            // cygwin 挙動変更による環境変数多重定義の回避コード
            RemoveMultipleEnvironmentVariableWorkaround_("tmp");
            RemoveMultipleEnvironmentVariableWorkaround_("temp");

            return (int)Convert(args);
        }

        /// <summary>
        /// カルチャの設定
        /// </summary>
        private static void SetCulture()
        {
            string entoryFolder = Path.GetDirectoryName(Assembly.GetEntryAssembly().Location);
            if (Directory.Exists(Path.Combine(entoryFolder, "en")))
            {
                CultureInfo cultureInfo = CultureInfo.CreateSpecificCulture("en-us");
                Thread.CurrentThread.CurrentCulture = cultureInfo;
                Thread.CurrentThread.CurrentUICulture = cultureInfo;
            }
        }
    }
}
