﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>

// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Linq;
using Nintendo.Foundation.IO;

namespace NW4F.LayoutBinaryConverter
{
    partial class BinaryConverter
    {
        /// <summary>
        /// メッシュ形状トリミング最適化情報の出力処理
        /// </summary>
        /// <param name="infoList"></param>
        /// <param name="outPath"></param>
        /// <param name="hideIdentifyInformation"></param>
        private void OutputFillrateOptimizeInformation(List<LytInfo.FillrateOptimizeInformation> infoList, string outPath, bool hideIdentifyInformation)
        {
            StreamWriter writer;
            try
            {
                // カレントディレクトリにログを出力する
                writer = new StreamWriter(outPath + "\\OptimizationReport.html");
            }
            catch (ArgumentException)
            {
                Report.Err.WriteLine("can't create log file.");
                return;
            }

            int totalPicturePane = 0;
            int totalOptimizedPane = 0;
            foreach (var info in infoList)
            {
                totalPicturePane += info.PicturePaneCount;
                totalOptimizedPane += info.FillRateOptimizedPicturePaneCount;
            }

            writer.WriteLine("<!DOCTYPE html>");
            writer.WriteLine("<html>");

            writer.WriteLine("<head>");
            writer.WriteLine("<meta charset=\"utf-8\"/>");
            writer.WriteLine("<title>メッシュ形状トリミング最適化情報</title>");
            writer.WriteLine("</head>");

            writer.WriteLine("<body>");
            writer.WriteLine("<hr>");
            if (!hideIdentifyInformation)
            {
                writer.WriteLine(outPath + " の全体情報<br>");
            }
            else
            {
                writer.WriteLine("???? の全体情報<br>");
            }
            writer.WriteLine("<b>レイアウトファイル数：" + infoList.Count + "</b><br>");
            writer.WriteLine("<br>");
            writer.WriteLine("<b>出力日時：" + DateTime.Now.ToString() + "</b><br>");
            writer.WriteLine("<br>");
            writer.WriteLine("<b>レイアウトファイル数：" + infoList.Count + "</b><br>");
            writer.WriteLine("<b>メッシュ形状トリミング最適化適用率：" + (((float)totalOptimizedPane / (float)totalPicturePane) * 100.0f) + "%</b><br>");

            int totalPixels = 0;
            int totalOptimizedPixels = 0;
            foreach (var info in infoList)
            {
                foreach (var pict in info.PictInfo)
                {
                    if (pict.Value.OptimizedTexWidth < pict.Value.TexWidth ||
                        pict.Value.OptimizedTexHeight < pict.Value.TexHeight)
                    {
                        totalPixels += pict.Value.TexWidth * pict.Value.TexHeight;
                        totalOptimizedPixels += pict.Value.OptimizedTexWidth * pict.Value.OptimizedTexHeight;
                    }
                }
            }

            if (totalPixels == 0)
            {
                writer.WriteLine("<b>メッシュ形状トリミング最適化対象仮想ピクセル数：0 pixels</b><br>");
                writer.WriteLine("<b>メッシュ形状トリミング最適化後仮想ピクセル数：0 pixels</b><br>");
                writer.WriteLine("<b>メッシュ形状トリミング最適化後面積比：-</b><br>");
            }
            else
            {
                writer.WriteLine("<b>メッシュ形状トリミング最適化対象仮想ピクセル数：" + totalPixels + " pixels</b><br>");
                writer.WriteLine("<b>メッシュ形状トリミング最適化後仮想ピクセル数：" + totalOptimizedPixels + " pixels</b><br>");
                writer.WriteLine("<b>メッシュ形状トリミング最適化後面積比：" + ((float)totalOptimizedPixels / (float)totalPixels * 100.0f) + "%</b><br>");
            }

            writer.WriteLine("<hr>");

            var outputPathUri = new Uri(outPath + "\\");

            foreach (var info in infoList)
            {
                if (!hideIdentifyInformation)
                {
                    writer.WriteLine("<a href = \"file:///" + info.FileName + "\">" + Path.GetFileName(info.FileName) + "</a><br>");
                }
                writer.WriteLine("<b>ペイン数情報</b><br>");
                writer.Write("<table border=1>");
                writer.Write("<tr><td>ピクチャ</td><td>テキスト</td><td>ウインドウ</td><td>パーツ</td><td>バウンディング</td><td>ヌル</td><td>キャプチャ</td><td>アラインメント</td><td>シザー</td><td>総数</td></tr>");
                writer.Write("<tr>"
                                        + "<td>" + info.PicturePaneCount + "</td>"
                                        + "<td>" + info.TextBoxPaneCount + "</td>"
                                        + "<td>" + info.WindowPaneCount + "</td>"
                                        + "<td>" + info.PartsPaneCount + "</td>"
                                        + "<td>" + info.BoundingPaneCount + "</td>"
                                        + "<td>" + info.NullPaneCount + "</td>"
                                        + "<td>" + info.CapturePaneCount + "</td>"
                                        + "<td>" + info.AlignmentPaneCount + "</td>"
                                        + "<td>" + info.ScissorPaneCount + "</td>"
                                        + "<td>" + info.TotalPaneCount + "</td>"
                                        + "</tr>");
                writer.Write("</table>");
                if (info.PicturePaneCount > 0)
                {
                    writer.WriteLine("<br>");
                    writer.WriteLine("<b>メッシュ形状トリミング最適化適用率：" + info.FillRateOptimizedPicturePaneCount + "/" + info.PicturePaneCount
                                        + "(" + (((float)info.FillRateOptimizedPicturePaneCount / (float)info.PicturePaneCount) * 100.0f) + "%)</b><br>");
                    writer.WriteLine("<br>");
                }

                if (info.PictInfo.Count > 0)
                {
                    int originalLayoutPixelCount = 0;
                    int optimizedLayoutPixelCount = 0;
                    foreach (var pict in info.PictInfo)
                    {
                        if (pict.Value.OptimizedTexWidth < pict.Value.TexWidth ||
                            pict.Value.OptimizedTexHeight < pict.Value.TexHeight)
                        {
                            originalLayoutPixelCount += pict.Value.TexWidth * pict.Value.TexHeight;
                            optimizedLayoutPixelCount += pict.Value.OptimizedTexWidth * pict.Value.OptimizedTexHeight;
                        }
                    }

                    // メモリ使用量増加量
                    writer.WriteLine("<br>");
                    if (info.IncreasedMemorySize < 1024)
                    {
                        writer.WriteLine("<b>レイアウトバイナリファイルサイズ増加量：" + info.IncreasedMemorySize + " Bytes</b>");
                    }
                    else
                    {
                        writer.WriteLine("<b>レイアウトバイナリファイルサイズ増加量：" + (info.IncreasedMemorySize / 1024) + " KB</b>");
                    }
                    writer.WriteLine("<br>");

                    if (originalLayoutPixelCount == 0)
                    {
                        writer.WriteLine("<b>メッシュ形状トリミング最適化対象仮想ピクセル数：0 pixels</b><br>");
                        writer.WriteLine("<b>メッシュ形状トリミング最適化後仮想ピクセル数：0 pixels</b><br>");
                        writer.WriteLine("<b>メッシュ形状トリミング最適化効果：-</b><br>");
                    }
                    else
                    {
                        writer.WriteLine("<b>メッシュ形状トリミング最適化対象仮想ピクセル数：" + originalLayoutPixelCount + " pixels</b><br>");
                        writer.WriteLine("<b>メッシュ形状トリミング最適化後仮想ピクセル数：" + optimizedLayoutPixelCount + " pixels</b><br>");
                        writer.WriteLine("<b>メッシュ形状トリミング最適化後面積比：" + ((float)optimizedLayoutPixelCount / (float)originalLayoutPixelCount * 100.0f) + "%</b><br>");
                    }
                    writer.WriteLine("<br>");

                    writer.WriteLine("<b>ペインごとの最適化情報</b>");
                    writer.WriteLine("<table border=1>");
                    writer.WriteLine("<tr><td>ピクチャペイン名</td><td>仮想ピクセル数</td><td>最適化結果</td><td>最適化阻害要因</td></tr>");
                    foreach (var pict in info.PictInfo)
                    {
                        int originalPanePixelCount = pict.Value.TexWidth * pict.Value.TexHeight;
                        int optimizedPanePixelCount = pict.Value.OptimizedTexWidth * pict.Value.OptimizedTexHeight;
                        writer.Write("<tr>");

                        if (!hideIdentifyInformation)
                        {
                            if (pict.Value.ImagePath != "")
                            {
                                var imagePath = new Uri(pict.Value.ImagePath);
                                writer.Write("<td>" + pict.Value.Name + "<br>(" + outputPathUri.MakeRelativeUri(imagePath).ToString() + ")</td>");
                            }
                            else
                            {
                                writer.Write("<td>" + pict.Value.Name + "</td>");

                            }
                        }
                        else
                        {
                            writer.Write("<td> --- </td>");
                        }

                        writer.Write("<td>" + originalPanePixelCount + " pixels(" + pict.Value.TexWidth + "x" + pict.Value.TexHeight + ")</td>");
                        if (pict.Value.OptimizedTexWidth < pict.Value.TexWidth ||
                            pict.Value.OptimizedTexHeight < pict.Value.TexHeight)
                        {
                            writer.Write("<td><b>" + (((float)optimizedPanePixelCount / (float)originalPanePixelCount) * 100.0f) + "%");
                            writer.Write("(" + pict.Value.OptimizationType + ")");
                            writer.Write("</b><br>");
                            writer.Write(optimizedPanePixelCount + " pixels(" + pict.Value.OptimizedTexWidth + "x" + pict.Value.OptimizedTexHeight + ")</td>");
                        }
                        else
                        {
                            writer.Write("<td>-</td>");
                        }
                        foreach (var reason in pict.Value.Reasons)
                        {
                            writer.Write("<td>" + reason + "</td>");
                        }
                        writer.Write("</tr>");
                        writer.WriteLine("");
                    }
                    writer.WriteLine("</table>");
                }
                writer.WriteLine("<hr>");
            }
            writer.WriteLine("</body>");
            writer.WriteLine("</html>");

            writer.Close();
        }
    }
}
