﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "stdafx.h"

#include <algorithm>
#include <cstring>
#include <nn/ui2d/ui2d_Resources.h>
#include "WriteCommon.h"
#include "BinaryLytWriter.h"

// seadでのnullptrのdefineで方のエラーが起こるのでundef
#undef nullptr

using namespace System;
using namespace System::Diagnostics;
using namespace System::IO;
using namespace System::Collections::Generic;
using namespace System::Text;

namespace tg = nn::ui2d;

namespace NW4F
{
namespace LayoutBinaryConverter
{

namespace sch = Schema::Flyt;

namespace
{

void
CopyVec2(tg::ResVec2* dst, sch::Vec2^ src)
{
    if(src != nullptr)
    {
        dst->x = src->x;
        dst->y = src->y;
    }
}

void
CopyTexVec2(tg::ResVec2* dst, sch::TexVec2^ src)
{
    dst->x = src->s;
    dst->y = src->t;
}

void
CopySize(nn::ui2d::Size* dst, sch::Vec2^ src)
{
    dst->width  = src->x;
    dst->height = src->y;
}

void
CopyVec3(tg::ResVec3* dst, sch::Vec3^ src)
{
    dst->x = src->x;
    dst->y = src->y;
    dst->z = src->z;
}

uint8_t
GetBasePositionValue(sch::Position^ position)
{
    uint8_t hPos = 0, vPos = 0;

    if (position == nullptr)
    {
        return 0;
    }

    switch (position->x)
    {
    case sch::HorizontalPosition::Left:
        hPos = tg::HorizontalPosition_Left;
        break;
    case sch::HorizontalPosition::Center:
        hPos = tg::HorizontalPosition_Center;
        break;
    case sch::HorizontalPosition::Right:
        hPos = tg::HorizontalPosition_Right;
        break;
    }

    switch (position->y)
    {
    case sch::VerticalPosition::Top:
        vPos = tg::VerticalPosition_Top;
        break;
    case sch::VerticalPosition::Center:
        vPos = tg::VerticalPosition_Center;
        break;
    case sch::VerticalPosition::Bottom:
        vPos = tg::VerticalPosition_Bottom;
        break;
    }

    return (vPos << 2) + hPos;
}

uint8_t
GetPerCharacterTransformLoopType(sch::PerCharacterTransformLoopType loopType)
{
    switch (loopType)
    {
    case sch::PerCharacterTransformLoopType::OneTime:
        return tg::PerCharacterTransformLoopType_OneTime;
    case sch::PerCharacterTransformLoopType::Loop:
        return tg::PerCharacterTransformLoopType_Loop;
    }
    return tg::PerCharacterTransformLoopType_Loop;
}

uint8_t
GetPerCharacterTransformOriginV(sch::VerticalPosition originV)
{
    switch(originV)
    {
    case sch::VerticalPosition::Center:
        return tg::PerCharacterTransformOriginV_Center;
    case sch::VerticalPosition::Bottom:
        return tg::PerCharacterTransformOriginV_Bottom;
    }
    return tg::PerCharacterTransformOriginV_Center;
}

uint8_t
GetTextAlignment(sch::TextAlignment textAlignment)
{
    switch (textAlignment)
    {
    case sch::TextAlignment::Synchronous:
        return tg::TextAlignment_Synchronous;
    case sch::TextAlignment::Left:
        return tg::TextAlignment_Left;
    case sch::TextAlignment::Center:
        return tg::TextAlignment_Center;
    case sch::TextAlignment::Right:
        return tg::TextAlignment_Right;
    }

    return tg::TextAlignment_Synchronous;
}

tg::TexWrap
GetTexWrapMode(sch::TexWrapMode texWrapMode)
{
    switch (texWrapMode)
    {
    case sch::TexWrapMode::Clamp:
        return tg::TexWrap_Clamp;
    case sch::TexWrapMode::Repeat:
        return tg::TexWrap_Repeat;
    case sch::TexWrapMode::Mirror:
        return tg::TexWrap_Mirror;
    }

    return tg::TexWrap_Repeat;
}

uint8_t
GetTextureFlip(sch::TextureFlip textureFlip)
{
    using namespace nn::ui2d;

    switch (textureFlip)
    {
    case sch::TextureFlip::FlipH:            return tg::TextureFlip_FlipU;
    case sch::TextureFlip::FlipV:            return tg::TextureFlip_FlipV;
    case sch::TextureFlip::Rotate90:         return tg::TextureFlip_Rotate90;
    case sch::TextureFlip::Rotate180:        return tg::TextureFlip_Rotate180;
    case sch::TextureFlip::Rotate270:        return tg::TextureFlip_Rotate270;
    }

    return tg::TextureFlip_None;
}

uint8_t
GetOriginType(sch::ScreenOriginType originType)
{
    using namespace nn::ui2d;

    switch (originType)
    {
    case sch::ScreenOriginType::Classic:     return tg::ScreenOriginType_Classic;
    case sch::ScreenOriginType::Normal:      return tg::ScreenOriginType_Normal;
    }

    return tg::ScreenOriginType_Classic;
}

tg::TexGenSrc
GetTexGenSrc(sch::TexGenSrc texGenSrc)
{
    using namespace nn::ui2d;

    switch (texGenSrc)
    {
    case sch::TexGenSrc::Tex0:                  return tg::TexGenSrc_Tex0;
    case sch::TexGenSrc::Tex1:                  return tg::TexGenSrc_Tex1;
    case sch::TexGenSrc::Tex2:                  return tg::TexGenSrc_Tex2;
    case sch::TexGenSrc::OrthogonalProjection:  return tg::TexGenSrc_OrthoProjection;
    case sch::TexGenSrc::PaneBasedProjection:   return tg::TexGenSrc_PaneBaseOrthoProjection;
    case sch::TexGenSrc::PerspectiveProjection:    return tg::TexGenSrc_PerspectiveProjection;
    case sch::TexGenSrc::PaneBasedPerspectiveProjection:   return tg::TexGenSrc_PaneBasePerspectiveProjection;
    }

    return tg::TexGenSrc();
}

// 変換テーブル
// ★ 不要になるように、修正を行います。（中間ファイルとランタイムを同期編集をするルールとすれば、ただのキャストで十分そう）
tg::TevMode
GetTevMode(sch::TevMode value)
{
    switch (value)
    {
    case sch::TevMode::Replace:
        return tg::TevMode_Replace;
    case sch::TevMode::Modulate:
        return tg::TevMode_Modulate;
    case sch::TevMode::Add:
        return tg::TevMode_Add;
    case sch::TevMode::AddSigned:
        return tg::TevMode_AddSigned;
    case sch::TevMode::Interpolate:
        return tg::TevMode_Interpolate;
    case sch::TevMode::Subtract:
        return tg::TevMode_Subtract;
    case sch::TevMode::AddMult:
        return tg::TevMode_AddMultiply;
    case sch::TevMode::MultAdd:
        return tg::TevMode_MultiplyAdd;
    case sch::TevMode::Overlay:
        return tg::TevMode_Overlay;
    case sch::TevMode::Lighten:
        return tg::TevMode_Lighten;
    case sch::TevMode::Darken:
        return tg::TevMode_Darken;
    case sch::TevMode::Indirect:
        return tg::TevMode_Indirect;
    case sch::TevMode::BlendIndirect:
        return tg::TevMode_BlendIndirect;
    case sch::TevMode::EachIndirect:
        return tg::TevMode_EachIndirect;
    default:
        Debug::Assert(false);
        return tg::TevMode_Replace;
    }
}

tg::ResDetailedCombinerStage::Combine
GetRawTevCombine(sch::TevMode value)
{
    switch (value)
    {
    case sch::TevMode::Replace:
        return tg::ResDetailedCombinerStage::Combine_Replace;
    case sch::TevMode::Modulate:
        return tg::ResDetailedCombinerStage::Combine_Modulate;
    case sch::TevMode::Add:
        return tg::ResDetailedCombinerStage::Combine_Add;
    case sch::TevMode::AddSigned:
        return tg::ResDetailedCombinerStage::Combine_AddSigned;
    case sch::TevMode::Interpolate:
        return tg::ResDetailedCombinerStage::Combine_Interpolate;
    case sch::TevMode::Subtract:
        return tg::ResDetailedCombinerStage::Combine_Subtract;
    case sch::TevMode::AddMult:
        return tg::ResDetailedCombinerStage::Combine_AddMult;
    case sch::TevMode::MultAdd:
        return tg::ResDetailedCombinerStage::Combine_MultAdd;
    default:
        return tg::ResDetailedCombinerStage::Combine_Replace;
    }
}
tg::ResDetailedCombinerStage::Src
GetRawTevSrc(sch::TevSource value)
{
    switch (value)
    {
    case sch::TevSource::Texture0:
        return tg::ResDetailedCombinerStage::Src_Texture0;
    case sch::TevSource::Texture1:
        return tg::ResDetailedCombinerStage::Src_Texture1;
    case sch::TevSource::Texture2:
        return tg::ResDetailedCombinerStage::Src_Texture2;
    case sch::TevSource::Texture3:
        return tg::ResDetailedCombinerStage::Src_Texture3;
    case sch::TevSource::Constant:
        return tg::ResDetailedCombinerStage::Src_Constant;
    case sch::TevSource::Primary:
        return tg::ResDetailedCombinerStage::Src_Primary;
    case sch::TevSource::Previous:
        return tg::ResDetailedCombinerStage::Src_Previous;
    case sch::TevSource::Register:
        return tg::ResDetailedCombinerStage::Src_PreviousBuffer;
    default:
        return tg::ResDetailedCombinerStage::Src_Constant;
    }
}

tg::ResDetailedCombinerStage::OperandRgb
GetRawTevOpRgb(sch::TevOpRgb value)
{
    switch (value)
    {
    case sch::TevOpRgb::Rgb:
        return tg::ResDetailedCombinerStage::OperandRgb_Rgb;
    case sch::TevOpRgb::InvRgb:
        return tg::ResDetailedCombinerStage::OperandRgb_InverseRgb;
    case sch::TevOpRgb::Alpha:
        return tg::ResDetailedCombinerStage::OperandRgb_Alpha;
    case sch::TevOpRgb::InvAlpha:
        return tg::ResDetailedCombinerStage::OperandRgb_InverseAlpha;
    case sch::TevOpRgb::Rrr:
        return tg::ResDetailedCombinerStage::OperandRgb_Rrr;
    case sch::TevOpRgb::InvRrr:
        return tg::ResDetailedCombinerStage::OperandRgb_InverseRrr;
    case sch::TevOpRgb::Ggg:
        return tg::ResDetailedCombinerStage::OperandRgb_Ggg;
    case sch::TevOpRgb::InvGgg:
        return tg::ResDetailedCombinerStage::OperandRgb_InverseGgg;
    case sch::TevOpRgb::Bbb:
        return tg::ResDetailedCombinerStage::OperandRgb_Bbb;
    case sch::TevOpRgb::InvBbb:
        return tg::ResDetailedCombinerStage::OperandRgb_InverseBbb;
    default:
        return tg::ResDetailedCombinerStage::OperandRgb_Rgb;
    }
}

tg::ResDetailedCombinerStage::OperandAlpha
GetRawTevOpAlp(sch::TevOpAlp value)
{
    switch (value)
    {
    case sch::TevOpAlp::Alpha:
        return tg::ResDetailedCombinerStage::OperandAlpha_Alpha;
    case sch::TevOpAlp::InvAlpha:
        return tg::ResDetailedCombinerStage::OperandAlpha_InverseAlpha;
    case sch::TevOpAlp::R:
        return tg::ResDetailedCombinerStage::OperandAlpha_R;
    case sch::TevOpAlp::InvR:
        return tg::ResDetailedCombinerStage::OperandAlpha_InverseR;
    case sch::TevOpAlp::G:
        return tg::ResDetailedCombinerStage::OperandAlpha_G;
    case sch::TevOpAlp::InvG:
        return tg::ResDetailedCombinerStage::OperandAlpha_InverseG;
    case sch::TevOpAlp::B:
        return tg::ResDetailedCombinerStage::OperandAlpha_B;
    case sch::TevOpAlp::InvB:
        return tg::ResDetailedCombinerStage::OperandAlpha_InverseB;
    default:
        return tg::ResDetailedCombinerStage::OperandAlpha_Alpha;
    }
}

tg::ResDetailedCombinerStage::ConstColorSelect
GetRawTevKonstSel(sch::TevKonstSel value)
{
    switch (value)
    {
    case sch::TevKonstSel::WhiteColor:
        return tg::ResDetailedCombinerStage::ConstColorSelect_White;
    case sch::TevKonstSel::BlackColor:
        return tg::ResDetailedCombinerStage::ConstColorSelect_Black;
    case sch::TevKonstSel::K0:
        return tg::ResDetailedCombinerStage::ConstColorSelect_C0;
    case sch::TevKonstSel::K1:
        return tg::ResDetailedCombinerStage::ConstColorSelect_C1;
    case sch::TevKonstSel::K2:
        return tg::ResDetailedCombinerStage::ConstColorSelect_C2;
    case sch::TevKonstSel::K3:
        return tg::ResDetailedCombinerStage::ConstColorSelect_C3;
    case sch::TevKonstSel::K4:
        return tg::ResDetailedCombinerStage::ConstColorSelect_C4;
    default:
        return tg::ResDetailedCombinerStage::ConstColorSelect_White;
    }
}

tg::ResDetailedCombinerStage::Scale
GetRawTevScale(sch::TevScale value)
{
    switch (value)
    {
    case sch::TevScale::V1:
        return tg::ResDetailedCombinerStage::Scale_1;
    case sch::TevScale::V2:
        return tg::ResDetailedCombinerStage::Scale_2;
    case sch::TevScale::V4:
        return tg::ResDetailedCombinerStage::Scale_4;
    default:
        return tg::ResDetailedCombinerStage::Scale_1;
    }
}

tg::AlphaTest
GetAlphaTest(sch::Compare comp)
{
    switch (comp)
    {
    case sch::Compare::Never:
        return tg::AlphaTest_Never;
    case sch::Compare::Less:
        return tg::AlphaTest_Less;
    case sch::Compare::Equal:
        return tg::AlphaTest_Equal;
    case sch::Compare::LEqual:
        return tg::AlphaTest_LessEqual;
    case sch::Compare::Greater:
        return tg::AlphaTest_Greater;
    case sch::Compare::NEqual:
        return tg::AlphaTest_NotEqual;
    case sch::Compare::GEqual:
        return tg::AlphaTest_GreaterEqual;
    case sch::Compare::Always:
        return tg::AlphaTest_Always;
    default:
        Debug::Assert(false);
        return tg::AlphaTest_Always;
    }
}

tg::BlendOp
GetBlendOp(sch::BlendOp value)
{
    switch (value)
    {
    case sch::BlendOp::Add:
        return tg::BlendOp_Add;
    case sch::BlendOp::Subtract:
        return tg::BlendOp_Subtract;
    case sch::BlendOp::ReverseSubtract:
        return tg::BlendOp_ReverseSubtract;
    case sch::BlendOp::SelectMin:
        return tg::BlendOp_SelectMin;
    case sch::BlendOp::SelectMax:
        return tg::BlendOp_SelectMax;
    default:
        Debug::Assert(false);
        return tg::BlendOp_Disable;
    }
}

tg::BlendFactor
GetBlendFactorSrc(sch::BlendFactor factor)
{
    switch (factor)
    {
    case sch::BlendFactor::V0:
        return tg::BlendFactor_0;
    case sch::BlendFactor::V1_0:
        return tg::BlendFactor_1;
    case sch::BlendFactor::DstClr:
        return tg::BlendFactor_DstColor;
    case sch::BlendFactor::InvDstClr:
        return tg::BlendFactor_InvDstColor;
    case sch::BlendFactor::SrcAlpha:
        return tg::BlendFactor_SrcAlpha;
    case sch::BlendFactor::InvSrcAlpha:
        return tg::BlendFactor_InvSrcAlpha;
    case sch::BlendFactor::DstAlpha:
        return tg::BlendFactor_DstAlpha;
    case sch::BlendFactor::InvDstAlpha:
        return tg::BlendFactor_InvDstAlpha;
    case sch::BlendFactor::SrcClr:
        return tg::BlendFactor_SrcColor;
    case sch::BlendFactor::InvSrcClr:
        return tg::BlendFactor_InvSrcColor;
    default:
        Debug::Assert(false);
        return tg::BlendFactor_1;
    }
}

tg::BlendFactor
GetBlendFactorDst(sch::BlendFactor factor)
{
    switch (factor)
    {
    case sch::BlendFactor::V0:
        return tg::BlendFactor_0;
    case sch::BlendFactor::V1_0:
        return tg::BlendFactor_1;
    case sch::BlendFactor::DstClr:
        return tg::BlendFactor_DstColor;
    case sch::BlendFactor::InvDstClr:
        return tg::BlendFactor_InvDstColor;
    case sch::BlendFactor::SrcAlpha:
        return tg::BlendFactor_SrcAlpha;
    case sch::BlendFactor::InvSrcAlpha:
        return tg::BlendFactor_InvSrcAlpha;
    case sch::BlendFactor::DstAlpha:
        return tg::BlendFactor_DstAlpha;
    case sch::BlendFactor::InvDstAlpha:
        return tg::BlendFactor_InvDstAlpha;
    case sch::BlendFactor::SrcClr:
        return tg::BlendFactor_SrcColor;
    case sch::BlendFactor::InvSrcClr:
        return tg::BlendFactor_InvSrcColor;
    default:
        Debug::Assert(false);
        return tg::BlendFactor_0;
    }
}

tg::LogicOp
GetLogicOp(sch::LogicOp op)
{
    switch (op)
    {
    case sch::LogicOp::NoOp:
        return tg::LogicOp_Noop;
    case sch::LogicOp::Clear:
        return tg::LogicOp_Clear;
    case sch::LogicOp::Set:
        return tg::LogicOp_Set;
    case sch::LogicOp::Copy:
        return tg::LogicOp_Copy;
    case sch::LogicOp::InvCopy:
        return tg::LogicOp_InvCopy;
    case sch::LogicOp::Inv:
        return tg::LogicOp_Inv;
    case sch::LogicOp::And:
        return tg::LogicOp_And;
    case sch::LogicOp::Nand:
        return tg::LogicOp_Nand;
    case sch::LogicOp::Or:
        return tg::LogicOp_Or;
    case sch::LogicOp::Nor:
        return tg::LogicOp_Nor;
    case sch::LogicOp::Xor:
        return tg::LogicOp_Xor;
    case sch::LogicOp::Equiv:
        return tg::LogicOp_Equiv;
    case sch::LogicOp::RevAnd:
        return tg::LogicOp_RevAnd;
    case sch::LogicOp::InvAnd:
        return tg::LogicOp_InvAnd;
    case sch::LogicOp::RevOr:
        return tg::LogicOp_RevOr;
    case sch::LogicOp::InvOr:
        return tg::LogicOp_InvOr;
    default:
        Debug::Assert(false);
        return tg::LogicOp_Disable;
    }
}

void
SetLayoutBlockPane(tg::ResPane* pPaneBlock, sch::Pane^ pane, bool bViewerHidden)
{
    uint8_t flag = 0;
    uint8_t flagEx = 0;
    if (pane->visible)
    {
        flag |= 1 << tg::PaneFlag_Visible;
    }
    if (pane->influencedAlpha)
    {
        flag |= 1 << tg::PaneFlag_InfluencedAlpha;
    }
    if (pane->locationAdjust)
    {
        flag |= 1 << tg::PaneFlag_LocationAdjust;
    }
    if (bViewerHidden && pane->hidden)
    {
        flagEx |= 1 << tg::PaneFlagEx_ViewerInvisible;
    }
    if (pane->ignorePartsMagnify || pane->partsMagnifyInfluence == sch::PartsMagnifyInfluence::None)
    {
        flagEx |= 1 << tg::PaneFlagEx_IgnorePartsMagnify;
    }
    if (pane->partsMagnifyInfluence == sch::PartsMagnifyInfluence::AdjustToPartsBound)
    {
        flagEx |= 1 << tg::PaneFlagEx_PartsMagnifyAdjustToPartsBound;
    }
    if (pane->extUserDataAnimEnabled)
    {
        flagEx |= 1 << tg::PaneFlagEx_ExtUserDataAnimationEnabled;
    }
    pPaneBlock->flag = flag;
    pPaneBlock->basePosition = GetBasePositionValue(pane->basePositionType) | (GetBasePositionValue(pane->parentRelativePositionType) << 4);
    pPaneBlock->alpha = pane->alpha;
    pPaneBlock->flagEx = flagEx;

    CopyNameStr(pPaneBlock->name, pane->name, sizeof(pPaneBlock->name));

    // ユーザデータ
    // 現状では、最大8文字の文字列データのみ。
    {
        sch::UserDataString^ userDataStr = nullptr;
        if (pane->userData && nullptr != (userDataStr = dynamic_cast<sch::UserDataString^>(pane->userData[0])) && userDataStr->Value)
        {
            CopyNameStr(pPaneBlock->userData, userDataStr->Value, sizeof(pPaneBlock->userData));
        }
    }

    CopyVec3(&pPaneBlock->translate, pane->translate);
    CopyVec3(&pPaneBlock->rotate, pane->rotate);
    CopyVec2(&pPaneBlock->scale, pane->scale);
    CopyVec2(&pPaneBlock->size, pane->size);
}

uint32_t
GetColorU32(sch::Color4^ color)
{
    return (color->r << 24) | (color->g << 16) | (color->b << 8) | color->a;
}

void
CopyTexCoords(nn::ui2d::ResVec2* dst, sch::TexCoord^ src)
{
    CopyTexVec2(&dst[tg::PaneVertex_LeftTop], src->texLT);
    CopyTexVec2(&dst[tg::PaneVertex_RightTop], src->texRT);
    CopyTexVec2(&dst[tg::PaneVertex_LeftBottom], src->texLB);
    CopyTexVec2(&dst[tg::PaneVertex_RightBottom], src->texRB);
}

const tg::ResWindowInflation
CopyWindowContentInflation(sch::InflationRect^ inflation)
{
    tg::ResWindowInflation ret;

    const float mag = static_cast<float>(tg::WindowInflationFixedPoint_ScalingFactor);
    ret.left = static_cast<int16_t>(inflation->l * mag);
    ret.right = static_cast<int16_t>(inflation->r * mag);
    ret.top = static_cast<int16_t>(inflation->t * mag);
    ret.bottom = static_cast<int16_t>(inflation->b * mag);

    return ret;
}

const tg::ResWindowFrameSize
    CopyWindowFrameSize(sch::Window^ wnd)
{
    sch::WindowFrameSize^ frameSize = wnd->frameSize;
    tg::ResWindowFrameSize ret;

    ret.left = static_cast<uint16_t>(frameSize->l);
    ret.right = static_cast<uint16_t>(frameSize->r);
    ret.top = static_cast<uint16_t>(frameSize->t);
    ret.bottom = static_cast<uint16_t>(frameSize->b);

    return ret;
}

}   // namespace

void
BinaryLytWriter::WriteBinaryFileHeader(uint32_t fileSize, uint16_t dataBlocks, bool bBannerVersion, bool bCompatible_0_12_x)
{
    BinaryWriterBase::WriteBinaryFileHeader(tg::FileSignatureFlyt, fileSize, dataBlocks, bBannerVersion, bCompatible_0_12_x);
}

void
BinaryLytWriter::WriteBlock(sch::ScreenSetting^ setting, Schema::Flyt::Vec2^ partsSize, uint32_t blockSize)
{
    tg::ResLayout layout;
    std::memset(&layout, 0, sizeof(layout));
    layout.originType = GetOriginType(setting->origin);
    CopyVec2(&layout.layoutSize, setting->layoutSize);
    CopyVec2(&layout.partsSize, partsSize);
    WriteBlockHeader(&layout, tg::DataBlockKindLayout, blockSize);
}

void
BinaryLytWriter::WriteControlBlock(sch::Control^ control, uint32_t blockSize, uint32_t controlUserNameOffset, uint32_t controlParameterPaneNamesOffset,
    uint32_t controlFunctionalPaneParameterNameOffsetsOffset, uint32_t controlFunctionalAnimParameterNameOffsetsOffset)
{
    tg::ResControl controlBlock;
    std::memset(&controlBlock, 0, sizeof(controlBlock));
    controlBlock.controlUserNameOffset = controlUserNameOffset;
    controlBlock.controlFunctionalPaneNamesOffset = controlParameterPaneNamesOffset;
    controlBlock.controlFunctionalPaneParameterNameOffsetsOffset = controlFunctionalPaneParameterNameOffsetsOffset;
    controlBlock.controlFunctionalAnimParameterNameOffsetsOffset = controlFunctionalAnimParameterNameOffsetsOffset;
    controlBlock.controlFunctionalPaneCount = control->parameterPane ? control->parameterPane->Length : 0;
    controlBlock.controlFunctionalAnimCount = control->parameterAnimation ? control->parameterAnimation->Length : 0;
    WriteBlockHeader(&controlBlock, tg::DataBlockKindControl, blockSize);
}

void
BinaryLytWriter::WriteStruct(SortedList<String^, sch::TextureFile^>^ texFiles, uint32_t blockSize)
{
    tg::ResTextureList texList;
    std::memset(&texList, 0, sizeof(texList));
    // インダイレクトと非インダイレクトの両方を出力するテクスチャがある場合は、両方書き込む
    // ことになるので、texNumはtexFiles->Countより増える場合があるため、ここで数え直す
    int count = 0;
    for each (KeyValuePair<String^, sch::TextureFile^>^ pair in texFiles)
    {
        if (pair->Value->IsNoIndirect)
        {
            count += 1;
        }
        if (pair->Value->IsIndirect)
        {
            count += 1;
        }
    }
    texList.texCount = count;
    WriteBlockHeader(&texList, tg::DataBlockKindTextureList, blockSize);
}

void
BinaryLytWriter::WriteTextureStruct(uint32_t offset)
{
    tg::ResTexture resTex = { 0 };

    resTex.nameStrOffset = offset;
    WriteStructData(resTex);
}

void
BinaryLytWriter::WriteStruct(SortedList<String^, sch::FontFile^>^ fontFiles, uint32_t blockSize)
{
    const Int64 blockHeadFilePos = stream_->Position;

    tg::ResFontList fontList;
    std::memset(&fontList, 0, sizeof(fontList));
    fontList.fontCount = fontFiles->Count;
    WriteBlockHeader(&fontList, tg::DataBlockKindFontList, blockSize);
}

void
BinaryLytWriter::WriteFontStruct(uint32_t offset)
{
    tg::ResFont resFont = { 0 };

    resFont.nameStrOffset = offset;
    WriteStructData(resFont);
}

void BinaryLytWriter::WriteStruct(ICollection<sch::MaterialInfo^>^ matInfos, uint32_t blockSize)
{
    tg::ResMaterialList matList;
    std::memset(&matList, 0, sizeof(matList));
    Debug::Assert(matInfos->Count <= USHRT_MAX);
    matList.materialCount = static_cast<uint16_t>(matInfos->Count);
    WriteBlockHeader(&matList, tg::DataBlockKindMaterialList, blockSize);
}

void
BinaryLytWriter::WriteStruct(sch::MaterialInfo^ src)
{
    tg::ResMaterial dst;
    std::memset(&dst, 0, sizeof(dst));

    CopyNameStr(dst.name, src->name, sizeof(dst.name));

    {
        tg::ResMaterialResourceCount resNum;
        resNum.SetDefault();

        resNum.SetTexMapCount(src->texMap->Length);
        resNum.SetTexSrtCount(src->texMatrix->Length);
        resNum.SetTexCoordGenCount(src->texCoordGen->Length);

        resNum.SetTevStageCount(src->tevStage->Length);
        resNum.SetAlphaCompare(src->alphaCompare != nullptr);
        resNum.SetBlendMode(src->blendMode != nullptr);

        resNum.SetTextureOnly(src->UseTextureOnly);
        resNum.SetThresholdingAlphaInterpolation(src->UseThresholdingAlphaInterpolation);

        resNum.SetSeparateBlendMode(src->blendMode != nullptr && src->blendModeAlpha != nullptr && src->blendModeAlpha->type == sch::BlendMode::Blend);

        if (src->UseCombinerUserShaderSettings && src->combinerUserShader->ExistFileName())
        {
            resNum.SetCombinerUserShader(true);
        }
        else if (src->UseLowLevelCombinerSettings)
        {
            resNum.SetDetailedCombinerStage(true);
        }
        else
        {
            resNum.SetIndirectParameter(src->HasIndirectParameter());
        }
        resNum.SetProjectionTexGenCount(src->GetProjectionTexGenParameterNum());
        resNum.SetFontShadowParameter(src->HasFontShadowParameter());

        for (int i = 0; i < src->texMap->Length; ++i)
        {
            if (src->texMap[i]->textureResourceType == sch::TextureResourceType::LocalCaptured ||
                src->texMap[i]->textureResourceType == sch::TextureResourceType::OverrideCaptured)
            {
                resNum.SetTexMapAdditionalInfo(true);
                break;
            }
        }

        dst.resCount = resNum;
    }

    WriteStructData(dst);

    // ResConstantColor データの書き出し
    tg::ResConstantColor    constantColors;

    constantColors.SetDefault();

    // カラーを書き出し
    if (!src->color->byte_convert)
    {
        constantColors.types = nn::ui2d::detail::SetBits(constantColors.types, 0, 1, uint8_t(1));
    }
    ++constantColors.count;
    for (int i = 0; i < std::min(src->konstColors->Length, tg::MatColorMax - 1); ++i)
    {
        if (!src->konstColors[i]->byte_convert)
        {
            constantColors.types = nn::ui2d::detail::SetBits(constantColors.types, 1 + i, 1, uint8_t(1));
        }
        ++constantColors.count;
    }

    WriteStructData(constantColors);

    // 要素数を増やすのであればアライメントの調整が必要。
    // オフセットテーブル書き出し
    uint8_t offset = sizeof(tg::ResConstantColor) + sizeof(uint8_t) * 2;
    WriteU8(offset);
    offset += src->color->byte_convert ? sizeof(uint32_t) : sizeof(float) * 4;
    for (int i = 0; i < std::min(src->konstColors->Length, tg::MatColorMax - 1); ++i)
    {
        WriteU8(offset);
        offset += src->konstColors[i]->byte_convert ? sizeof(uint32_t) : sizeof(float) * 4;
    }

    // カラーを書き出し
    WriteColor4f(src->color);
    for (int i = 0; i < std::min(src->konstColors->Length, tg::MatColorMax - 1); ++i)
    {
        WriteColor4f(src->konstColors[i]);
    }
}

void BinaryLytWriter::WriteStruct(ICollection<Schema::Flyt::ShapeInfo^>^ shapeInfos, uint32_t blockSize)
{
    tg::ResShapeInfoList shapeInfoList;
    std::memset(&shapeInfoList, 0, sizeof(shapeInfoList));
    Debug::Assert(shapeInfos->Count <= USHRT_MAX);
    shapeInfoList.shapeInfoCount = static_cast<uint16_t>(shapeInfos->Count);
    WriteBlockHeader(&shapeInfoList, tg::DataBlockKindShapeInfoList, blockSize);
}

void
BinaryLytWriter::WriteStruct(Schema::Flyt::ShapeInfo^ src)
{
    tg::ResShapeInfo    dst;
    std::memset(&dst, 0, sizeof(dst));

    switch (src->ShapeType)
    {
    case Schema::Flyt::ShapeType::GfxPrimitiveRoundRect:
        {
            dst.type = tg::ResShapeInfo::ShapeInfo_GfxPrimitiveRoundRect;
            WriteStructData(dst);

            tg::ResShapeInfoGfxPrimitiveRoundRect   roundRect;
            roundRect.radius = src->GfxShapeParam->roundRadius;
            roundRect.slice = src->GfxShapeParam->roundSlice;
            WriteStructData(roundRect);
        }
        break;
    case Schema::Flyt::ShapeType::GfxPrimitiveCircle:
        {
            dst.type = tg::ResShapeInfo::ShapeInfo_GfxPrimitiveCircle;
            WriteStructData(dst);

            tg::ResShapeInfoGfxPrimitiveCircle  circle;
            circle.slice = src->GfxShapeParam->circleSlice;
            WriteStructData(circle);
        }
        break;
    }
}

void
BinaryLytWriter::WriteStruct(SortedList<String^, sch::CaptureTexture^>^ captureTextures, uint32_t blockSize)
{
    tg::ResCaptureTextureList texList;
    std::memset(&texList, 0, sizeof(texList));

    texList.texCount = captureTextures->Count;

    WriteBlockHeader(&texList, tg::DataBlockKindCaptureTextureList, blockSize);
}

size_t
BinaryLytWriter::GetCaptureTextureStructSize(sch::CaptureTexture^ capture)
{
    size_t size = sizeof(tg::ResCaptureTexture);

    for (uint32_t i = 0; i < capture->filterCount; ++i)
    {
        size += sizeof(tg::ResCaptureTextureFilter);
        // フィルタごとの特殊情報がある場合は追加していく。
    }

    return size;
}

void
BinaryLytWriter::WriteCaptureTextureStruct(uint32_t textureNameOffset, uint32_t paneNameOffset, sch::CaptureTexture^ capture)
{
    tg::ResCaptureTexture resCaptureTexture;
    std::memset(&resCaptureTexture, 0, sizeof(resCaptureTexture));

    resCaptureTexture.textureNameStrOffset = textureNameOffset;
    resCaptureTexture.paneNameStrOffset = paneNameOffset;
    resCaptureTexture.clearColor[0] = capture->clearColor->r;
    resCaptureTexture.clearColor[1] = capture->clearColor->g;
    resCaptureTexture.clearColor[2] = capture->clearColor->b;
    resCaptureTexture.clearColor[3] = capture->clearColor->a;
    resCaptureTexture.framebufferCaptureEnabled = capture->framebufferCaptureEnabled;
    resCaptureTexture.captureOnlyFirstFrame = capture->captureOnlyFirstFrame;

    switch (capture->format)
    {
    case sch::TexelFormat::R10G10B10A2:
        resCaptureTexture.textureFormat = nn::gfx::ImageFormat_R10_G10_B10_A2_Unorm;
        break;
    case sch::TexelFormat::RGBA8:
    default:
        resCaptureTexture.textureFormat = nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm;
        break;
    }

    resCaptureTexture.filterCount = capture->filterCount;

    WriteStructData(resCaptureTexture);

    for (uint32_t i = 0; i < capture->filterCount; ++i)
    {
        tg::ResCaptureTextureFilter resFilter = { 0 };
        std::memset(&resFilter, 0, sizeof(resFilter));

        resFilter.type = 0;
        resFilter.textureScale = capture->filters[i]->scale;
        WriteStructData(resFilter);
    }
}

void
BinaryLytWriter::WriteStruct(sch::TexMap^ src, SortedList<String^, sch::TextureFile^>^ textureFileList,  bool is_indirect)
{
    tg::ResTexMap dst;
    std::memset(&dst, 0, sizeof(dst));

    // textureFileListをイテレートして書き込むべきインデックスを求める。
    // textureFileListはこのリストの順番に書き込まれるが、インダイレクトとインダイレクトでない
    // 用途の両方に使われている場合は2つ連続で書き込まれる(インダイレクトでない用途が先)ので
    // 注意が必要。
    int count = 0;
    for each (KeyValuePair<String^, sch::TextureFile^>^ pair in textureFileList)
    {
        if (pair->Key->Equals(src->imageName))
        {
            // キャプチャテクスチャはインダイレクト関連のファイル参照オフセットを処理しない。
            if (src->textureResourceType == sch::TextureResourceType::LocalFile &&
                is_indirect)
            {
                if (pair->Value->IsNoIndirect)
                {
                    count += 1;
                }
            }
            dst.texIdx = count;
            break;
        }
        else
        {
            if (pair->Value->IsNoIndirect)
            {
                count += 1;
            }
            if (pair->Value->IsIndirect)
            {
                count += 1;
            }
        }
    }
    dst.SetWarpModeS(tg::TexWrap(src->wrap_s));
    dst.SetWarpModeT(tg::TexWrap(src->wrap_t));
    dst.SetMinFilter(tg::TexFilter(src->minFilter));
    dst.SetMagFilter(tg::TexFilter(src->magFilter));
    WriteStructData(dst);
}

void
BinaryLytWriter::WriteStruct(Schema::Flyt::TexMapAdditionalInfo^ info)
{
    tg::ResTexMapAdditionalInfo dst;
    ::memset(&dst, 0, sizeof(dst));

    dst.info = info->GetFlags();

    WriteStructData(dst);
}

void
BinaryLytWriter::WriteStruct(sch::TexMatrix^ src)
{
    tg::ResTexSrt dst;
    ::memset(&dst, 0, sizeof(dst));

    CopyVec2(&dst.translate, src->translate);
    dst.rotate = src->rotate;
    CopyVec2(&dst.scale, src->scale);
    WriteStructData(dst);
}

void
BinaryLytWriter::WriteStruct(sch::TexCoordGen^ src)
{
    tg::ResTexCoordGen dst;
    memset(&dst, 0, sizeof(dst));

    dst.Set(tg::TexGenType_Mtx2x4, GetTexGenSrc(src->srcParam));

    // TexCoordGen を書き出します。
    WriteStructData(static_cast<uint8_t>(dst.GetTexGenType()));
    WriteStructData(static_cast<uint8_t>(dst.GetTexGenSrc()));

    uint8_t paddingByte2[2] = { 0, 0 };
    WriteStructData(paddingByte2);

    uint8_t paddingByte4[4] = { 0, 0, 0, 0 };
    WriteStructData(paddingByte4);

    uint8_t dummyPtrForProjectionTexGenParamaters[8] = { 0, 0, 0, 0, 0, 0, 0, 0 };
    WriteStructData(dummyPtrForProjectionTexGenParamaters);
}

void
BinaryLytWriter::WriteLineWidthOffset(Schema::Flyt::TextBox^ textBox)
{
    WriteU8(static_cast<uint8_t>(textBox->lineWidthOffset->Length));
    for (int i = 0; i < textBox->lineWidthOffset->Length; i++)
    {
        WriteF32(textBox->lineWidthOffset[i]->lineOffset);
    }
    for (int i = 0; i < textBox->lineWidthOffset->Length; i++)
    {
        WriteF32(textBox->lineWidthOffset[i]->lineWidth);
    }
}

void
BinaryLytWriter::WritePerCharacterTransformStruct(Schema::Flyt::TextBox^ textBox, bool hasAnimationInfo)
{
    tg::ResPerCharacterTransformExtended dst;
    memset(&dst, 0, sizeof(dst));

    dst.evalTimeOffset = textBox->perCharacterTransformEvalTimeOffset;
    dst.evalTimeWidth = textBox->perCharacterTransformEvalTimeWidth;
    dst.loopType = GetPerCharacterTransformLoopType(textBox->perCharacterTransformLoopType);
    dst.originV = GetPerCharacterTransformOriginV(textBox->perCharacterTransformOriginV);
    dst.originVOffset = textBox->perCharacterTransformOriginVOffset;
    dst.hasAnimationInfo = hasAnimationInfo;

    dst.fixSpaceWidth = textBox->perCharacterTransformFixSpaceWidth;
    dst.fixSpaceOrigin = static_cast<uint8_t>(textBox->perCharacterTransformFixSpaceOrigin);

    WriteStructData(dst);
}

void
BinaryLytWriter::WriteColor(sch::Color4^ src)
{
    tg::ResColor dst = { 0 };
    if (src)
    {
        CopyColor(&dst, src);
    }
    WriteStructData(dst);
}

void
BinaryLytWriter::WriteStruct(sch::Material_CTRTevStage^ src)
{
    tg::ResTevStage dst;
    ::memset(&dst, 0, sizeof(dst));

    if (src->rgb)
    {
        dst.SetCombineRgb(GetTevMode(src->rgb->mode));
    }

    if (src->alpha)
    {
        dst.SetCombineAlpha(GetTevMode(src->alpha->mode));
    }

    WriteStructData(dst);
}

void
BinaryLytWriter::WriteStruct(sch::Material_CTRAlphaCompare^ src)
{
    tg::ResAlphaCompare dst;
    ::memset(&dst, 0, sizeof(dst));

    if (src)
    {
        dst.Set(GetAlphaTest(src->comp), src->ref);
    }
    WriteStructData(dst);
}

void
BinaryLytWriter::WriteStruct(sch::Material_CTRBlendMode^ src)
{
    tg::ResBlendMode dst;
    ::memset(&dst, 0, sizeof(dst));

    if (src)
    {
        dst.Set(
            (src->type == sch::BlendMode::Blend)? GetBlendOp(src->blendOp) : tg::BlendOp_Disable,
            GetBlendFactorSrc(src->srcFactor),
            GetBlendFactorDst(src->dstFactor),
            (src->type == sch::BlendMode::Logic)? GetLogicOp(src->logicOp) : tg::LogicOp_Disable);
    }
    WriteStructData(dst);
}

void
BinaryLytWriter::WriteIndirectParameter(Schema::Flyt::Material_CTRTevStage^ src)
{
    tg::ResIndirectParameter dst;
    ::memset(&dst, 0, sizeof(dst));

    if (src)
    {
        dst.rotate = src->rgb->indirectRotate;
        dst.scale.x = src->rgb->indirectScale->x;
        dst.scale.y = src->rgb->indirectScale->y;
    }
    WriteStructData(dst);
}

void
BinaryLytWriter::WriteProjectionTexGenParameter(Schema::Flyt::TexCoordGen^ src)
{
    tg::ResProjectionTexGenParamaters dst;
    ::memset(&dst, 0, sizeof(dst));

    if (src)
    {
        dst.scale.x = src->projectionScale->x;
        dst.scale.y = src->projectionScale->y;

        dst.translate.x = src->projectionTrans->x;
        dst.translate.y = src->projectionTrans->y;

        dst.flag = 0;
        if (src->fittingLayoutSizeEnabled)
        {
            dst.flag |= 1 << tg::TexProjectionFlag_FittingLayoutSize;
        }
        if (src->fittingPaneSizeEnabled)
        {
            dst.flag |= 1 << tg::TexProjectionFlag_FittingPaneSizeEnabled;
        }
        if (src->adjustProjectionSREnabled)
        {
            dst.flag |= 1 << tg::TexProjectionFlag_AdjustProjectionScaleRotateEnabled;
        }
    }
    WriteStructData(dst);
}

void
BinaryLytWriter::WriteRawTevStageInfo(Schema::Flyt::MaterialInfo^ src)
{
    tg::ResDetailedCombinerStageInfo dst;
    ::memset(&dst, 0, sizeof(dst));

    if (src)
    {
        if (src->konstColors && src->konstColors->Length > 1)
        {
            for (int i = 0; i < std::min(tg::DetailedCombinerConstantColor, src->konstColors->Length - 1); i++)
            {
                tg::ResColor color;
                sch::Color4  ConstColor;
                ConstColor.r = static_cast<uint8_t>(Math::Min(src->konstColors[i + 1]->r * 255.0f, 255.0f));
                ConstColor.g = static_cast<uint8_t>(Math::Min(src->konstColors[i + 1]->g * 255.0f, 255.0f));
                ConstColor.b = static_cast<uint8_t>(Math::Min(src->konstColors[i + 1]->b * 255.0f, 255.0f));
                ConstColor.a = static_cast<uint8_t>(Math::Min(src->konstColors[i + 1]->a * 255.0f, 255.0f));
                CopyColor(&color, %ConstColor);
                dst.SetConstantColor(i, color);
            }
        }
    }

    WriteStructData(dst);
}

void
BinaryLytWriter::WriteRawTevStage(Schema::Flyt::Material_CTRTevStage^ src)
{
    tg::ResDetailedCombinerStage dst = GetDetailedCombinerStage(src);
    WriteStructData(dst);
}

uint32_t BinaryLytWriter::GetDetailedCombinerStageBit0(Schema::Flyt::Material_CTRTevStage^ src)
{
    return GetDetailedCombinerStage(src).bits0;
}

uint32_t BinaryLytWriter::GetDetailedCombinerStageBit1(Schema::Flyt::Material_CTRTevStage^ src)
{
    return GetDetailedCombinerStage(src).bits1;
}

uint32_t BinaryLytWriter::GetDetailedCombinerStageBit2(Schema::Flyt::Material_CTRTevStage^ src)
{
    return GetDetailedCombinerStage(src).bits2;
}

uint32_t BinaryLytWriter::GetDetailedCombinerStageBit3(Schema::Flyt::Material_CTRTevStage^ src)
{
    return GetDetailedCombinerStage(src).bits3;
}

tg::ResDetailedCombinerStage
BinaryLytWriter::GetDetailedCombinerStage(Schema::Flyt::Material_CTRTevStage^ src)
{
    tg::ResDetailedCombinerStage dst;
    ::memset(&dst, 0, sizeof(dst));

    if (src)
    {
        if (src->rgb)
        {
            dst.SetCombineRgb(GetRawTevCombine(src->rgb->mode));
            dst.SetScaleRgb(GetRawTevScale(src->rgb->scale));
            dst.SetSrcRgb0(GetRawTevSrc(src->rgb->arg[0]->source));
            dst.SetSrcRgb1(GetRawTevSrc(src->rgb->arg[1]->source));
            dst.SetSrcRgb2(GetRawTevSrc(src->rgb->arg[2]->source));
            dst.SetOperandRgb0(GetRawTevOpRgb(src->rgb->arg[0]->op));
            dst.SetOperandRgb1(GetRawTevOpRgb(src->rgb->arg[1]->op));
            dst.SetOperandRgb2(GetRawTevOpRgb(src->rgb->arg[2]->op));
            dst.SetConstColorSelectRgb(GetRawTevKonstSel(src->rgb->konst));
            dst.SetSavePrevRgb(src->rgb->copyReg);
            dst.SetCombineRgbSourceCount(dst.GetCombineRgb());
        }

        if (src->alpha)
        {
            dst.SetCombineAlpha(GetRawTevCombine(src->alpha->mode));
            dst.SetScaleAlpha(GetRawTevScale(src->alpha->scale));
            dst.SetSrcAlpha0(GetRawTevSrc(src->alpha->arg[0]->source));
            dst.SetSrcAlpha1(GetRawTevSrc(src->alpha->arg[1]->source));
            dst.SetSrcAlpha2(GetRawTevSrc(src->alpha->arg[2]->source));
            dst.SetOperandAlpha0(GetRawTevOpAlp(src->alpha->arg[0]->op));
            dst.SetOperandAlpha1(GetRawTevOpAlp(src->alpha->arg[1]->op));
            dst.SetOperandAlpha2(GetRawTevOpAlp(src->alpha->arg[2]->op));
            dst.SetConstColorSelectAlpha(GetRawTevKonstSel(src->alpha->konst));
            dst.SetSavePrevAlpha(src->alpha->copyReg);
            dst.SetCombineAlphaSourceCount(dst.GetCombineAlpha());
        }
    }

    return dst;
}

void
BinaryLytWriter::WriteFontShadowParameter(Schema::Flyt::Color4^ blackColor, Schema::Flyt::Color4^ whiteColor)
{
    tg::ResFontShadowParameter dst;
    ::memset(&dst, 0, sizeof(dst));

    if (isDegamma_)
    {
        // ガンマ特性を与える、0～255になっていることに注意が必要
        dst.blackInterporateColor[0] = static_cast<uint8_t>(DegammaValue(blackColor->r));
        dst.blackInterporateColor[1] = static_cast<uint8_t>(DegammaValue(blackColor->g));
        dst.blackInterporateColor[2] = static_cast<uint8_t>(DegammaValue(blackColor->b));

        dst.whiteInterporateColor[0] = static_cast<uint8_t>(DegammaValue(whiteColor->r));
        dst.whiteInterporateColor[1] = static_cast<uint8_t>(DegammaValue(whiteColor->g));
        dst.whiteInterporateColor[2] = static_cast<uint8_t>(DegammaValue(whiteColor->b));
        dst.whiteInterporateColor[3] = whiteColor->a;
    }
    else
    {
        dst.blackInterporateColor[0] = blackColor->r;
        dst.blackInterporateColor[1] = blackColor->g;
        dst.blackInterporateColor[2] = blackColor->b;

        dst.whiteInterporateColor[0] = whiteColor->r;
        dst.whiteInterporateColor[1] = whiteColor->g;
        dst.whiteInterporateColor[2] = whiteColor->b;
        dst.whiteInterporateColor[3] = whiteColor->a;
    }

    dst.reserve = 0;

    WriteStructData(dst);
}

void
BinaryLytWriter::WriteBlock(uint32_t sig)
{
    WriteBlockHeader(sig);
}

void
BinaryLytWriter::WriteStruct(sch::TexCoord^ src)
{
    nn::ui2d::ResVec2 dst[tg::PaneVertex_MaxPaneVertex];
    ::memset(&dst, 0, sizeof(dst));

    CopyTexCoords(dst, src);
    WriteStructData(dst);
}

void
BinaryLytWriter::WriteNullPane(sch::Pane^ pane)
{
    tg::ResPane paneBlock = { 0 };
    SetLayoutBlockPane(&paneBlock, pane, isEnableViewerHidden_);
    WriteBlockHeader(&paneBlock, tg::DataBlockKindPane);
}

void BinaryLytWriter::WriteBlock(sch::Picture^ picture, sch::Pane^ pane, uint32_t blockSize, uint16_t materialIdx, bool shapeInfoEnabled)
{
    tg::ResPicture pictureBlock;
    std::memset(&pictureBlock, 0, sizeof(pictureBlock));

    SetLayoutBlockPane(&pictureBlock, pane, isEnableViewerHidden_);

    CopyColor(&pictureBlock.vtxCols[tg::VertexColor_LeftTop], picture->vtxColLT);
    CopyColor(&pictureBlock.vtxCols[tg::VertexColor_RightTop], picture->vtxColRT);
    CopyColor(&pictureBlock.vtxCols[tg::VertexColor_LeftBottom], picture->vtxColLB);
    CopyColor(&pictureBlock.vtxCols[tg::VertexColor_RightBottom], picture->vtxColRB);

    pictureBlock.materialIdx = materialIdx;
    pictureBlock.materialIdx = pictureBlock.materialIdx;

    pictureBlock.texCoordCount = picture->texCoord->Length;

    // 追加情報フラグ
    //
    //  0 bit ... シェイプバイナリインデックス情報があるかどうか。
    pictureBlock.flags = shapeInfoEnabled ? (1 << tg::PicturePaneFlag_ShapeBinaryIndex) : 0;

    WriteBlockHeader(&pictureBlock, tg::DataBlockKindPicture, blockSize);
}

void BinaryLytWriter::WriteBlock(sch::TextBox^ textBox, sch::Pane^ pane, uint32_t blockSize, uint16_t textBufBytes, uint16_t textStrBytes, uint32_t textStrOffset, uint16_t materialIdx, uint16_t fontIdx, uint32_t textIDOffset, uint32_t lineWidthOffsetOffset, uint32_t perCharacterTransformOffset, bool isKeepingFontScaleEnabled)
{
    tg::ResTextBox textBoxBlock;
    std::memset(&textBoxBlock, 0, sizeof(textBoxBlock));

    uint16_t textBoxFlag = 0;
    if (textBox->shadowEnabled)
    {
        textBoxFlag |= 1 << tg::TextBoxFlag_ShadowEnabled;
    }
    if (textBox->allocateStringLengthSpecified)
    {
        textBoxFlag |= 1 << tg::TextBoxFlag_ForceAssignTextLength;
    }
    if (textBox->invisibleBorderEnabled)
    {
        textBoxFlag |= 1 << tg::TextBoxFlag_InvisibleBorderEnabled;
    }
    if (textBox->doubleDrawnBorderEnabled)
    {
        textBoxFlag |= 1 << tg::TextBoxFlag_DoubleDrawnBorderEnabled;
    }
    if (textBox->perCharacterTransformEnabled)
    {
        textBoxFlag |= 1 << tg::TextBoxFlag_PerCharacterTransformEnabled;
    }
    if (textBox->centerCeilingEnabled)
    {
        textBoxFlag |= 1 << tg::TextBoxFlag_CenterCeilingEnabled;
    }
    if (textBox->drawFromRightToLeft)
    {
        textBoxFlag |= 1 << tg::TextBoxFlag_DrawFromRightToLeft;
    }
    if (textBox->lineWidthOffsetEnabled && !textBox->perCharacterTransformOriginToCenter)
    {
        textBoxFlag |= 1 << tg::TextBoxFlag_LineWidthOffsetEnabled;
    }
    if (textBox->extendedTagEnabled)
    {
        textBoxFlag |= 1 << tg::TextBoxFlag_ExtendedTagEnabled;
    }
    if (textBox->perCharacterTransformSplitByCharWidth)
    {
        textBoxFlag |= 1 << tg::TextBoxFlag_PerCharacterTransformSplitByCharWidth;
    }
    if (textBox->perCharacterTransformAutoShadowAlpha)
    {
        textBoxFlag |= 1 << tg::TextBoxFlag_PerCharacterTransformAutoShadowAlpha;
    }
    if (textBox->perCharacterTransformOriginToCenter && textBox->perCharacterTransformEnabled)
    {
        textBoxFlag |= 1 << tg::TextBoxFlag_PerCharacterTransformOriginToCenter;
    }
    if (isKeepingFontScaleEnabled)
    {
        textBoxFlag |= 1 << tg::TextBoxFlag_KeepingFontScaleEnabled;
    }
    if (textBox->perCharacterTransformFixSpaceEnabled)
    {
        textBoxFlag |= 1 << tg::TextBoxFlag_PerCharacterTransformFixSpace;
    }
    textBoxBlock.textBoxFlag = textBoxFlag;

    SetLayoutBlockPane(&textBoxBlock, pane, isEnableViewerHidden_);

    textBoxBlock.textBufBytes = textBufBytes;
    textBoxBlock.textStrBytes = textStrBytes;

    textBoxBlock.materialIdx = materialIdx;
    textBoxBlock.fontIdx = fontIdx;

    textBoxBlock.textPosition = GetBasePositionValue(textBox->positionType);
    textBoxBlock.textAlignment = GetTextAlignment(textBox->textAlignment);
    textBoxBlock.italicRatio = textBox->italicFactor;

    textBoxBlock.textStrOffset = textStrOffset;
    CopyColor(&textBoxBlock.textCols[tg::TextColor_Top], textBox->topColor);
    CopyColor(&textBoxBlock.textCols[tg::TextColor_Bottom], textBox->bottomColor);

    if (isKeepingFontScaleEnabled)
    {
        // フォントが差し変わってもスケールを維持するために、
        // フォントサイズの代わりにフォントスケールを格納しておく
        textBoxBlock.fontSize.Set(
            textBox->fontSize->x / textBox->fontSizeOriginal->x,
            textBox->fontSize->y / textBox->fontSizeOriginal->y
        );
    }
    else
    {
        CopyVec2(&textBoxBlock.fontSize, textBox->fontSize);
    }

    textBoxBlock.charSpace = textBox->charSpace;
    textBoxBlock.lineSpace = textBox->lineSpace;
    textBoxBlock.textIdOffset = textIDOffset;

    CopyVec2(&textBoxBlock.shadowOffset, textBox->shadowOffset);
    CopyVec2(&textBoxBlock.shadowScale, textBox->shadowScale);
    textBoxBlock.shadowItalicRatio = textBox->shadowItalicFactor;
    CopyColor(&textBoxBlock.shadowCols[tg::TextColor_Top], textBox->shadowTopColor);
    CopyColor(&textBoxBlock.shadowCols[tg::TextColor_Bottom], textBox->shadowBottomColor);

    textBoxBlock.lineWidthOffsetOffset = lineWidthOffsetOffset;
    textBoxBlock.perCharacterTransformOffset = perCharacterTransformOffset;

    WriteBlockHeader(&textBoxBlock, tg::DataBlockKindTextBox, blockSize);
}

void BinaryLytWriter::WriteBlock(sch::Window^ window, sch::Pane^ pane, uint32_t blockSize, uint32_t contentOffset, uint32_t frameOffsetTableOffset)
{
    tg::ResWindow windowBlock;
    std::memset(&windowBlock, 0, sizeof(windowBlock));

    SetLayoutBlockPane(&windowBlock, pane, isEnableViewerHidden_);

    windowBlock.inflation = CopyWindowContentInflation(window->contentInflation);
    windowBlock.frameSize = CopyWindowFrameSize(window);
    windowBlock.frameCount = window->frame->Length;
    windowBlock.windowFlags = nn::ui2d::detail::SetBits(windowBlock.windowFlags, nn::ui2d::WindowFlag_UseOneMaterialForAll, 1, static_cast<uint8_t>(window->useLTMaterial ? window->useLTMaterial->value : 0));
    windowBlock.windowFlags = nn::ui2d::detail::SetBits(windowBlock.windowFlags, nn::ui2d::WindowFlag_UseVertexColorAll, 1, static_cast<uint8_t>(window->useVtxColorForAllWindow));
    windowBlock.windowFlags = nn::ui2d::detail::SetBits(windowBlock.windowFlags, nn::ui2d::WindowFlag_WindowKind0, nn::ui2d::WindowFlag_WindowKindLength, static_cast<uint8_t>(window->kind));
    windowBlock.windowFlags = nn::ui2d::detail::SetBits(windowBlock.windowFlags, nn::ui2d::WindowFlag_NotDrawContent, 1, static_cast<uint8_t>(window->notDrawContent));

    windowBlock.contentOffset = contentOffset;

    windowBlock.frameOffsetTableOffset = frameOffsetTableOffset;

    WriteBlockHeader(&windowBlock, tg::DataBlockKindWindow, blockSize);
}

void BinaryLytWriter::WriteBlock(sch::WindowContent^ content, uint16_t materialIdx)
{
    // contentの書き込み
    tg::ResWindowContent resCont;
    memset(&resCont, 0, sizeof(resCont));

    CopyColor(&resCont.vtxCols[tg::VertexColor_LeftTop], content->vtxColLT);
    CopyColor(&resCont.vtxCols[tg::VertexColor_RightTop], content->vtxColRT);
    CopyColor(&resCont.vtxCols[tg::VertexColor_LeftBottom], content->vtxColLB);
    CopyColor(&resCont.vtxCols[tg::VertexColor_RightBottom], content->vtxColRB);


    resCont.materialIdx = materialIdx;

    resCont.texCoordCount = content->texCoord->Length;

    WriteStructData(resCont);
}

uint8_t
BinaryLytWriter::GetWindowFrameType(sch::WindowFrameType frameType)
{
    using namespace nn::ui2d;

    switch (frameType)
    {
    case sch::WindowFrameType::CornerLT:    return tg::WindowFrame_LeftTop;
    case sch::WindowFrameType::CornerRT:    return tg::WindowFrame_RightTop;
    case sch::WindowFrameType::CornerLB:    return tg::WindowFrame_LeftBottom;
    case sch::WindowFrameType::CornerRB:    return tg::WindowFrame_RightBottom;
    case sch::WindowFrameType::FrameL:      return tg::WindowFrame_Left;
    case sch::WindowFrameType::FrameR:      return tg::WindowFrame_Right;
    case sch::WindowFrameType::FrameT:      return tg::WindowFrame_Top;
    case sch::WindowFrameType::FrameB:      return tg::WindowFrame_Bottom;
    default:
        throw gcnew LayoutDataException(String::Format("invalid frame type. - {0}", frameType), nullptr);
    }
}

void
BinaryLytWriter::WriteBlock(sch::WindowFrame^ frame, uint16_t materialIdx)
{
    tg::ResWindowFrame resFrame = { 0 };

    resFrame.textureFlip = GetTextureFlip(frame->textureFlip[0]);
    resFrame.materialIdx = materialIdx;
    WriteStructData(resFrame);
}

void BinaryLytWriter::WriteBlock(sch::Bounding^ bounding, sch::Pane^ pane)
{
    tg::ResBounding resBounding;
    std::memset(&resBounding, 0, sizeof(resBounding));
    SetLayoutBlockPane(&resBounding, pane, isEnableViewerHidden_);
    WriteBlockHeader(&resBounding, tg::DataBlockKindBounding);
}

void BinaryLytWriter::WriteBlock(sch::Capture^ capture, sch::Pane^ pane, uint32_t blockSize)
{
    tg::ResCapture resCapture;
    std::memset(&resCapture, 0, sizeof(resCapture));
    SetLayoutBlockPane(&resCapture, pane, isEnableViewerHidden_);
    WriteBlockHeader(&resCapture, tg::DataBlockKindCapture, blockSize);
}

void BinaryLytWriter::WriteBlock(sch::Alignment^ alignment, sch::Pane^ pane)
{
    // 整列（水平）と整列（垂直）の両方が同時に設定される事はない、また両方未設定にされる事もない。
    Debug::Assert(
        (alignment->isAlignmentHorizontalEnabled && !alignment->isAlignmentVerticalEnabled) ||
        (!alignment->isAlignmentHorizontalEnabled && alignment->isAlignmentVerticalEnabled)
    );

    tg::ResAlignment resAlignment;
    std::memset(&resAlignment, 0, sizeof(resAlignment));
    uint8_t flag = 0;                   // 0:整列（水平）
    if(alignment->isAlignmentHorizontalEnabled)
    {
        resAlignment.horizontalAlignment = (int)alignment->horizontal;
        resAlignment.defaultMergine = alignment->defaultMargin;
        resAlignment.isExtendEdgeEnabled = alignment->isExtendEdgeEnabled;
    }
    else if(alignment->isAlignmentVerticalEnabled)
    {
        flag |= (uint8_t)(1 << 0);      // 1:整列（垂直）
        resAlignment.horizontalAlignment = (int)alignment->vertical;
        resAlignment.defaultMergine = alignment->defaultVerticalMargin;
        resAlignment.isExtendEdgeEnabled = alignment->isExtendEdgeVerticalEnabled;
    }
    resAlignment.flag = flag;
    SetLayoutBlockPane(&resAlignment, pane, isEnableViewerHidden_);
    WriteBlockHeader(&resAlignment, tg::DataBlockKindAlignment);
}

void BinaryLytWriter::WriteBlock(sch::Scissor^ scissor, sch::Pane^ pane)
{
    tg::ResScissor resScissor;

    std::memset(&resScissor, 0, sizeof(resScissor));
    SetLayoutBlockPane(&resScissor, pane, isEnableViewerHidden_);
    WriteBlockHeader(&resScissor, tg::DataBlockKindScissor);
}

//----------------------------------------------------------
void BinaryLytWriter::WriteStateStringBlock(System::String^ value)
{
    char temp[tg::CountOfStateMachineName];
    if(value != nullptr)
    {
        CopyNameStr(temp, value, sizeof(temp));
    }
    else
    {
        std::memset(temp, 0, sizeof(temp));
    }

    WriteStructData(temp);
}

//----------------------------------------------------------
void BinaryLytWriter::WriteStateBlock(Schema::Flyt::State^ state, WriterUtil^ util, BinaryLytWriter^ writer)
{
    writer->WriteStateStringBlock(state->name);

    int paramaterPos = (int)writer->Stream->Position;
    writer->WriteS64(0);
    writer->WriteU16(state->featureParamaters != nullptr ? state->featureParamaters->Length : 0);
    writer->WriteU16(0);
    writer->WriteU32(0);

    if (state->featureParamaters == nullptr)
    {
        return;
    }

    System::IO::MemoryStream^ ms = gcnew System::IO::MemoryStream();
    BinaryLytWriter^ temp = gcnew BinaryLytWriter(ms, this->isDegamma_, this->isEnableViewerHidden_);
    {
        for each(Schema::Flyt::StateFeatureParamater^ paramater in state->featureParamaters)
        {
            temp->WriteStateStringBlock(paramater->targetName);
            temp->WriteU32((uint32_t)WriterUtil::ConvertBackFlagEnumToArray<Schema::Flyt::FeatureParamaterKind>(paramater->kind));

            temp->WriteF32(paramater->x);
            temp->WriteF32(paramater->y);
            temp->WriteF32(paramater->z);
            temp->WriteF32(paramater->w);

            temp->WriteU32(0);
        }
        util->AddOffset(Schema::Flyt::StateFeatureParamater::typeid, writer->stream_, paramaterPos, ms);
    }
}

//----------------------------------------------------------
void BinaryLytWriter::WriteStateTransitionTrackKeyBlock(Schema::Flyt::StateTransitionTrackKey^ key, WriterUtil^ util, BinaryLytWriter^ writer)
{
    // ResStateTransitionTrackKey
    writer->WriteF32(key->offset);
    writer->WriteF32(key->easing->paramater);
    writer->WriteU8((uint8_t)key->easing->type);
    writer->WriteU8(0);
    writer->WriteU16(0);

    // ResStateParamaterStore
    writer->WriteStateStringBlock(key->featureParamater->targetName);
    writer->WriteU32((uint32_t)WriterUtil::ConvertBackFlagEnumToArray<Schema::Flyt::FeatureParamaterKind>(key->featureParamater->kind));
    writer->WriteF32(key->featureParamater->x);
    writer->WriteF32(key->featureParamater->y);
    writer->WriteF32(key->featureParamater->z);
    writer->WriteF32(key->featureParamater->w);
    writer->WriteU32(0);

    // ResStateTransitionTrackKeyEvent
    writer->WriteF32(key->stateMachineEvent->delay);
    writer->WriteU8((uint8_t)key->stateMachineEvent->kind);
    writer->WriteU8(0);
    writer->WriteU16(0);

    writer->WriteStateStringBlock(key->stateMachineEvent->param1);
    writer->WriteStateStringBlock(key->stateMachineEvent->param2);
}

//----------------------------------------------------------
void BinaryLytWriter::WriteStateTransitionTrackBlock(Schema::Flyt::StateTransitionTrack^ track, WriterUtil^ util, BinaryLytWriter^ writer)
{
    writer->WriteStateStringBlock(track->targetName);

    writer->WriteF32(track->offset);
    writer->WriteF32(track->duration);
    writer->WriteF32(track->easing->paramater);

    writer->WriteU8((uint8_t)track->easing->type);

    writer->WriteU8((uint8_t)WriterUtil::ConvertBackFlagEnumToArray<Schema::Flyt::FeatureParamaterKind>(track->kind));
    writer->WriteU16(0);

    int keyPos = (int)writer->Stream->Position;
    writer->WriteS64(0);
    writer->WriteU16(track->keies != nullptr ? track->keies->Length : 0);
    writer->WriteU16(0);
    writer->WriteU32(0);

    if (track->keies != nullptr)
    {
        //tg::ResStateTransitionTrackKey
        System::IO::MemoryStream^ ms = gcnew System::IO::MemoryStream();
        BinaryLytWriter^ temp = gcnew BinaryLytWriter(ms, this->isDegamma_, this->isEnableViewerHidden_);

        for each(Schema::Flyt::StateTransitionTrackKey^ key in track->keies)
        {
            WriteStateTransitionTrackKeyBlock(key, util, temp);
        }
        util->AddOffset(Schema::Flyt::StateTransitionTrackKey::typeid, writer->stream_, keyPos, ms);
    }
}

//----------------------------------------------------------
void BinaryLytWriter::WriteStateTransitionBlock(Schema::Flyt::StateTransition^ transition, WriterUtil^ util, BinaryLytWriter^ writer)
{
    // tg::ResStateTransition
    writer->WriteF32(transition->offset);
    writer->WriteF32(0.0f);
    writer->WriteF32(transition->totalDuration);
    writer->WriteF32(transition->easing->paramater);
    writer->WriteU8(transition->isCancelable);
    writer->WriteU8(transition->isLoop);
    writer->WriteU8((uint8_t)transition->easing->type);
    writer->WriteU8(0);

    writer->WriteU32(0);

    writer->WriteStateStringBlock(transition->startStateName);
    writer->WriteStateStringBlock(transition->endStateName);

    int trackPos = (int)writer->Stream->Position;
    writer->WriteS64(0);
    writer->WriteU16(transition->tracks != nullptr ? transition->tracks->Length : 0);
    writer->WriteU16(0);
    writer->WriteU32(0);

    // tg::ResStateTransitionTrack
    if (transition->tracks != nullptr)
    {
        System::IO::MemoryStream^ ms = gcnew System::IO::MemoryStream();
        BinaryLytWriter^ temp = gcnew BinaryLytWriter(ms, this->isDegamma_, this->isEnableViewerHidden_);

        for each(Schema::Flyt::StateTransitionTrack^ track in transition->tracks)
        {
            WriteStateTransitionTrackBlock(track, util, temp);
        }
        util->AddOffset(Schema::Flyt::StateTransitionTrack::typeid, writer->stream_, trackPos, ms);
    }

    //tg::ResStateTransitionTrigger
    Schema::Flyt::StateTransitionTrigger^ trigger = transition->trigger;
    writer->WriteS32((int)trigger->kind);
    writer->WriteS32((int)trigger->stateMachineOperator);

    int variablePos = (int)writer->Stream->Position;
    writer->WriteS64(0);
    writer->WriteU16(trigger->stateMachineVariablesSet != nullptr ? trigger->stateMachineVariablesSet->Length : 0);
    writer->WriteU16(0);
    writer->WriteU32(0);

    if (trigger->stateMachineVariablesSet != nullptr)
    {
        System::IO::MemoryStream^ ms = gcnew System::IO::MemoryStream();
        BinaryLytWriter^ temp = gcnew BinaryLytWriter(ms, this->isDegamma_, this->isEnableViewerHidden_);

        for each(Schema::Flyt::StateVariables^ variables in trigger->stateMachineVariablesSet)
        {
            WriteStateVariablesBlock(variables, util, temp);
        }
        util->AddOffset(Schema::Flyt::StateVariables::typeid, writer->stream_, variablePos, ms);
    }
}

//----------------------------------------------------------
void BinaryLytWriter::WriteBlockStateLayer(Schema::Flyt::StateLayer^ layer, WriterUtil^ util, BinaryLytWriter^ writer)
{
    writer->WriteStateStringBlock(layer->name);
    writer->WriteStateStringBlock(layer->name);

    writer->WriteS32(0);
    writer->WriteS32(-1);

    int statePos = (int)writer->Stream->Position;
    writer->WriteS64(0);

    int transitionPos = (int)writer->Stream->Position;
    writer->WriteS64(0);

    writer->WriteU16(layer->states->Length);
    writer->WriteU16(layer->transitions != nullptr ? layer->transitions->Length : 0);

    // tg::ResState stateBlock;
    {
        System::IO::MemoryStream^ ms = gcnew System::IO::MemoryStream();
        BinaryLytWriter^ temp = gcnew BinaryLytWriter(ms, this->isDegamma_, this->isEnableViewerHidden_);

        for each(Schema::Flyt::State^ state in layer->states)
        {
            WriteStateBlock(state, util, temp);
        }
        util->AddOffset(Schema::Flyt::State::typeid, writer->stream_, statePos, ms);
    }

    if (layer->transitions != nullptr)
    {
        System::IO::MemoryStream^ ms = gcnew System::IO::MemoryStream();
        BinaryLytWriter^ temp = gcnew BinaryLytWriter(ms, this->isDegamma_, this->isEnableViewerHidden_);

        // tg::ResStateTransition transitionBlock;
        for each(Schema::Flyt::StateTransition^ transition in layer->transitions)
        {
            WriteStateTransitionBlock(transition, util, temp);
        }
        util->AddOffset(Schema::Flyt::StateTransition::typeid, writer->stream_, transitionPos, ms);
    }
}

//----------------------------------------------------------
void BinaryLytWriter::WriteStateVariablesBlock(Schema::Flyt::StateVariables^ variables, WriterUtil^ util, BinaryLytWriter^ writer)
{
    writer->WriteStateStringBlock(variables->name);

    writer->WriteS32((int)variables->type);
    writer->WriteF32(variables->value);
    writer->WriteF32(variables->value);
    writer->WriteF32(variables->value);
    writer->WriteF32(variables->value);

    writer->WriteU32(0);
}

//----------------------------------------------------------
void BinaryLytWriter::WriteBlockStateMachine(Schema::Flyt::StateMachine^ stateMachine)
{
    WriterUtil^ util = gcnew WriterUtil((int)this->Stream->Position);
    const size_t blockHeadFilePos = this->Stream->Position;

    // tg::ResStateMachine
    {
        // 一度ダミーを書いておく。最後に正しい値で更新します。
        {
            tg::ResStateMachine stateBlock;
            WriteStructData(stateBlock.blockHeader);
        }

        WriteStateStringBlock("StateName_Test");

        int layerPos = (int)this->Stream->Position;
        WriteS64(0);

        int variablesPos = (int)this->Stream->Position;
        WriteS64(0);

        WriteU16(stateMachine->layers->Length);
        WriteU16(stateMachine->variables != nullptr ? stateMachine->variables->Length : 0);
        WriteU16(0);
        WriteU16(0);

        // tg::ResStateLayer layerBlock;
        {
            System::IO::MemoryStream^ ms = gcnew System::IO::MemoryStream();
            BinaryLytWriter^ temp = gcnew BinaryLytWriter(ms, this->isDegamma_, this->isEnableViewerHidden_);

            for each(Schema::Flyt::StateLayer^ layer in stateMachine->layers)
            {
                WriteBlockStateLayer(layer, util, temp);
            }
            util->AddOffset(Schema::Flyt::StateLayer::typeid, this->stream_, layerPos, ms);
        }

        {
            // tg::ResStateVariable
            if (stateMachine->variables != nullptr)
            {
                System::IO::MemoryStream^ ms = gcnew System::IO::MemoryStream();
                BinaryLytWriter^ temp = gcnew BinaryLytWriter(ms, this->isDegamma_, this->isEnableViewerHidden_);

                for each(Schema::Flyt::StateVariables^ variables in stateMachine->variables)
                {
                    WriteStateVariablesBlock(variables, util, temp);
                }
                util->AddOffset(Schema::Flyt::StateVariables::typeid, this->stream_, variablesPos, ms);
            }
        }
    }

    // オフセットポインタを最終的な値で書き直す
    {
        // 最終的な書き込み位置を算出して記憶する
        int defferedWriteBasePos = (int)this->Stream->Position - util->FileBasePosition;
        for each(WriterUtil::RewriteOffsetValue^ value in util->RewriteOffsetValues)
        {
            value->FinalPosition = defferedWriteBasePos;
            defferedWriteBasePos += value->Content->ToArray()->Length;
        }

        // オフセットとして書き戻す
        for each(WriterUtil::RewriteOffsetValue^ value in util->RewriteOffsetValues)
        {
            int current = (int)value->Stream->Position;
            value->Stream->Position = value->Position;

            array<System::Byte>^ writeBuf = GetByteArray((int64_t)value->FinalPosition);
            value->Stream->Write(writeBuf, 0, writeBuf->Length);

            value->Stream->Position = current;
        }

        // 遅延書き込みを実際に行う
        for each(WriterUtil::RewriteOffsetValue^ value in util->RewriteOffsetValues)
        {
            WriteBytes(value->Content->ToArray());
        }
    }

    // 正しいブロックヘッダーに更新
    {
        const size_t blockTailFilePos = this->Stream->Position;
        const size_t blockSize = this->Stream->Position - blockHeadFilePos;

        this->Stream->Position = blockHeadFilePos;
        {
            tg::ResStateMachine stateBlock;
            std::memset(&stateBlock, 0, sizeof(stateBlock));
            InitBinaryBlockHeader(&stateBlock.blockHeader, tg::DataBlockKindStateMachine, (uint32_t)blockSize);
            WriteStructData(stateBlock.blockHeader);
        }
        this->Stream->Position = blockTailFilePos;
    }
}

//----------------------------------------------------------
void BinaryLytWriter::WriteBlock(sch::Parts^ parts, sch::Pane^ pane, uint32_t blockSize)
{
    tg::ResParts partsBlock;
    std::memset(&partsBlock, 0, sizeof(partsBlock));
    SetLayoutBlockPane(&partsBlock, pane, isEnableViewerHidden_);
    partsBlock.propertyCount = parts->property ? parts->property->Length : 0;
    if (parts->magnify)
    {
        partsBlock.magnify.x = parts->magnify->x;
        partsBlock.magnify.y = parts->magnify->y;
    }
    else
    {
        partsBlock.magnify.x = 1.f;
        partsBlock.magnify.y = 1.f;
    }
    WriteBlockHeader(&partsBlock, tg::DataBlockKindParts, blockSize);
}

void
BinaryLytWriter::WritePartsPropertyStruct(sch::Property^ prop, uint32_t propertyOffset, uint32_t userDataOffset, uint32_t paneBasicInfoOffset)
{
    tg::ResPartsProperty resPartsProperty;
    std::memset(&resPartsProperty, 0, sizeof(resPartsProperty));
    CopyNameStr(resPartsProperty.name, prop->target, sizeof(resPartsProperty.name));
    resPartsProperty.propertyOffset = propertyOffset;
    if (prop->userData != nullptr && prop->userData->Length == 0)
    {
        // 拡張ユーザーデータリストがあり、かつ長さ0ということは、「拡張ユーザーデータがない」と
        // 上書きしていることを示す。この場合、特別な値に設定する。
        resPartsProperty.extUserDataOffset = tg::ExtUserDataOverrideOffset_NoData;
    }
    else
    {
        // それ以外はuserDataOffsetの値を入れる。
        // userDataOffsetが0の場合は、EXTUSERDATAOVERRIDEOFFSET_NO_OVERRIDEになり、
        // 上書きを行わないという意味になる。
        resPartsProperty.extUserDataOffset = userDataOffset;
    }
    resPartsProperty.paneBasicInfoOffset = paneBasicInfoOffset;
    // usageFlagを求める、この部分の解釈はペインの種類によって異なる
    uint8_t flag = 0;
    if (prop->usageOptions)
    {
        for (int i = 0; i < prop->usageOptions->Length; i++)
        {
            if (prop->usageOptions[i] == sch::PartsPropertyUsageOptions::UseTextBoxText)
            {
                flag |= 1 << tg::TextBoxOverrideUsageFlag_TextEnabled;
            }
        }
    }
    resPartsProperty.usageFlag = flag;
    // マテリアルの部分上書きのフラグを求める
    flag = 0;
    if (prop->usageOptions)
    {
        for (int i = 0; i < prop->usageOptions->Length; i++)
        {
            if (prop->usageOptions[i] == sch::PartsPropertyUsageOptions::UseMaterialColorBlend)
            {
                flag |= 1 << tg::MaterialOverrideUsageFlag_InterpolateColorEnabled;
            }
            if (prop->usageOptions[i] == sch::PartsPropertyUsageOptions::UseMaterialTexture)
            {
                flag |= 1 << tg::MaterialOverrideUsageFlag_TextureEnabled;
            }
        }
    }

#if 0
    if (prop->kind == Schema::Flyt::PaneKind::Picture)
    {
        Schema::Flyt::Picture^ pict = dynamic_cast<Schema::Flyt::Picture^>(prop->Item);
        pict->material->texMap->
    }
#endif

    resPartsProperty.materialUsageFlag = flag;
    // 基本情報の部分上書きのフラグを求める
    flag = 0;
    if (prop->visible.HasValue)
    {
        flag |= 1 << tg::BasicOverrideUsageFlag_VisibleEnabled;
        if (prop->visible.Value)
        {
            flag |= 1 << tg::BasicOverrideUsageFlag_VisibleValue;
        }
    }
    if (prop->basicUserData)
    {
        flag |= 1 << tg::BasicOverrideUsageFlag_BasicUserDataEnabled;
    }
    if (prop->translate != nullptr)
    {
        flag |= 1 << tg::BasicOverrideUsageFlag_TranslateEnabled;
    }
    if (prop->size != nullptr)
    {
        flag |= 1 << tg::BasicOverrideUsageFlag_SizeEnabled;
    }
    if (prop->scale != nullptr)
    {
        flag |= 1 << tg::BasicOverrideUsageFlag_ScaleEnabled;
    }
    if (prop->rotate != nullptr)
    {
        flag |= 1 << tg::BasicOverrideUsageFlag_RotateEnabled;
    }
    if (prop->alpha.HasValue)
    {
        flag |= 1 << tg::BasicOverrideUsageFlag_AlphaEnabled;
    }
    resPartsProperty.basicUsageFlag = flag;

    WriteStructData(resPartsProperty);
}

void
BinaryLytWriter::WritePartsPaneBasicInfoStruct(Schema::Flyt::Property^ prop)
{
    tg::ResPartsPaneBasicInfo resPartsPaneBasicInfo;
    std::memset(&resPartsPaneBasicInfo, 0, sizeof(resPartsPaneBasicInfo));

    if (prop->basicUserData)
    {
        CopyNameStr(resPartsPaneBasicInfo.userData, prop->basicUserData, sizeof(resPartsPaneBasicInfo.userData));
    }
    if (prop->translate != nullptr)
    {
        CopyVec3(&resPartsPaneBasicInfo.translate, prop->translate);
    }
    if (prop->size != nullptr)
    {
        CopyVec2(&resPartsPaneBasicInfo.size, prop->size);
    }
    if (prop->scale != nullptr)
    {
        CopyVec2(&resPartsPaneBasicInfo.scale, prop->scale);
    }
    if (prop->rotate != nullptr)
    {
        CopyVec3(&resPartsPaneBasicInfo.rotate, prop->rotate);
    }
    if (prop->alpha.HasValue)
    {
        resPartsPaneBasicInfo.alpha = prop->alpha.Value;
    }

    WriteStructData(resPartsPaneBasicInfo);
}

void
BinaryLytWriter::WriteStruct(sch::Group^ group)
{
    const uint16_t paneNum = group->paneRef ? group->paneRef->Length: 0;

    tg::ResGroup resGroup;
    std::memset(&resGroup, 0, sizeof(resGroup));
    CopyNameStr(resGroup.name, group->name, sizeof(resGroup.name));
    resGroup.paneCount = paneNum;
    WriteBlockHeader(&resGroup, tg::DataBlockKindGroup, sizeof(resGroup) + tg::ResourceNameStrMax * paneNum);
}

void
BinaryLytWriter::WriteStruct(sch::GroupPaneRef^ paneRef)
{
    char paneName[tg::ResourceNameStrMax];
    CopyNameStr(paneName, paneRef->name, sizeof(paneName));
    WriteStructData(paneName);
    Debug::WriteLine(String::Format(L"Group pane {0}", paneRef->name));
}

void
BinaryLytWriter::WriteCombinerUserShader(Schema::Flyt::MaterialInfo^ src)
{
    tg::ResCombinerUserShader resCombinerUserShader;

    const int individualVariationCodeLength = tg::DetailedCombinerStageBitsCountWithVariationTable * sizeof(uint32_t) + 1; // 個別バリエーション用のコード分の容量 ＋ '\0'
    char strIndividualVariationCode[individualVariationCodeLength];
    memset(strIndividualVariationCode, 0, sizeof(strIndividualVariationCode));
    CopyNameStr(strIndividualVariationCode, src->combinerUserShader->GetFileNameWithoutExtension(), sizeof(strIndividualVariationCode));
    resCombinerUserShader.SetKeyCode((uint32_t*)strIndividualVariationCode, strlen(strIndividualVariationCode));

    if (src->konstColors && src->konstColors->Length > 1)
    {
        for (int i = 0; i < std::min(tg::CombinerUserShaderConstantColor, src->konstColors->Length - 1); i++)
        {
            tg::ResColor color;
            sch::Color4  ConstColor;
            ConstColor.r = static_cast<uint8_t>(Math::Min(src->konstColors[i + 1]->r * 255.0f, 255.0f));
            ConstColor.g = static_cast<uint8_t>(Math::Min(src->konstColors[i + 1]->g * 255.0f, 255.0f));
            ConstColor.b = static_cast<uint8_t>(Math::Min(src->konstColors[i + 1]->b * 255.0f, 255.0f));
            ConstColor.a = static_cast<uint8_t>(Math::Min(src->konstColors[i + 1]->a * 255.0f, 255.0f));
            CopyColor(&color, %ConstColor);
            resCombinerUserShader.SetConstantColor(i, color);
        }
    }

    WriteStructData(resCombinerUserShader);
}

void
BinaryLytWriter::CopyColor(tg::ResColor* pDst, sch::Color4^ src)
{
    if(src != nullptr)
    {
        if (isDegamma_)
        {
            // ガンマ特性を与える、0～255になっていることに注意が必要
            pDst->r = static_cast<uint8_t>(DegammaValue(src->r));
            pDst->g = static_cast<uint8_t>(DegammaValue(src->g));
            pDst->b = static_cast<uint8_t>(DegammaValue(src->b));
            pDst->a = src->a;
        }
        else
        {
            pDst->r = src->r;
            pDst->g = src->g;
            pDst->b = src->b;
            pDst->a = src->a;
        }
    }
}

bool
BinaryLytWriter::WriteColor4f(Schema::Flyt::Color4f^ src)
{
    if (src->byte_convert)
    {
        uint8_t r = static_cast<uint8_t>(Math::Min(src->r * 255.0f, 255.0f));
        uint8_t g = static_cast<uint8_t>(Math::Min(src->g * 255.0f, 255.0f));
        uint8_t b = static_cast<uint8_t>(Math::Min(src->b * 255.0f, 255.0f));
        uint8_t a = static_cast<uint8_t>(Math::Min(src->a * 255.0f, 255.0f));

        if (isDegamma_)
        {
            r = static_cast<uint8_t>(DegammaValue(r));
            g = static_cast<uint8_t>(DegammaValue(g));
            b = static_cast<uint8_t>(DegammaValue(b));
        }

        WriteU8(r);
        WriteU8(g);
        WriteU8(b);
        WriteU8(a);

        return false;
    }
    else
    {
        WriteF32(src->r);
        WriteF32(src->g);
        WriteF32(src->b);
        WriteF32(src->a);

        return true;
    }
}


}   // namespace LayoutBinaryConverter
}   // namespace NW4F



