﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <string>
#include <vector>

namespace sharc {
//------------------------------------------------------------------------------
class ArchiveConfig;
//------------------------------------------------------------------------------
/**
 *  アーカイブを作成します
 */
class ArchiveBuilder
{
public:
    /// コンストラクタ
    ArchiveBuilder();
    /// デストラクタ
    virtual ~ArchiveBuilder();

public:
    /**
     *  初期化を行います
     *
     *  @param heap         作業用のヒープ
     *  @param entry_num    アーカイブに含めるファイルの最大数
     */
    void initialize(uint32_t entry_num );
    /**
     *  アーカイブに含めるファイルをリストに追加します
     *
     *  @param path_win     Windows上でのファイルパス
     *  @param path_arc     アーカイブ内での仮想パス
     *  @param align        アライメント
     */
    void addEntry( const std::string& path_win, const std::string& path_arc, uint32_t align );
    /**
     *  アーカイブを生成します
     *
     *  @param dst          アーカイブを書き込むファイルのハンドル
     *  @param config       アーカイブ設定
     *  @param key          ハッシュキー
     *
     *  @return 成功すればtrue
     */
    bool build( FILE* dst, const ArchiveConfig& config, uint32_t key );

private:
    /// アーカイブに登録されるファイルのエントリ情報です
    struct Entry
    {
        std::string     path_win;           ///< Windows上でのファイルパス
        std::string     path_arc;           ///< アーカイブ内でのファイルパス
        uint32_t                     align;              ///< アライメント
        uint32_t                     hash;               ///< ハッシュ値
        uint32_t                     name_offset;        ///< 名前が格納されている位置
        uint32_t                     data_offset_start;  ///< データが格納されている開始位置
        uint32_t                     data_offset_end;    ///< データが格納されている終端位置
        uint32_t                     same_id;            ///< このエントリと同じハッシュ値をもつエントリがリストの前にいくつあるか
        /// デストラクタ
        virtual ~Entry()
        {
        };
    };

private:
    typedef std::vector<Entry*> EntryList;
private:
    uint32_t                         mEntryNum;      ///< アーカイブに登録するファイル数
    EntryList                   mEntryList;     ///< アーカイブに追加するファイルのエントリリスト
    Entry*                      mEntryItems;  /// エントリの実体の列

private:
    /// ファイルパスからハッシュ値を計算します
    void calcHash_(uint32_t key);
    /// 実際にアーカイブしたファイルの情報を標準出力へ出力します
    void displayFileList_();
    /// ハッシュ値でエントリリストをヒープソートするための比較関数
    static int entryCompare_( const Entry* lhs, const Entry* rhs );
};

//------------------------------------------------------------------------------
} // namespace sharc
