﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <iostream>
#include <fstream>
#include <algorithm>
#include <filesystem>
#include <shlwapi.h>
#include <locale.h>
#include <ctype.h>

#include "sharcArchive.h"
#include "sharcArchiveConfig.h"
#include "sharcArchiveBuilder.h"
#include "sharcArchiveReader.h"
#include "sharcDebug.h"

using namespace sharc;

namespace {
    // ヘルプメッセージ.
    const char* sHelp[] =
    {
        "",
        "Usage: LayoutArchiver [mode][options] archive [file or directory ...]",
        "",
        "mode",
        "  -c       : Creates a new archive. <default mode>",
        "  -l       : Displays a file list inside the archive or compare archives",
        "                  with -M option.",
        "  -r       : Resolve every files inside the archive",
        "",
        "options",
        "  -n       : Don't make whole FNT. unsafe but smaller and faster.",
        "  -h       : Displays this Help.(--help)",
        "  -v       : Displays progress during the execution for debug.(--verbose)",
        "  -s       : Set silent log-mode.(--silent)",
        "  -k       : Keep ftxb-files.(--keep-ftxb)",
        "  -p name  : *** OBSOLETE *** Set target platform type. e.g. \"Win\"(--platform)",
        "  -t name  : Set tile mode. e.g. \"Linear\"(--tile-mode)",
        "  -E type  : Set target endian type. \"big\" or \"little\". default=>big.",
        "  -K key   : Set hash key.",
        "  -A align : Set alignment[192 or 2 factorial(more than 4) ]. default=>16.",
        "  -D dir   : Specifies a working directory. default=>current directory.",
        "  -X dir   : Specifies a exclude directory.",
        "  -M arc   : Make a simple comparison between the two archive.",
        "  --shader-cache-path  : Set the shader cache path.",
        "  --api-type name      : Set the api type name. e.g. \"Gl\" default=>Gl",
        "  --code-type name     : Set the code type name. e.g. \"Source\" default=>Source",
        "  --tile-optimize name        : Set the tiling optimization. e.g. \"performance\" \"size\" \"size_auto\" use size_auto with tile-size-threshold option. default=>performance",
        "  --tile-size-threshold value : Set the threshold for size_auto option in the range[0, 100].",
        "",
        "",
        nullptr
    };

    void help()
    {
        for( int i=0; sHelp[i]; ++i ){
            printf("%s\n", sHelp[i]);
        }
        printf("build date      : %s %s\n", __DATE__, __TIME__);
    }
}

//---------------------------------------------------------------------------------

static std::wstring CalcTextureConverterPath()
{
    // 実行ファイルのフォルダを取得する。
    TCHAR temp[MAX_PATH];
    GetModuleFileName(NULL, temp, MAX_PATH);
    PathRemoveFileSpec(temp);

    // デバック実行時と通常実行時で参照パスを変えます。
    if(IsDebuggerPresent())
    {
        return std::wstring(temp) + L"\\..\\..\\..\\..\\..\\..\\..\\..\\..\\..\\Tools\\Graphics\\3dTools\\3dTextureConverter.exe";
    }else{
        return std::wstring(temp) + L"\\..\\3dTools\\3dTextureConverter.exe";
    }
}

//---------------------------------------------------------------------------------

static std::wstring CalcArchiveShaderCombiner()
{
    // 実行ファイルのフォルダを取得する。
    TCHAR temp[MAX_PATH];
    GetModuleFileName(NULL, temp, MAX_PATH);
    PathRemoveFileSpec(temp);

    // デバック実行時と通常実行時で参照パスを変えます。
    if (IsDebuggerPresent())
    {
        return std::wstring(temp) + L"\\..\\..\\..\\..\\..\\..\\..\\..\\..\\..\\Tools\\Graphics\\LayoutTools\\\\ArchiveShaderCombiner.exe";
    }
    else {
        return std::wstring(temp) + L"\\ArchiveShaderCombiner.exe";
    }
}

//---------------------------------------------------------------------------------

bool CompareWideChar_(wchar_t a, wchar_t b)
{
    return towlower(a) == towlower(b);
}

//---------------------------------------------------------------------------------

bool CompareIgnoreCase(std::wstring const& lhs, std::wstring const& rhs)
{
    if (lhs.length() == rhs.length())
    {
        return std::equal(rhs.begin(), rhs.end(), lhs.begin(), CompareWideChar_);
    }
    else {
        return false;
    }
}

//---------------------------------------------------------------------------------

static bool ConvertAllFtxbToOneBntx(std::list<std::wstring>*  pFtxbFiles, const char* pOutDirName, const char* pPlatformName, const char* pTileMode, const char* pTileOptimize, const char* pTileSizeThreshold, bool isLogSilent)
{
    // wstring に変換する。
    std::wstring outPathString;
    {
        setlocale( LC_CTYPE, "jpn" );

        wchar_t outPath[MAX_PATH];
        mbstowcs( outPath, pOutDirName, MAX_PATH);
        outPathString = std::wstring(outPath);
    }

    std::wstring imageFilePath = outPathString + L"timg";
    if(!std::tr2::sys::is_directory(std::tr2::sys::path(imageFilePath)))
    {
        return true;
    }

    for(std::tr2::sys::directory_iterator it(imageFilePath); it != std::tr2::sys::directory_iterator(); ++it)
    {
        // 注意: it->path() には、imageFilePath も含んだパスが入る
        std::tr2::sys::path p = it->path();
        auto s = std::tr2::sys::status(p);
        if (std::tr2::sys::is_regular_file(p))
        {
            if(p.extension() == L".ftxb")
            {
                pFtxbFiles->push_back(p.native());
            }
        }
    }

    // 結合すべき ftxb があるかどうか
    if(pFtxbFiles->size() <= 0)
    {
       return true;
    }

    if(!isLogSilent)
    {
        printf("Convert all ftxb files into one bntx.\n");
        fflush(stdout);
    }

    std::wstring platformNameString;
    {
        const size_t platformNameLength = 100;
        wchar_t platformName[platformNameLength];
        mbstowcs( platformName, pPlatformName, platformNameLength);
        platformNameString = std::wstring(platformName);
    }

    std::wstring tileModeString;
    {
        const size_t tileModeNameLength = 100;
        wchar_t tileModeName[tileModeNameLength];
        mbstowcs( tileModeName, pTileMode, tileModeNameLength);
        tileModeString = std::wstring(tileModeName);
    }

    std::wstring tileOptimizeString;
    {
        const size_t tileOptimizeNameLength = 100;
        wchar_t tileOptimizeName[tileOptimizeNameLength];
        mbstowcs(tileOptimizeName, pTileOptimize, tileOptimizeNameLength);
        tileOptimizeString = std::wstring(tileOptimizeName);
    }

    std::wstring tileSizeThresholdString;
    {
        const size_t tileSizeThresholdLength = 100;
        wchar_t tileSizeThresholdName[tileSizeThresholdLength];
        mbstowcs(tileSizeThresholdName, pTileSizeThreshold, tileSizeThresholdLength);
        tileSizeThresholdString = std::wstring(tileSizeThresholdName);
    }

    // プラットフォーム名が指定されている場合
    if(platformNameString.size() > 0)
    {
        // 警告を表示して、適切に tileModeString を設定
        fprintf(stderr, "----\nwarning.\n""LayoutArchiver: -p is obsolete.(please use -t(--tile-mode) option instead.)\n");

        if(CompareIgnoreCase(platformNameString, L"win") ||
           CompareIgnoreCase(platformNameString, L"win32") ||
           CompareIgnoreCase(platformNameString, L"Generic") )
        {
            tileModeString = L"linear";
        }else if(CompareIgnoreCase(platformNameString, L"cafe")){
            tileModeString = L"cafe";
        }else if(CompareIgnoreCase(platformNameString, L"nx")){
            tileModeString = L"nx";
        }else{
            tileModeString = L"unknown";
        }
    }
    else
    {
        if(tileModeString.size() <= 0)
        {
            fprintf(stderr, "----\nfailed.\n""LayoutArchiver: tile-mode is not specified.(please use -t(--tile-mode) option. e.g. \"-t Linear\")\n");
            return false;
        }
    }

    // 出力バイナリが変化しないように、念のためソートを行います。
    pFtxbFiles->sort(std::less<std::wstring>());

    // 引数ファイルのパスを計算します。
    std::wstring argFilePath;
    {
        TCHAR temp[MAX_PATH];
        GetTempPath(MAX_PATH, temp);

        GetTempFileName(temp, L"LayoutArchiver_", 0, temp);

        argFilePath = std::wstring(temp);
    }

    // 引数ファイルを書き出します。
    {
        std::wstring bntxFilePath = imageFilePath + L"/__Combined.bntx";

        std::wstring argContent;
        {
            std::list<std::wstring>::iterator it = pFtxbFiles->begin();
            while( it != pFtxbFiles->end() )
            {
                argContent.append(L"\"" + (*it) + L"\"" + L"\n");
                it++;
            }

            argContent += L"\n";
            argContent += L"--tile-mode=" + tileModeString;
            argContent += L"\n";


            if (tileOptimizeString.length() > 0)
            {
                argContent += L"--tile-optimize=" + tileOptimizeString;
                argContent += L"\n";
            }

            if (tileSizeThresholdString.length() > 0)
            {
                argContent += L"--tile-size-threshold=" + tileSizeThresholdString;
                argContent += L"\n";
            }

            if(isLogSilent)
            {
                argContent += L"--silent\n";
            }
            argContent += L"-o" + bntxFilePath;
        }

        std::wofstream outputfile(argFilePath);
        outputfile << argContent.c_str();
        outputfile.close();
    }

    // TextureConverter で bntx を出力します。
    {
        std::wstring converterPath = CalcTextureConverterPath();
        std::wstring arg = L" --args-file=" + argFilePath;

        PROCESS_INFORMATION pi = { 0 };

        STARTUPINFO si  = { sizeof(STARTUPINFO) };
        si.cb           = sizeof(STARTUPINFO);

        CreateProcess(converterPath.c_str(), (LPWSTR)arg.c_str(), NULL, NULL, FALSE, NORMAL_PRIORITY_CLASS, NULL, NULL, &si, &pi);
        WaitForSingleObject(pi.hProcess,INFINITE);

        DWORD exit_code;
        GetExitCodeProcess(pi.hProcess, &exit_code);
        CloseHandle(pi.hProcess);
        CloseHandle(pi.hThread);

        if(exit_code != 0)
        {
            fprintf(stderr, "Can't convert ftxb to bntx. code = %d", exit_code);
            return false;
        }

        DeleteFile(argFilePath.c_str());
    }

    return true;
}

//---------------------------------------------------------------------------------

int main(int argc, char** argv)
{
    const OptionName longOptions[] =
    {
        { "verbose", 'v' },
        { "help", 'h' },
        { "platform", 'p' },
        { "tile-mode", 't' },
        { "silent", 's' },
        { "keep-ftxb", 'k' },
        { "keep-shader-variation-xml", 'x' },
        { "shader-cache-path", 'P' },
        { "api-type", 'a' },
        { "code-type", 'C' },
        { "tile-optimize", 'O' },
        { "tile-size-threshold", 'T' },
    };

    // コマンドラインの解釈
    GetOpt opt( argc, const_cast<const char**>( argv ), "clnhvrskxp:t:E:K:A:D:S:X:M:P:a:C:O:T:", sizeof(longOptions)/sizeof(OptionName), longOptions);

    // アーカイブコンフィグ
    ArchiveConfig configInstance;
    ArchiveConfig* config = &configInstance;

    char c;
    int32_t align = cDefaultAlignment;
    uint32_t hash_key  = cDefaultHashKey;

    while( ( c = opt.next() ) != GetOpt::END )
    {
        switch( c )
        {
            case 'l':   config->setMode( ArchiveConfig::cMode_List );       break;
            case 'c':   config->setMode( ArchiveConfig::cMode_Create );     break;
            case 'r':   config->setMode( ArchiveConfig::cMode_Resolve );    break;
            case 'n':   config->setOption( cArchiveOption_Unsafe );         break;
            case 's':   config->setLogSilent( true );                       break;
            case 'k':
            {
                config->setKeepFtxb(true);
            } break;
            case 'x':
            {
                config->setKeepShaderVariationXml(true);
            } break;
            case 'p':
            {
                std::string arg_string = opt.arg();
                config->setPlatformsName(arg_string.c_str());
            } break;
            case 't':
            {
                std::string arg_string = opt.arg();
                config->setTileMode(arg_string.c_str());
            } break;
            case 'E':
            {
                std::string arg_string = opt.arg();
                if( arg_string == "little" )
                {
                    config->setEndianType( EndianTypes::cLittle );
                }
                else
                {
                    if( ! (arg_string == "big") )
                    {
                        fprintf(stderr, "%s is not accepted. current endian => [ big ]", arg_string.c_str() );
                    }
                    config->setEndianType( EndianTypes::cBig );
                }
            } break;
            case 'K':
            {
                {
                    char* end_ptr;
                    _set_errno(0);
                    hash_key = strtoul(opt.arg(), &end_ptr, 10);
                    if (*end_ptr != '\0' || errno == ERANGE || errno == EINVAL)
                    {
                        fprintf(stderr, " %s is not accepted. current hash key => [ %u ].\n", opt.arg(), hash_key);
                    }
                }
                config->setHashKey( hash_key );
            } break;
            case 'A':
            {
                {
                    char* end_ptr;
                    _set_errno(0);
                    align = strtol(opt.arg(), &end_ptr, 10);
                    if (*end_ptr != '\0' || errno == ERANGE || errno == EINVAL)
                    {
                        fprintf(stderr, " %s is not accepted. current alignment => [ %d ].\n", opt.arg(), align);
                    }
                }
                config->setAlignment( align );
            } break;
            case 'D':   config->setWorkDir( opt.arg() );        break;
            case 'v':   config->enableDisplayFileInfo();        break;
            case 'S':   config->addScript( opt.arg() );     break;
            case 'X':   config->addExcept( opt.arg() );     break;
            case 'M':   config->storeArcName( opt.arg() );  break;
            case 'P': // --shader-cache-path
                config->SetShaderConvertCachePath(opt.arg()); // opt.arg() の戻り値はアプリ終了時まで解放されない
                break;
            case 'a': // --api-type
                config->SetApiTypeName(opt.arg()); // opt.arg() の戻り値はアプリ終了時まで解放されない
                break;
            case 'C': // --code-type
                config->SetCodeTypeName(opt.arg()); // opt.arg() の戻り値はアプリ終了時まで解放されない
                break;
            case 'O': // --tile-optimize
                config->setTileOptimize(opt.arg());
                break;
            case 'T': // --tile-size-threshold
                config->setTileSizeThreshold(opt.arg());
                break;
            default:
                help();
                return -1;
        }
    }

    // コンフィグ設定でエラーが発生したので処理を中断します。
    if (config->hasError())
    {
        fprintf(stderr, "----\nfailed.\n""LayoutArchiver: Config has some error.\n");
        help();
        return -1;
    }

    // ftxb を bntx に統合します。
    // アーカイブ対象ファイルを列挙するまえに実行します。
    std::list<std::wstring> ftxbFiles = std::list<std::wstring>();

    // xml を bnsh に統合します。
    // アーカイブ対象ファイルを列挙するまえに実行します。
    if (config->getMode() == ArchiveConfig::cMode_Create)
    {
        STARTUPINFO startUpInfo;
        ZeroMemory(&startUpInfo, sizeof(startUpInfo));
        startUpInfo.cb = sizeof(startUpInfo);
        PROCESS_INFORMATION processInformation;
        ZeroMemory(&processInformation, sizeof(processInformation));
        // bgsh ディレクトリのパス
        TCHAR bnshDir[MAX_PATH];
        mbstowcs(bnshDir, config->getWorkDirString().c_str(), MAX_PATH);
        wcscat_s(bnshDir, L"bgsh");

        // apiTypeName と codeTypeName
        const size_t typeNameLength = 32;
        TCHAR apiTypeName[typeNameLength];
        mbstowcs(apiTypeName, config->GetApiTypeName(), typeNameLength);
        TCHAR codeTypeName[typeNameLength];
        mbstowcs(codeTypeName, config->GetCodeTypeName(), typeNameLength);

        TCHAR cachePathName[MAX_PATH];
        mbstowcs(cachePathName, config->GetShaderConvertCachePath(), MAX_PATH);

        // ArchiveShaderCombiner.exe のパス
        std::wstring combinerPath = CalcArchiveShaderCombiner();

        // コマンドライン
        TCHAR cmdLine[1024];
        TCHAR cmdLineShaderCache[1024];
        swprintf_s(cmdLineShaderCache, L" --shader-cache-path \"%s\"", cachePathName);
        swprintf_s(cmdLine, L"\"%s\" \"%s\" %s %s%s%s",
            combinerPath.c_str(),
            bnshDir,
            apiTypeName,
            codeTypeName,
            config->getKeepShaderVariationXml() ? L" --keep-shader-variation-xml" : L"",
            config->GetShaderConvertCachePath() ? cmdLineShaderCache : L"");

        // シェーダコンバイナ実行
        if (!CreateProcess(NULL, cmdLine, NULL, NULL, FALSE, 0, NULL, NULL, &startUpInfo, &processInformation))
        {
            fprintf(stderr, "----\nfailed.\n""LayoutArchiver: ArchiveShaderCombiner could not be run.\n");
            return -1;
        }

        // シェーダコンバイナの終了を待たずに bntx への統合を実行
        if (!ConvertAllFtxbToOneBntx(&ftxbFiles, config->getWorkDirString().c_str(), config->getPlatformsName().c_str(), config->getTileMode().c_str(), config->getTileOptimize().c_str(), config->getTileSizeThreshold().c_str(), config->getLogSilent()))
        {
            fprintf(stderr, "----\nfailed.\n""LayoutArchiver: Can not unite ftxb files into bntx.\n");
            WaitForSingleObject(processInformation.hProcess, INFINITE);
            CloseHandle(processInformation.hProcess);
            CloseHandle(processInformation.hThread);
            return -1;
        }

        WaitForSingleObject(processInformation.hProcess, INFINITE);
        CloseHandle(processInformation.hProcess);
        CloseHandle(processInformation.hThread);
    }

    int32_t num = opt.getNonOptionArgNum();
    if ( num < 1 )
    {
        help();
        return -1;
    }
    // 第一引数が対象ファイル名.
    const char* arc_name = opt.getNonOptionArg( 0 );
    PathString archive_name(arc_name);

    // 第二引数以降がファイル.
    for( int32_t i = 1; i < num; i++ )
    {
        config->addArchivePath( opt.getNonOptionArg( i ), config->getAlignment() );
    }

    switch( config->getMode() )
    {
        case ArchiveConfig::cMode_Create:
        {
            if( ! config->initialize() )
            {
                fprintf(stderr, "----\nfailed.\narchive configure is incorrect.\n");
                return -1;
            }

            if( config->isDisplayFileInfo() )
            {
                if( config->getEndianType() == EndianTypes::cBig )
                {
                    fprintf(stdout, "directory [ %s ]\nhash key  [ %u ], endian [ big ], data block align [ %u ]\n----\n",
                        config->getWorkDirString().c_str(), config->getHashKey(), config->getMaxAlignment());
                }
                else
                {
                    fprintf(stdout, "directory [ %s ]\nhash key  [ %u ], endian [ little ], data block align [ %u ]\n----\n",
                        config->getWorkDirString().c_str(), config->getHashKey(), config->getMaxAlignment());
                }

                if( config->getOption() & cArchiveOption_Unsafe )
                {
                    fprintf(stdout, "unsafe option is enabled.\n");
                }
            }
            ArchiveBuilder builder;
            uint32_t entry_num = config->getPathList().size();
            if( entry_num > 0 )
            {
                if (checkExistenceExt(arc_name))
                {
                    // 拡張子があるときは、そのままコピーする
                    archive_name = arc_name;
                }
                else
                {
                    // 拡張子がないときは自動で .arc を付与
                    archive_name = PathString(arc_name) + PathString(".arc");
                }

                builder.initialize( entry_num );
                const ArchiveConfig::PathList& list = config->getPathList();
                ArchiveConfig::PathList::const_iterator end = list.end();
                for( ArchiveConfig::PathList::const_iterator itr = list.begin(); itr != end; ++itr ){
                    // ftxb ファイルはアーカイブに含めません。
                    if (itr->path_win.rfind(".ftxb") != std::string::npos)
                    {
                        continue;
                    }

                    builder.addEntry( itr->path_win, itr->path_arc, itr->align );
                }
                FILE* handle;
                {
                    // オープンする前にいったん削除する。東京制作部より、アーカイブを作成する際に
                    // 書き込み先にファイルがあるとアーカイブ作成に時間がかかるという指摘があったため。
                    DeleteFileA( archive_name.c_str() );
                }
                errno_t res_code = fopen_s(&handle, archive_name.c_str(), "wb");
                // 作成しようとしたアーカイブが書き込み禁止などで開けないとき
                if( res_code != 0 ){
                    fprintf(stderr, "----\nfailed.\narchive building was not complete. %s is not able to open.\n", archive_name.c_str());
                    return -1;
                }

                bool ret = builder.build( handle, *config, config->getHashKey() );
                fclose(handle);
                if( ! ret ) // アーカイブのビルド中にエラーになったとき
                {
                    // 中途半端にできたアーカイブの削除を試みる
                    DeleteFileA( archive_name.c_str() );
                    fprintf(stderr, "----\nfailed.\narchive building was not complete.\n");
                    return -1;
                }

                if(!config->getLogSilent())
                {
                    printf("----\nsucceed.\n[ %s ]\n", archive_name.c_str() );
                }

                // ftxb ファイルを削除します。
                if(!config->getKeepFtxb())
                {
                    std::list<std::wstring>::iterator it = ftxbFiles.begin();
                    while( it != ftxbFiles.end() )
                    {
                        DeleteFile(it->c_str());
                        it++;
                    }
                }

            } else {
                fprintf(stderr, "no file to include in archive.\n");
                return -1;
            }
        } break;
        case ArchiveConfig::cMode_List:
        {
            if( config->getArcName().empty() ){
                // M オプションなし > リスト表示
                ArchiveReader reader;
                reader.setLogSilent(config->getLogSilent());

                if( ! reader.extractFileList( arc_name ) )
                {
                    fprintf(stderr, "----\nfailed.\ncan't extract file name from the archive.\n");
                    return -1;
                }
            } else {
                // M オプションあり > 比較
                ArchiveReader reader_a, reader_b;
                reader_a.setLogSilent(config->getLogSilent());
                reader_b.setLogSilent(config->getLogSilent());

                reader_a.extractCommon( arc_name );
                reader_b.extractCommon( config->getArcName() );
                if( reader_a.compare( reader_b ) ){

                    if(!config->getLogSilent())
                    {
                        printf("----\nmatched.\neach endian, hash, size and binary data are same.\n");
                    }

                } else {
                    return -1;
                }
            }
        } break;
        case ArchiveConfig::cMode_Resolve:
        {
            ArchiveReader reader;
            reader.setLogSilent(config->getLogSilent());

            if( reader.resolve( config->getWorkDirString(), arc_name ) ){

                if(!config->getLogSilent())
                {
                    printf("----\nsucceed.\n[ %s%s ]\n", config->getWorkDirString().c_str(), arc_name );
                }

            } else {
                return -1;
            }
        } break;
        default:
            NW_ERR( "not support mode." );
            return -1;
    }

    return 0;
}
