﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace FcpxMaker
{
    using DataModel;
    using Dialog;
    using FcpxMaker.Node;
    using Xml;
    using static Type;

    public partial class UserNodePanel : UserControl
        , IEventListener
    {
        private DataNode _savedNode = null;
        ToolStripItem _itemCopy = null;
        ToolStripItem _itemPasteChild = null;
        ToolStripItem _itemPasteNode = null;
        ToolStripItem _itemPasteProperty = null;

        /// <summary>
        /// ルートノードです。
        /// </summary>
        private DataNode RootNode
        {
            get
            {
                return _ctrlTreeView.Nodes[0] as DataNode;
            }
        }

        /// <summary>
        /// 選択中のノードです。
        /// </summary>
        private DataNode SelectedNode
        {
            get
            {
                return _ctrlTreeView.SelectedNode as DataNode;
            }
        }

        /// <summary>
        /// 保存実行時のノードの状態を保持します。
        /// </summary>
        private DataNode SavedNode
        {
            get
            {
                return _savedNode;
            }
            set
            {
                _savedNode = value.Clone() as DataNode;
            }
        }

        /// <summary>
        /// ノードに変更があったか。
        /// </summary>
        public bool IsModified
        {
            get
            {
                if (_ctrlTreeView.Nodes.Count > 0)
                {
                    return CheckModyfied(this.SavedNode, _ctrlTreeView.Nodes[0] as DataNode);
                }

                return false;
            }
        }

        /// <summary>
        /// ノードを持っているか。
        /// </summary>
        public bool HasNode
        {
            get { return _ctrlTreeView.Nodes.Count > 0; }
        }

        /// <summary>
        /// 追加ボタンが有効かどうか。
        /// </summary>
        private bool CanPushAddButton
        {
            get
            {
                if (SelectedNode != null)
                {
                    if ((SelectedNode.DataType == DataType.Root ||
                        SelectedNode.DataType == DataType.ScalableFont ||
                        SelectedNode.DataType == DataType.BitmapFont ||
                        SelectedNode.DataType == DataType.ScalableFontDescription) &&
                        SelectedNode.Nodes.Count == 0)
                    {
                        return true;
                    }

                    if (SelectedNode.DataType == DataType.PairFont &&
                        SelectedNode.Nodes.Count < 2)
                    {
                        return true;
                    }

                    if (SelectedNode.DataType == DataType.MultiScalableFont)
                    {
                        return true;
                    }
                }

                return false;
            }
        }

        /// <summary>
        /// 削除ボタンが有効かどうか。
        /// </summary>
        private bool CanPushRemoveButton
        {
            get
            {
                if (SelectedNode != null)
                {
                    if (SelectedNode.DataType != DataType.Root &&
                        SelectedNode.DataType != DataType.Font)
                    {
                        return true;
                    }
                }

                return false;
            }
        }

        /// <summary>
        /// ツリーにノードを追加します。
        /// </summary>
        private void AddTreeNode(DataNode node)
        {
            _ctrlTreeView.Nodes.Add(node);
            node.ExpandAll();
            _ctrlTreeView.SelectedNode = node;
        }

        /// <summary>
        /// 初期状態のツリーを作成します。
        /// </summary>
        private void CreateNewTree()
        {
            // Rootを追加する
            DataNode node = DataNodeFactory.CreateDataNode(DataModelFactory.CreateDataModel(DataType.Root));
            AddTreeNode(node);

            this.SavedNode = node;
        }

        /// <summary>
        /// ファイルからツリーを作成します。
        /// </summary>
        private void CreateTree(string filePath)
        {
            DataModelBase dataModel = XmlConverter.ReadXml(filePath);
            if (dataModel == null)
            {
                FileIOManager.FcpxOpenFilePath = null;
                return;
            }
            DataNode node = DataNodeFactory.CreateDataNode(dataModel);
            AddTreeNode(node);

            this.SavedNode = node;
        }

        /// <summary>
        /// ボタンを更新します。
        /// </summary>
        private void UpdateButton()
        {
            _btnAdd.Enabled = this.CanPushAddButton;
            _btnRemove.Enabled = this.CanPushRemoveButton;
        }

        /// <summary>
        /// コンテキストメニューを更新します。
        /// </summary>
        private void UpdateToolStripMenu()
        {
            _itemPasteChild.Enabled = this.CanPasteAsChild();
            _itemPasteNode.Enabled = this.CanPaste();
            _itemPasteProperty.Enabled = this.CanPaste();
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public UserNodePanel()
        {
            InitializeComponent();

            // ツリービューからフォーカスが外れても選択項目を強調表示する
            _ctrlTreeView.HideSelection = false;

            // ノード用のコンテキストメニューを作成する
            _itemCopy = DataNodeContextMenu.AddMenu(Properties.Resources.CONTEXTMENUITEM_COPY, Event_CopyNode);
            DataNodeContextMenu.AddMenu("-", null);
            _itemPasteChild = DataNodeContextMenu.AddMenu(Properties.Resources.CONTEXTMENUITEM_PASTECHILD, Event_PasteNodeChild);
            _itemPasteNode = DataNodeContextMenu.AddMenu(Properties.Resources.CONTEXTMENUITEM_PASTENODE, Event_PasteNodeTree);
            _itemPasteProperty = DataNodeContextMenu.AddMenu(Properties.Resources.CONTEXTMENUITEM_PASTEPROPERTY, Event_PasteNodeProperty);
        }

        /// <summary>
        /// 解放処理です。
        /// </summary>
        private void Clear()
        {
            _ctrlTreeView.Nodes.Clear();
            _ctrlTreeView.SelectedNode = null;
        }

        /// <summary>
        /// ツリーにノードを追加します。
        /// </summary>
        private void AddNode(DataNode parent, DataNode child)
        {
            // ルートの設定
            if (parent.DataModel is RootDataModel)
            {
                RootDataModel dataModel = parent.DataModel as RootDataModel;
                dataModel.SubFont = child.DataModel;
            }
            // ペアフォントの設定
            else if (parent.DataModel is PairFontDataModel)
            {
                PairFontDataModel dataModel = parent.DataModel as PairFontDataModel;
                bool isFirstFontSet = dataModel.IsFirstFontEmptyAndSecondFontUsed;

                dataModel.Add(child.DataModel);

                // フォント1が未使用でフォント2が使用中の場合
                if (isFirstFontSet)
                {
                    List<DataNode> tmp = new List<DataNode>();
                    tmp.Add(child);
                    tmp.AddRange(parent.Nodes.OfType<DataNode>());

                    parent.Nodes.Clear();
                    parent.Nodes.AddRange(tmp.ToArray());

                    parent.Expand();

                    return;
                }
            }
            // ビットマップフォントの設定
            else if (parent.DataModel is BitmapFontDataModel)
            {
                BitmapFontDataModel dataModel = parent.DataModel as BitmapFontDataModel;
                (child.DataModel as FontDataModel).FontType = FontFileType.Bitmap; // ツリーから追加した場合ここでFontTypeを設定する必要がある
                dataModel.SetFontDataModel(child.DataModel);

                // Fontノードの名前を変更する
                ModifyNode(child, child.DataModel);

                parent.Nodes.Clear();
            }
            // スケーラブルフォントの設定
            else if (parent.DataModel is ScalableFontDataModel)
            {
                ScalableFontDataModel dataModel = parent.DataModel as ScalableFontDataModel;
                dataModel.ScalableFontDescription = child.DataModel;
            }
            // マルチスケーラブルフォントの設定
            else if (parent.DataModel is MultiScalableFontDataModel)
            {
                MultiScalableFontDataModel dataModel = parent.DataModel as MultiScalableFontDataModel;
                dataModel.ScalableFontDescriptionSet.Add(child.DataModel);
            }
            // スケーラブルフォント設定の設定
            else if (parent.DataModel is ScalableFontDescriptionDataModel)
            {
                ScalableFontDescriptionDataModel dataModel = parent.DataModel as ScalableFontDescriptionDataModel;
                (child.DataModel as FontDataModel).FontType = FontFileType.Scalable; // ツリーから追加した場合ここでFontTypeを設定する必要がある
                dataModel.SetFontDataModel(child.DataModel);

                // Fontノードの名前を変更する
                ModifyNode(child, child.DataModel);

                parent.Nodes.Clear();
            }

            // 子ノードの追加
            {
                // ビットマップフォントの設定
                if (child.DataModel is BitmapFontDataModel)
                {
                    if ((child.DataModel as BitmapFontDataModel).FontDataModel == null)
                    {
                        // フォントも一緒に追加する
                        DataNode fontNode = CreateFontNode(FontFileType.Bitmap);
                        (child.DataModel as BitmapFontDataModel).SetFontDataModel(fontNode.DataModel);
                        child.Nodes.Add(fontNode);
                    }
                }
                // スケーラブルフォントの設定
                else if (child.DataModel is ScalableFontDataModel)
                {
                    if ((child.DataModel as ScalableFontDataModel).ScalableFontDescription == null)
                    {
                        // スケーラブルフォント設定も一緒に追加する
                        DataNode sfdNode = CreateScalableFontDescriptionNode();
                        (child.DataModel as ScalableFontDataModel).ScalableFontDescription = sfdNode.DataModel;
                        child.Nodes.Add(sfdNode);
                    }
                }
                // マルチスケーラブルフォントの設定
                else if (child.DataModel is MultiScalableFontDataModel)
                {
                    if ((child.DataModel as MultiScalableFontDataModel).ScalableFontDescriptionSet.Count == 0)
                    {
                        // スケーラブルフォント設定も一緒に追加する
                        DataNode sfdNode = CreateScalableFontDescriptionNode();
                        (child.DataModel as MultiScalableFontDataModel).ScalableFontDescriptionSet.Add(sfdNode.DataModel);
                        child.Nodes.Add(sfdNode);
                    }
                }
                // スケーラブルフォント設定の設定
                else if (child.DataModel is ScalableFontDescriptionDataModel)
                {
                    if ((child.DataModel as ScalableFontDescriptionDataModel).FontDataModel == null)
                    {
                        // フォントも一緒に追加する
                        DataNode fontNode = CreateFontNode(FontFileType.Scalable);
                        (child.DataModel as ScalableFontDescriptionDataModel).SetFontDataModel(fontNode.DataModel);
                        child.Nodes.Add(fontNode);
                    }
                }
            }

            parent.Nodes.Add(child);
            parent.Expand();
            child.ExpandAll();
        }

        /// <summary>
        /// スケーラブルフォント設定ノードを生成します。
        /// 子ノードにフォントノードが追加されます。
        /// </summary>
        private DataNode CreateScalableFontDescriptionNode()
        {
            DataNode node = DataNodeFactory.CreateDataNode(DataModelFactory.CreateDataModel(DataType.ScalableFontDescription));

            // フォントも一緒に追加する
            DataNode fontNode = CreateFontNode(FontFileType.Scalable);
            (node.DataModel as ScalableFontDescriptionDataModel).SetFontDataModel(fontNode.DataModel);
            node.Nodes.Add(fontNode);

            node.Expand();

            return node;
        }

        /// <summary>
        /// フォントノードを生成します。
        /// </summary>
        private DataNode CreateFontNode(FontFileType type)
        {
            DataNode node = DataNodeFactory.CreateDataNode(DataModelFactory.CreateDataModel(DataType.Font));
            (node.DataModel as FontDataModel).FontType = type; // ツリーから追加した場合ここでFontTypeを設定する必要がある

            return node;
        }

        /// <summary>
        /// ツリーからノードを削除します。
        /// </summary>
        private void RemoveNode(DataNode node)
        {
            // ペアフォントの設定
            DataNode parent = node.Parent as DataNode;
            if (parent.DataModel is PairFontDataModel)
            {
                PairFontDataModel dataModel = parent.DataModel as PairFontDataModel;
                dataModel.Remove(node.DataModel);
            }
            // ビットマップフォントの設定
            else if (parent.DataModel is BitmapFontDataModel)
            {
                BitmapFontDataModel dataModel = parent.DataModel as BitmapFontDataModel;
                dataModel.RemoveFontDataModel();
            }
            // スケーラブルフォントの設定
            else if (parent.DataModel is ScalableFontDataModel)
            {
                ScalableFontDataModel dataModel = parent.DataModel as ScalableFontDataModel;
                dataModel.ScalableFontDescription = null;
            }
            // マルチスケーラブルフォントの設定
            else if (parent.DataModel is MultiScalableFontDataModel)
            {
                MultiScalableFontDataModel dataModel = parent.DataModel as MultiScalableFontDataModel;
                dataModel.ScalableFontDescriptionSet.Remove(node.DataModel);
            }
            // スケーラブルフォント設定の設定
            else if (parent.DataModel is ScalableFontDescriptionDataModel)
            {
                ScalableFontDescriptionDataModel dataModel = parent.DataModel as ScalableFontDescriptionDataModel;
                dataModel.RemoveFontDataModel();
            }

            node.Remove();
            parent.Expand();
        }

        /// <summary>
        /// ツリーのノードを変更します。
        /// </summary>
        private void ModifyNode(DataNode node, DataModelBase dataModel)
        {
            // ビットマップフォントの設定
            if (node.DataModel is BitmapFontDataModel)
            {
                (node.Nodes[0] as DataNode).Text = dataModel.ToString();
            }
            // スケーラブルフォント設定の設定
            else if (node.DataModel is ScalableFontDescriptionDataModel)
            {
                (node.Nodes[0] as DataNode).Text = dataModel.ToString();
            }
            // フォントの設定
            else if (node.DataModel is FontDataModel)
            {
                node.Text = dataModel.ToString();
            }
        }

        /// <summary>
        /// ツリーのノードを入れ替えます。
        /// </summary>
        private void ReplaceNode(DataNode node)
        {
            // ペアフォントの設定
            if (node.DataModel is PairFontDataModel)
            {
                List<DataNode> tmp = new List<DataNode>();
                tmp.AddRange(node.Nodes.OfType<DataNode>());
                tmp.Reverse();
                node.Nodes.Clear();
                node.Nodes.AddRange(tmp.ToArray());
                node.Expand();
            }
            // マルチスケーラブルフォントの設定
            else if (node.DataModel is MultiScalableFontDataModel)
            {
                MultiScalableFontDataModel dataModel = node.DataModel as MultiScalableFontDataModel;
                node.Nodes.Clear();

                foreach (ScalableFontDescriptionDataModel descDataModel in dataModel.ScalableFontDescriptionSet)
                {
                    node.Nodes.Add(DataNodeFactory.CreateDataNode(descDataModel));
                }

                node.ExpandAll();
            }

            // 後処理イベントを発行します
            EventParam args = new EventParam(EventType.AfterReplace, null);
            EventManager.entry(this, args);
        }

        /// <summary>
        /// データに変更があったかをチェックします。
        /// </summary>
        private bool CheckModyfied(DataNode nodeA, DataNode nodeB)
        {
            // データモデルを比較します
            if (!nodeA.DataModel.Equals(nodeB.DataModel))
            {
                return true;
            }

            // 子ノードの数を調べます
            if (nodeA.Nodes.Count != nodeB.Nodes.Count)
            {
                return true;
            }

            // 子ノードを再帰的にチェックします
            for (int i = 0; i < nodeA.Nodes.Count; i++)
            {
                if (CheckModyfied(nodeA.Nodes[i] as DataNode, nodeB.Nodes[i] as DataNode))
                {
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// クリップボードにコピーします。
        /// </summary>
        public void CopyToClipBoard()
        {
            if (this.SelectedNode != null)
            {
                // ノードに紐付いたデータモデルをクリップボードにコピー
                ClipBoardData data = new ClipBoardData(this.SelectedNode.DataModel);
                Clipboard.SetDataObject(data, true);

                // イベント発行
                EventParam args = new EventParam(EventType.CopyClipBoard, this.SelectedNode);
                EventManager.entry(this, args);
            }
        }

        /// <summary>
        /// クリップボードから貼り付けます。
        /// </summary>
        public void PasteFromClipBoard(bool copyChild)
        {
            IDataObject dataObject = Clipboard.GetDataObject();
            if (dataObject != null)
            {
                if (this.SelectedNode != null && dataObject.GetDataPresent(typeof(ClipBoardData)))
                {
                    ClipBoardData data = dataObject.GetData(typeof(ClipBoardData)) as ClipBoardData;

                    // 貼り付け
                    if (this.SelectedNode.DataModel.CopyParam(data.Data as DataModelBase, copyChild))
                    {
                        // ノードツリーの再構築
                        if (copyChild)
                        {
                            DataNode node = DataNodeFactory.CreateDataNode(this.SelectedNode.DataModel);
                            this.SelectedNode.Nodes.Clear();
                            foreach (DataNode child in node.Nodes)
                            {
                                this.SelectedNode.Nodes.Add(child);
                            }
                            this.SelectedNode.ExpandAll();
                        }

                        // フォント変更イベント
                        if (data.Data is FontDataModel)
                        {
                            EventParam args = new EventParam(EventType.ModifyFont, data.Data);
                            EventManager.entry(this, args);
                        }

                        // ペーストイベント
                        {
                            EventParam args = new EventParam(EventType.PasteParams, null);
                            EventManager.entry(this, args);
                        }
                    }
                    else
                    {
                        Debug.Assert(false);
                    }
                }
            }
        }

        /// <summary>
        /// クリップボードから子ノードとして貼り付けます。
        /// </summary>
        private void PasteFromClipBoardAsChild()
        {
            IDataObject dataObject = Clipboard.GetDataObject();
            if (dataObject != null)
            {
                if (this.SelectedNode != null && dataObject.GetDataPresent(typeof(ClipBoardData)))
                {
                    ClipBoardData data = dataObject.GetData(typeof(ClipBoardData)) as ClipBoardData;

                    // 貼り付け
                    if (this.SelectedNode.DataModel.CanAddChild(data.Data as DataModelBase))
                    {
                        // フォント追加イベント
                        {
                            DataModelBase cloneDataModel = (data.Data as DataModelBase).Clone();
                            DataNode node = DataNodeFactory.CreateDataNode(cloneDataModel);

                            EventParam args = new EventParam(EventType.AddFont, node);
                            EventManager.entry(this, args);
                        }

                        // ペーストイベント
                        {
                            EventParam args = new EventParam(EventType.PasteParams, null);
                            EventManager.entry(this, args);
                        }
                    }
                    else
                    {
                        Debug.Assert(false);
                    }
                }
            }
        }

        /// <summary>
        /// 貼り付け可能か。
        /// </summary>
        public bool CanPaste()
        {
            IDataObject dataObject = Clipboard.GetDataObject();
            if (dataObject != null)
            {
                if (this.SelectedNode != null && dataObject.GetDataPresent(typeof(ClipBoardData)))
                {
                    ClipBoardData data = dataObject.GetData(typeof(ClipBoardData)) as ClipBoardData;
                    if (data != null)
                    {
                        bool bRet = (this.SelectedNode.DataModel.DataType == (data.Data as DataModelBase).DataType);
                        if (this.SelectedNode.DataModel is FontDataModel && data.Data is FontDataModel)
                        {
                            bRet = (this.SelectedNode.DataModel as FontDataModel).FontType == (data.Data as FontDataModel).FontType;
                        }

                        return bRet;
                    }
                }
            }

            return false;
        }

        /// <summary>
        /// 子ノードとして貼り付け可能か。
        /// </summary>
        public bool CanPasteAsChild()
        {
            IDataObject dataObject = Clipboard.GetDataObject();
            if (dataObject != null)
            {
                if (this.SelectedNode != null && dataObject.GetDataPresent(typeof(ClipBoardData)))
                {
                    ClipBoardData data = dataObject.GetData(typeof(ClipBoardData)) as ClipBoardData;
                    if (data != null)
                    {
                        return this.SelectedNode.DataModel.CanAddChild(data.Data as DataModelBase);
                    }
                }
            }

            return false;
        }

        /// <summary>
        /// イベントを通知します。
        /// </summary>
        public void dispatch(object sender, EventArgs arg)
        {
            EventParam param = arg as EventParam;

            switch (param.Type)
            {
                case EventType.New:
                    {
                        Clear();
                        CreateNewTree();
                        UpdateButton();
                    }
                    break;
                case EventType.Open:
                    {
                        string filePath = param.Option as string;
                        Clear();
                        CreateTree(filePath);
                        UpdateButton();
                    }
                    break;
                case EventType.SelectedNode:
                    {
                        UpdateButton();
                        UpdateToolStripMenu();
                    }
                    break;
                case EventType.AddFont:
                    {
                        DataNode node = param.Option as DataNode;
                        AddNode(this.SelectedNode, node);
                        UpdateButton();
                        UpdateToolStripMenu();
                    }
                    break;
                case EventType.RemoveFont:
                    {
                        DataNode node = param.Option as DataNode;
                        RemoveNode(node);
                        UpdateButton();
                        UpdateToolStripMenu();
                    }
                    break;
                case EventType.ModifyFont:
                    {
                        DataModelBase dataModel = param.Option as DataModelBase;
                        ModifyNode(this.SelectedNode, dataModel);
                        UpdateButton();
                    }
                    break;
                case EventType.Replace:
                    {
                        ReplaceNode(this.SelectedNode);
                    }
                    break;
                case EventType.Save:
                case EventType.SaveAs:
                    {
                        string filePath = param.Option as string;
                        FileIOManager.FcpxOpenFilePath = filePath;
                        XmlConverter.WriteXml(filePath, RootNode.DataModel);

                        this.SavedNode = _ctrlTreeView.Nodes[0] as DataNode;
                    }
                    break;
                case EventType.DialogClose:
                    {
                        // ツリービューをフォーカスする
                        _ctrlTreeView.Focus();
                    }
                    break;
                case EventType.UpdateTreeView:
                    {
                        _ctrlTreeView.Invalidate();
                    }
                    break;
                case EventType.PasteParams:
                    {
                        _ctrlTreeView.Invalidate();
                        UpdateToolStripMenu();
                    }
                    break;
                case EventType.CopyClipBoard:
                    {
                        UpdateToolStripMenu();
                    }
                    break;
            }
        }

        /// <summary>
        /// ツリービューの項目が選択された時のイベントハンドラです。
        /// </summary>
        private void Event_AfterSelect(object sender, TreeViewEventArgs e)
        {
            EventParam args = new EventParam(EventType.SelectedNode, this.SelectedNode);
            EventManager.entry(sender, args);
        }

        /// <summary>
        /// 追加ボタンが押された時のイベントハンドラです。
        /// </summary>
        private void Event_AddButtonClick(object sender, EventArgs e)
        {
            // ダイアログを表示
            AddFontDialog dialog = new AddFontDialog();
            dialog.Appear(this.SelectedNode.DataType);

            // 結果を取得
            DataType result = dialog.GetResult();

            // 結果に対応するノードをツリーに追加
            if (result != DataType.None)
            {
                DataNode node = DataNodeFactory.CreateDataNode(DataModelFactory.CreateDataModel(result));
                if (node != null)
                {
                    EventParam args = new EventParam(EventType.AddFont, node);
                    EventManager.entry(sender, args);

                    // データの変更を通知します
                    NotifyDataModified();
                }
            }
        }

        /// <summary>
        /// 削除ボタンが押された時のイベントハンドラです。
        /// </summary>
        private void Event_RemoveButtonClick(object sender, EventArgs e)
        {
            // ダイアログを表示
            WarningDialog dialog = new WarningDialog();
            dialog.Appear(WarningDialogType.RemoveFont);

            if (dialog.GetResult() == DialogResult.OK)
            {
                EventParam args = new EventParam(EventType.RemoveFont, this.SelectedNode);
                EventManager.entry(sender, args);

                // データの変更を通知します
                NotifyDataModified();
            }
        }

        /// <summary>
        /// データの変更を通知します。
        /// </summary>
        private void NotifyDataModified()
        {
            EventParam args = new EventParam(EventType.DataModified, null);
            EventManager.entry(this, args);
        }

        /// <summary>
        /// ツリービューのオーナードローです。
        /// </summary>
        private void Event_OwnerDrawText(object sender, DrawTreeNodeEventArgs e)
        {
            bool isSelected = !((e.State & TreeNodeStates.Selected) == 0);

            // 背景色
            Color backColor = ((TreeView)sender).BackColor;
            if (isSelected)
            {
                backColor = Color.FromArgb(255, 51, 153, 255);
            }

            // 文字色
            Color foreColor = ((TreeView)sender).ForeColor;
            if (!(e.Node as DataNode).DataModel.Validate)
            {
                foreColor = Color.Red;
            }
            else if (isSelected)
            {
                foreColor = Color.White;
            }

            // 背景塗りつぶし
            using (Brush b = new SolidBrush(backColor))
            {
                e.Graphics.FillRectangle(b, e.Node.Bounds);
            }

            // 文字を描画
            using (Brush b = new SolidBrush(foreColor))
            {
                e.Graphics.DrawString(
                    e.Node.Text,
                    e.Node.NodeFont ?? ((TreeView)sender).Font,
                    b,
                    new Rectangle(e.Bounds.X, e.Bounds.Y, e.Bounds.Width + 1, e.Bounds.Height)); // 文字が見切れる場合があるので幅を+1する
            }
        }

        /// <summary>
        /// ツリービューのキー判定です。
        /// </summary>
        private void Event_TreeViewKeyDown(object sender, KeyEventArgs e)
        {
            // Ctrl + C
            if (e.KeyCode == Keys.C && e.Control)
            {
                CopyToClipBoard();
            }
            // Ctrl + Shift + V
            else if (e.KeyCode == Keys.V && e.Control && e.Shift)
            {
                if (this.CanPaste())
                {
                    PasteFromClipBoard(true);
                }
            }
            // Ctrl + Alt + V
            else if (e.KeyCode == Keys.V && e.Control && e.Alt)
            {
                if (this.CanPaste())
                {
                    PasteFromClipBoard(false);
                }
            }
            // Ctrl + V
            else if (e.KeyCode == Keys.V && e.Control)
            {
                if (this.CanPasteAsChild())
                {
                    PasteFromClipBoardAsChild();
                }
            }
        }

        /// <summary>
        /// ツリービューのマウス判定です。
        /// </summary>
        private void Event_TreeViewMouseDown(object sender, MouseEventArgs e)
        {
            // 右クリックでもノードを選択させる
            if (e.Button == MouseButtons.Right)
            {
                _ctrlTreeView.SelectedNode = _ctrlTreeView.GetNodeAt(e.X, e.Y);
            }
        }

        /// <summary>
        /// コンテキストメニュー用のイベントハンドラです（コピー）。
        /// </summary>
        private void Event_CopyNode(object sender, EventArgs e)
        {
            CopyToClipBoard();
        }

        /// <summary>
        /// コンテキストメニュー用のイベントハンドラです（プロパティを貼り付け）。
        /// </summary>
        private void Event_PasteNodeProperty(object sender, EventArgs e)
        {
            PasteFromClipBoard(false);
        }

        /// <summary>
        /// コンテキストメニュー用のイベントハンドラです（ノードを貼り付け）。
        /// </summary>
        private void Event_PasteNodeTree(object sender, EventArgs e)
        {
            PasteFromClipBoard(true);
        }

        /// <summary>
        /// コンテキストメニュー用のイベントハンドラです（ノードを貼り付け）。
        /// </summary>
        private void Event_PasteNodeChild(object sender, EventArgs e)
        {
            PasteFromClipBoardAsChild();
        }
    }
}
