﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace FcpxMaker.DataModel
{
    using static FcpxMaker.Type;

    /// <summary>
    /// スケーラブルフォントのデータモデルです。
    /// </summary>
    [Serializable]
    public class ScalableFontDataModel : DataModelBase
    {
        /// <summary>
        /// フォントのサイズです。
        /// </summary>
        public float FontSize { get; set; } = 40;

        /// <summary>
        /// 代替文字を 10 進数で指定します。
        /// 省略した場合は 63 ('?' の文字)になります。
        /// </summary>
        public uint AlternateChar { get; set; } = '?';

        /// <summary>
        /// 行間のオフセット値を指定します。
        /// 省略した場合は 0 になります。
        /// </summary>
        public int LineFeedOffset { get; set; } = 0;

        /// <summary>
        /// スケーラブルフォントの設定です。
        /// </summary>
        public DataModelBase ScalableFontDescription { get; set; }

        /// <summary>
        /// データの種類です。
        /// </summary>
        override public DataType DataType
        {
            get { return DataType.ScalableFont; }
        }

        /// <summary>
        /// データの検証結果です。
        /// </summary>
        override public bool Validate
        {
            get
            {
                // ScalableFontDescriptionが存在する
                if ((ScalableFontDescription == null) ||
                    (AlternateChar == 0))
                {
                    return false;
                }

                return true;
            }
        }

        /// <summary>
        /// データモデルを比較します。
        /// </summary>
        override public bool Equals(DataModelBase dataModel)
        {
            ScalableFontDataModel dm = dataModel as ScalableFontDataModel;
            if (dm == null) return false;

            if (this.ScalableFontDescription == null && dm.ScalableFontDescription != null)
            {
                return false;
            }

            if ((this.FontSize != dm.FontSize) ||
                (this.AlternateChar != dm.AlternateChar) ||
                (this.LineFeedOffset != dm.LineFeedOffset) ||
                (this.ScalableFontDescription != null && !this.ScalableFontDescription.Equals(dm.ScalableFontDescription)))
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// データモデルを複製します。
        /// </summary>
        override public DataModelBase Clone()
        {
            ScalableFontDataModel newDataModel = new ScalableFontDataModel();
            newDataModel.FontSize = this.FontSize;
            newDataModel.AlternateChar = this.AlternateChar;
            newDataModel.LineFeedOffset = this.LineFeedOffset;
            newDataModel.ScalableFontDescription = this.ScalableFontDescription?.Clone();

            return newDataModel;
        }

        /// <summary>
        /// データモデルをコピーします。
        /// </summary>
        override public bool CopyParam(DataModelBase dataModel, bool copyChild)
        {
            ScalableFontDataModel dm = dataModel as ScalableFontDataModel;
            if (dm == null) return false;

            this.FontSize = dm.FontSize;
            this.AlternateChar = dm.AlternateChar;
            this.LineFeedOffset = dm.LineFeedOffset;

            if (copyChild)
            {
                this.ScalableFontDescription = dm.ScalableFontDescription;
            }

            return true;
        }

        /// <summary>
        /// 子ノードとして追加可能かどうかを返します。
        /// </summary>
        override public bool CanAddChild(DataModelBase dataModel)
        {
            if (dataModel is ScalableFontDescriptionDataModel)
            {
                if (this.ScalableFontDescription == null)
                {
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// 表示名です。
        /// </summary>
        override public string ToString()
        {
            return DataType.ToString();
        }
    }
}
