﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace FcpxMaker.DataModel
{
    [Serializable]
    public class CharCodeRangeDataModel : DataModelBase
    {
        private const int __charCodeRangeNum = 16;
        private CharCodeRange[] _charCodeRangeArray = new CharCodeRange[__charCodeRangeNum];

        /// <summary>
        /// 16進数表示か。
        /// </summary>
        public bool IsHex { get; set; } = true;

        /// <summary>
        /// 文字範囲のリストを取得します。
        /// </summary>
        public IEnumerable<CharCodeRange> CharCodeRangeList
        {
            get { return _charCodeRangeArray; }
        }

        /// <summary>
        /// データの検証結果です。
        /// </summary>
        override public bool Validate
        {
            get
            {
                foreach (CharCodeRange range in _charCodeRangeArray)
                {
                    if (range.Checked)
                    {
                        if (range.First > range.Last)
                        {
                            return false;
                        }
                    }
                }

                return true;
            }
        }

        /// <summary>
        /// インデックスを指定して文字範囲を取得します。
        /// </summary>
        public CharCodeRange GetFontRange(int index)
        {
            return _charCodeRangeArray[index];
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public CharCodeRangeDataModel()
        {
            for (int i = 0; i < __charCodeRangeNum; i++)
            {
                _charCodeRangeArray[i] = new CharCodeRange();
            }
        }

        /// <summary>
        /// データモデルを比較します。
        /// </summary>
        override public bool Equals(DataModelBase dataModel)
        {
            CharCodeRangeDataModel dm = dataModel as CharCodeRangeDataModel;
            if (dm == null) return false;

            for (int i = 0; i < __charCodeRangeNum; i++)
            {
                if (!this.GetFontRange(i).Equals(dm.GetFontRange(i)))
                {
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// データモデルを複製します。
        /// </summary>
        override public DataModelBase Clone()
        {
            CharCodeRangeDataModel newDataModel = new CharCodeRangeDataModel();

            newDataModel.IsHex = this.IsHex;

            for (int i = 0; i < __charCodeRangeNum; i++)
            {
                newDataModel._charCodeRangeArray[i] = this.GetFontRange(i).Clone();
            }

            return newDataModel;
        }

        /// <summary>
        /// データモデルをコピーします。
        /// </summary>
        override public bool CopyParam(DataModelBase dataModel, bool copyChild)
        {
            CharCodeRangeDataModel dm = dataModel as CharCodeRangeDataModel;
            if (dm == null) return false;

            this.IsHex = dm.IsHex;

            for (int i = 0; i < __charCodeRangeNum; i++)
            {
                this._charCodeRangeArray[i].CopyParam(dm._charCodeRangeArray[i]);
            }

            return true;
        }
    }

    [Serializable]
    public class CharCodeRange
    {
        /// <summary>
        /// チェック。
        /// </summary>
        public bool Checked { get; set; }

        /// <summary>
        /// 文字の範囲の最初を10進数で指定します。
        /// </summary>
        public uint First { get; set; }

        /// <summary>
        /// 文字の範囲の最後を10進数で指定します。
        /// </summary>
        public uint Last { get; set; }

        /// <summary>
        /// 比較します。
        /// </summary>
        public bool Equals(CharCodeRange range)
        {
            if ((this.Checked == range.Checked) &&
                (this.First == range.First) &&
                (this.Last == range.Last))
            {
                return true;
            }

            return false;
        }

        /// <summary>
        /// 複製します。
        /// </summary>
        public CharCodeRange Clone()
        {
            CharCodeRange newdata = new CharCodeRange();
            newdata.Checked = this.Checked;
            newdata.First = this.First;
            newdata.Last = this.Last;

            return newdata;
        }

        /// <summary>
        /// パラメータをコピーします。
        /// </summary>
        public void CopyParam(CharCodeRange data)
        {
            this.Checked = data.Checked;
            this.First = data.First;
            this.Last = data.Last;
        }
    }
}
