﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.Drawing;
using System.Windows.Forms;

namespace LayoutEditor.Structures.SerializableObject
{
    using Controls;
    using Forms.ToolWindows;
    using System.IO;
    using System.Diagnostics;

    /// <summary>
    /// 設定保存のユーティリティクラスです。
    /// </summary>
    public static class LEToolFormSettingHelper
    {
        /// <summary>
        /// リストビューヘッダ位置の保存
        /// </summary>
        public static void SaveColumnPotisionSettings(this LEToolFormSetting setting, ListView listView)
        {
            foreach (ColumnHeader column in listView.Columns)
            {
                setting.UserDataSet.Add(new LEControlUserDataChunk(listView.Name + column.Text, column.Width.ToString()));
            }
        }

        /// <summary>
        /// リストビューヘッダ位置の読み込み
        /// </summary>
        public static void LoadColumnPotisionSettings(this LEToolFormSetting setting, ListView listView)
        {
            foreach (ColumnHeader column in listView.Columns)
            {
                int value = 0;
                if (TryToLoadDataByName(setting, listView.Name + column.Text, ref value))
                {
                    column.Width = value;
                }
            }
        }

        //----------------------------------------------------------

        /// <summary>
        /// コンボボックスの保存(文字列だけ)
        /// </summary>
        public static void SaveComboboxStrings(this LEToolFormSetting setting, ToolStripComboBox combobox)
        {

            string itemStrings = string.Empty;
            foreach (string itemString in combobox.Items)
            {
                itemStrings += itemString + "\t";
            }

            Debug.Assert(!string.IsNullOrEmpty(combobox.Name));
            string controlName = combobox.Name;

            setting.UserDataSet.Add(new LEControlUserDataChunk(controlName, itemStrings));
        }

        /// <summary>
        /// コンボボックスの読み込み(文字列だけ)
        /// </summary>
        public static void LoadComboboxStrings(this LEToolFormSetting setting, ToolStripComboBox combobox)
        {
            Debug.Assert(!string.IsNullOrEmpty(combobox.Name));
            string controlName = combobox.Name;
            LEControlUserDataChunk data = setting.FindUserDataByName(controlName);
            if (data != null)
            {
                foreach (string itemString in data.Value.Split('\t'))
                {
                    if (itemString.Trim() == string.Empty)
                    {
                        continue;
                    }

                    if (!combobox.Items.Contains(itemString))
                    {
                        combobox.Items.Add(itemString);
                    }
                }

                UIComboBox.ExpandDropDownWidth(combobox.ComboBox);
            }
        }

        //----------------------------------------------------------

        /// <summary>
        /// 設定読み込み
        /// </summary>
        public static bool TryToLoadDataByName(this LEToolFormSetting setting, string dataName, ref int value)
        {
            LEControlUserDataChunk data = setting.FindUserDataByName(dataName);
            if (data != null)
            {
                value = Int32.Parse(data.Value);
                return true;
            }

            return false;
        }

        /// <summary>
        /// 設定読み込み
        /// </summary>
        public static bool TryToLoadDataByName(this LEToolFormSetting setting, string dataName, ref float value)
        {
            LEControlUserDataChunk data = setting.FindUserDataByName(dataName);
            if (data != null)
            {
                value = float.Parse(data.Value);
                return true;
            }

            return false;
        }
    }

    #region LEToolFormSetting
    /// <summary>
    /// 設定保存用クラス
    /// </summary>
    public class LEToolFormSetting
    {
        Size _size;
        Point _position;
        string _name;
        FormWindowState _formWindowState;

        List<LEToolFormSetting> _subSettingData = new List<LEToolFormSetting>();
        List<LEControlUserDataChunk> _userData = new List<LEControlUserDataChunk>();

        #region プロパティ
        public Size Size
        {
            get { return _size; }
            set { _size = value; }
        }

        public Point Position
        {
            get { return _position; }
            set { _position = value; }
        }

        public string Name
        {
            get { return _name; }
            set { _name = value; }
        }

        public List<LEToolFormSetting> SubSettingData
        {
            get { return _subSettingData; }
            set { _subSettingData = value; }
        }

        public List<LEControlUserDataChunk> UserDataSet
        {
            get { return _userData; }
            set { _userData = value; }
        }

        public FormWindowState FormWindowState
        {
            get { return _formWindowState; }
            set { _formWindowState = value; }
        }

        #endregion プロパティ

        /// <summary>
        /// ユーザデータを追加登録します。
        /// </summary>
        public void AddUserData( string dataName, string val )
        {
            LEControlUserDataChunk dataChunk =
                new LEControlUserDataChunk( dataName, val );

            _userData.Add( dataChunk );
        }

        /// <summary>
        /// UserDataChunk を UserDataChunk.Name で検索します。
        /// </summary>
        public LEControlUserDataChunk FindUserDataByName( string name )
        {
            return _userData.Find( delegate( LEControlUserDataChunk chunk )
            {
                return chunk.Name == name;
            } );
        }

        /// <summary>
        /// UserDataChunk を UserDataChunk.Name で検索します。
        /// </summary>
        public IEnumerable<LEControlUserDataChunk> FindUserDataAllByName(string name)
        {
            return _userData.FindAll(delegate(LEControlUserDataChunk chunk)
            {
                return chunk.Name == name;
            });
        }

        /// <summary>
        /// 名前から、UserDataChunk 情報の取得を試みます。
        /// </summary>
        public bool TryToFindUserDataStrByName( string dataName, out string result )
        {
            LEControlUserDataChunk chunk = FindUserDataByName( dataName );
            if( chunk != null )
            {
                result = chunk.Value;
                return true;
            }
            else
            {
                result = string.Empty;
                return false;
            }
        }

        /// <summary>
        /// 浮動小数ユーザデータの取得を試みます。
        /// </summary>
        public bool TryToFindUserFloatDataByName( string dataName, out float result )
        {
            string valStr;
            if( TryToFindUserDataStrByName( dataName, out valStr ) )
            {
                result = float.Parse( valStr );
                return true;
            }

            result = 0.0f;
            return false;
        }

        /// <summary>
        /// 整数ユーザデータの取得を試みます。
        /// </summary>
        public bool TryToFindUserIntDataByName( string dataName, out int result )
        {
            string valStr;
            if( TryToFindUserDataStrByName( dataName, out valStr ) )
            {
                result = int.Parse( valStr );
                return true;
            }

            result = 0;
            return false;
        }

        /// <summary>
        /// 列挙型ユーザデータの取得を試みます。
        /// </summary>
        public bool TryToFindUserEnumDataByName<T>(string dataName, out T result)
        {
            string valStr;
            if (TryToFindUserDataStrByName(dataName, out valStr))
            {
                try
                {
                    result = (T)Enum.Parse(typeof(T), valStr);
                    return true;
                }
                catch
                {
                    Debug.Assert(false);
                }
            }

            // 適当に最初の値を設定しておく。
            result = (T)typeof(T).GetEnumValues().GetValue(0);
            return false;
        }

        /// <summary>
        /// boolユーザデータの取得を試みます。
        /// </summary>
        public bool TryToFindUserBoolDataByName( string dataName, out bool result )
        {
            string valStr;
            if( TryToFindUserDataStrByName( dataName, out valStr ) )
            {
                result = bool.Parse( valStr );
                return true;
            }

            result = false;
            return false;
        }

        /// <summary>
        /// Decimalユーザデータの取得を試みます。
        /// </summary>
        public bool TryToFindUserDecimalDataByName( string dataName, out Decimal result )
        {
            string valStr;
            if( TryToFindUserDataStrByName( dataName, out valStr ) )
            {
                result = Decimal.Parse( valStr );
                return true;
            }

            result = Decimal.Zero;
            return false;
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public LEToolFormSetting()
        {
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public LEToolFormSetting( ILEDataSerializebleWindow form , bool workspace)
        {
            form.SaveSetting(this, new SaveSettingOption() { AlsoSaveOtherThanWorkspace = !workspace });
        }

        /// <summary>
        /// ウインドウ位置を適切に調整します。
        /// </summary>
        public static Point ClampWindowPosition(Point pos, Size size)
        {
            var screen = System.Windows.Forms.Screen.FromPoint(pos);
            if (screen == null)
            {
                return Point.Empty;
            }

            if (!screen.WorkingArea.Contains(pos) || !screen.WorkingArea.Contains(pos + size))
            {
                if (pos.X < screen.WorkingArea.Left)
                {
                    pos.X = screen.WorkingArea.Left;
                }

                if (pos.Y < screen.WorkingArea.Top)
                {
                    pos.Y = screen.WorkingArea.Top;
                }

                if (pos.X + size.Width > screen.WorkingArea.Right)
                {
                    pos.X = screen.WorkingArea.Right - size.Width;
                }

                if (pos.Y + size.Height > screen.WorkingArea.Bottom)
                {
                    pos.Y = screen.WorkingArea.Bottom - size.Height;
                }
            }

            return pos;
        }

        /// <summary>
        /// ウインドウ位置を適切に調整します。
        /// </summary>
        public static Rectangle ClampWindowRectangle(Rectangle rect)
        {
            var screen = System.Windows.Forms.Screen.FromPoint(rect.Location);
            if (screen == null)
            {
                return Rectangle.FromLTRB(0, 0, rect.Width, rect.Height);
            }

            int l = rect.Left;
            int t = rect.Top;
            int r = rect.Right;
            int b = rect.Bottom;

            if (screen.WorkingArea.Left > rect.Left)
            {
                l = screen.WorkingArea.Left;
                r = l + rect.Width;
            }

            if (screen.WorkingArea.Top > rect.Top)
            {
                t = screen.WorkingArea.Top;
                b = t + rect.Height;
            }

            if (screen.WorkingArea.Right < rect.Right)
            {
                r = screen.WorkingArea.Right;
                l = r - rect.Width;
            }

            if (screen.WorkingArea.Bottom < rect.Bottom)
            {
                b = screen.WorkingArea.Bottom;
                t = b - rect.Height;
            }

            return Rectangle.FromLTRB(l, t, r, b);
        }
    }
    #endregion LEToolFormSetting
}
