﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Diagnostics;
using System.Windows.Forms;

namespace LayoutEditor.Forms.ToolWindows.common
{
    using LECore.Structures;

    /// <summary>
    /// TreeNode.Tagに IPaneを格納するノードの状態をチェックする
    /// 処理を定義するクラスです。
    ///
    /// TreeNodeFactory に、ポリシークラスとして、
    /// コンストラクタの引数で与えられます。
    ///
    /// </summary>
    class PaneStateCheckPolicy : ITreeNodeStateCheckPolicy
    {
        /// <summary>
        /// ステート保存クラス
        /// </summary>
        class PaneState
        {
            string _name;
            bool _bHidden;
            bool _bLocked;

            public PaneState( IPane pane )
            {
                Set( pane );
            }

            public void Set( IPane pane )
            {
                _name = pane.PaneName;
                _bHidden = pane.Hidden;
                _bLocked = pane.Locked;
            }

            public bool IsChanged( IPane pane )
            {
                return _name != pane.PaneName ||
                    _bHidden != pane.Hidden ||
                    _bLocked != pane.Locked;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public bool CheckChanged( object current, object clone )
        {
            if( current is IPane && clone is PaneState )
            {
                IPane pane = current as IPane;
                PaneState state = clone as PaneState;

                if( state.IsChanged( pane ) )
                {
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        ///
        /// </summary>
        public void MakeClone( object current, out object clone )
        {
            if( current is IPane )
            {
                clone = new PaneState( current as IPane );
            }
            else
            {
                clone = null;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void UpdateClone( object current, object clone )
        {
            if( clone is PaneState )
            {
                ( clone as PaneState ).Set( current as IPane );
            }
        }
    }


    /// <summary>
    /// PaneTreeViewHelper の概要の説明です。
    /// </summary>
    public static class PaneTreeViewHelper
    {
        public static readonly Color _ActiveFontColor      = Color.Black;
        public static readonly Color _selectedActiveFontColor = Color.White;

        public static readonly Color _NonActiveFontColor   = Color.Gray;

        public static readonly Color _selectedBgColor = SystemColors.MenuHighlight;
        public static readonly Color _unselectedBgColor = Color.White;

        static readonly Font _NodefontNormal = new Font("MS UI Gothic", 9.0f);
        static readonly Font _NodefontTarget = new Font(_NodefontNormal, FontStyle.Bold);

        static public Color GetPaneNodeForeColor( bool bActive, bool bSelected )
        {
            if( bActive  )
            {
                return bSelected ? _selectedActiveFontColor : _ActiveFontColor;
            }
            else
            {
                return _NonActiveFontColor;
            }
        }

        static public Font NodefontNormal{ get { return _NodefontNormal;} }
        static public Font NodefontTarget { get { return _NodefontTarget; } }
        static public Font GetPaneNodeFont( bool bSelected )
        {
            return bSelected ? NodefontTarget : NodefontNormal;
        }


        static public Color GetPaneNodeBackColor( bool bSelected )
        {
            return bSelected ? _selectedBgColor : _unselectedBgColor;
        }

        /// <summary>
        /// 廃止になりました。
        /// </summary>
        static public string GetPaneNodeText( IPane pane )
        {
            string nodeText = pane.PaneName;
            return nodeText;
        }

        /// <summary>
        /// コメント文字列を取得します。
        /// </summary>
        static public string GetUICommentText(string comment)
        {
            return comment != string.Empty ?
                string.Format("{0} : {1}", LayoutEditor.StringResMgr.Get("TAG_COMMENT"), comment) : string.Empty;
        }

        #region SelectNodeByName
        /// <summary>
        /// 名前でノードを選択します。
        /// </summary>
        static public bool SelectNodeByNodeText( TreeView tv, string modeName )
        {
            TreeNode node = GetNodeByNodeText(tv, modeName);
            if (node != null)
            {
                tv.SelectedNode = node;
                return true;
            }

            return false;
        }

        /// <summary>
        /// 名前でノードを取得します。
        /// </summary>
        static public TreeNode GetNodeByNodeText(TreeNode rootNode, string modeText)
        {
            return GetNodeIf_(rootNode, (node) => node.Text == modeText).FirstOrDefault();
        }

        /// <summary>
        /// 名前でノードを取得します。
        /// </summary>
        static public TreeNode GetNodeByNodeText(TreeView tv, string modeText)
        {
            return GetNodeIf(tv, (node) => node.Text == modeText);
        }

        static private IEnumerable<TreeNode> GetNodeIf_(TreeNode node, Predicate<TreeNode> predicate)
        {
            if (predicate(node))
            {
                yield return node;
            }
            else
            {
                foreach (TreeNode childNode in node.Nodes)
                {
                    foreach (var foundNode in GetNodeIf_(childNode, predicate))
                    {
                        yield return foundNode;
                    }
                }
            }
        }

        /// <summary>
        /// 名前でノードを取得します。
        /// </summary>
        static public TreeNode GetNodeIf(TreeView tv, Predicate<TreeNode> predicate)
        {
            TreeNode resultNode = null;
            foreach (TreeNode node in tv.Nodes)
            {
                resultNode = GetNodeIf_(node, predicate).FirstOrDefault();
                if (resultNode != null)
                {
                    return resultNode;
                }
            }
            return resultNode;
        }

        /// <summary>
        /// 名前でノードを取得します。
        /// </summary>
        static public IEnumerable<TreeNode> GetAllNodeIf(TreeView tv, Predicate<TreeNode> predicate)
        {
            foreach (TreeNode node in tv.Nodes)
            {
                foreach (TreeNode foundNode in GetNodeIf_(node, predicate))
                {
                    yield return foundNode;
                }
            }
        }

        #endregion SelectNodeByNodeText
    }
}
