﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;
using System.Drawing;

namespace LayoutEditor.Forms.ToolWindows.common
{
    using MathUtil = LECore.Util.MathUtil;

    /// <summary>
    /// 他のウインドウに吸着するウインドウを表現するクラス。
    /// 内部に、吸着処理の対象となるウインドウのリストを持ちます。
    /// クラスは、対象に登録されたウインドウについて、ウインドウメッセージを監視し、
    /// ウインドウ移動、サイズ変更メッセージをハンドルして、吸着処理を実現します。
    /// </summary>
    class StickyWindow : NativeWindow
    {
        //-----------------------------------------------------
        #region 定義

        // 補正処理を行うイベントハンドラ
        private delegate bool MessageHandler( ref Message m );

        private enum ResizeDirection
        {
            None	= 0x0,
            Top		= 0x2,
            Bottom = 0x4,
            Left	= 0x8,
            Right	= 0x10
        };

        /// <summary>
        /// 吸着する閾値
        /// </summary>
        private const int _StickThreshold = 10;

        #endregion


        //-----------------------------------------------------
        #region フィールド
        /// <summary>
        /// 対象となる、フォームセット
        /// </summary>
        static List<Form>		_GrobalTargetFormSet = new List<Form>();
        // 現在のイベントハンドラ
        MessageHandler			_currentMessageHandler;
        // 吸着処理が有効か
        bool					_bStickyEnable = true;
        // サイズ変更方向
        ResizeDirection		_resizeDirection = ResizeDirection.None;
        // フォーム右上を原点としたマウスクリック位置
        Point					_mouseDownPosOnForm = Point.Empty;

        // 各種イベントハンドラ
        readonly MessageHandler _IdleMessageHandler;
        readonly MessageHandler _MoveMessageHandler;
        readonly MessageHandler _ResizeMessageHandler;

        /// <summary>
        /// 処理対象となるフォーム
        /// </summary>
        readonly Form			  _ownerForm;

        #endregion フィールド


        //-----------------------------------------------------
        #region 対象登録
        /// <summary>
        /// 対象の登録
        /// </summary>
        static void RegisterTarget( Form target )
        {
            if( !_GrobalTargetFormSet.Contains( target ) )
            {
                _GrobalTargetFormSet.Add( target );
            }
        }

        /// <summary>
        /// 対象の登録
        /// </summary>
        static void RemoveTarget( Form target )
        {
            _GrobalTargetFormSet.Remove( target );
        }
        #endregion 対象登録


        //-----------------------------------------------------
        #region プロパティ

        /// <summary>
        /// 吸着処理が有効か
        /// </summary>
        public bool StickyEnable
        {
            get { return _bStickyEnable; }
            set { _bStickyEnable = value; }
        }

        /// <summary>
        /// サイズ変更中か取得します。
        /// </summary>
        bool _ResizingForm
        {
            get { return _currentMessageHandler == _ResizeMessageHandler; }
        }

        /// <summary>
        /// 移動中か取得します。
        /// </summary>
        bool _MovingForm
        {
            get { return _currentMessageHandler == _MoveMessageHandler; }
        }

        #endregion


        //-----------------------------------------------------
        #region 構築
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public StickyWindow( Form form )
        {
            _ownerForm = form;

            // イベントハンドラの初期化
            _IdleMessageHandler = new MessageHandler( OnIdleMessageHandler_ );
            _MoveMessageHandler = new MessageHandler( OnMoveMessageHandler_ );
            _ResizeMessageHandler = new MessageHandler( OnSizeChangeMessageHandler_ );

            _currentMessageHandler = _IdleMessageHandler;

            AssignHandle( _ownerForm.Handle );
        }

        /// <summary>
        /// ウインドウハンドルが変更されたときの処理
        /// </summary>
        protected override void OnHandleChange()
        {
            if( (int)this.Handle != 0 )
            {
                _GrobalTargetFormSet.Add( this._ownerForm );
            }
            else
            {
                _GrobalTargetFormSet.Remove( this._ownerForm );
            }
        }


        #endregion

        //-----------------------------------------------------
        #region メッセージハンドラ

        /// <summary>
        /// 操作終了
        /// </summary>
        void Cancel_()
        {
            _resizeDirection = ResizeDirection.None;
            _mouseDownPosOnForm = Point.Empty;
            _ownerForm.Capture = false;
            _currentMessageHandler = _IdleMessageHandler;
        }

        /// <summary>
        ///
        /// </summary>
        Point GetMousePoint_( Message m )
        {
            return new Point(
                (short)((int)m.LParam & 0xFFFF),
                (short)(((int)m.LParam >> 16 ) & 0xFFFF) );
        }

        /// <summary>
        /// サイズ変更フラグが有効か取得します。
        /// </summary>
        bool IsResizeFlagEnabled_( ResizeDirection resizeDirection )
        {
            return ( _resizeDirection & resizeDirection ) == resizeDirection;
        }

        /// <summary>
        /// NC領域のクリックから、ウインドウ移動、サイズ変更の開始を
        /// チェックします。
        /// </summary>
        /// <param name="iHitTest"></param>
        /// <param name="point"></param>
        /// <returns></returns>
        bool CheckNCLButtonDown_( int iHitTest, Point point )
        {
            Rectangle rParent = _ownerForm.Bounds;

            if( !StickyEnable )
            {
                return false;
            }

            _mouseDownPosOnForm = new Point(
                point.X - _ownerForm.Left,
                point.Y - _ownerForm.Top );


            if( iHitTest == LECore.Win32.HT.HTCAPTION )
            {
                // 移動を処理するモードに移行します。
                // メッセージキャプチャを開始する
                if( !_ownerForm.Capture )
                {
                    _ownerForm.Capture = true;
                }

                _currentMessageHandler = _MoveMessageHandler;
                return true;
            }
            else
            {
                // サイズ変更を処理するモードに移行します。
                switch( iHitTest )
                {
                    case LECore.Win32.HT.HTTOPLEFT: return StartResize_( ResizeDirection.Top | ResizeDirection.Left );
                    case LECore.Win32.HT.HTTOP: return StartResize_( ResizeDirection.Top );
                    case LECore.Win32.HT.HTTOPRIGHT: return StartResize_( ResizeDirection.Top | ResizeDirection.Right );
                    case LECore.Win32.HT.HTRIGHT: return StartResize_( ResizeDirection.Right );
                    case LECore.Win32.HT.HTBOTTOMRIGHT: return StartResize_( ResizeDirection.Bottom | ResizeDirection.Right );
                    case LECore.Win32.HT.HTBOTTOM: return StartResize_( ResizeDirection.Bottom );
                    case LECore.Win32.HT.HTBOTTOMLEFT: return StartResize_( ResizeDirection.Bottom | ResizeDirection.Left );
                    case LECore.Win32.HT.HTLEFT: return StartResize_( ResizeDirection.Left );
                }
            }
            return false;
        }


        #region 移動処理関連

        /// <summary>
        /// 移動を開始します。
        /// </summary>
        private void StartMove()
        {
            // メッセージキャプチャを開始する
            if( !_ownerForm.Capture )
            {
                _ownerForm.Capture = true;
            }

            _currentMessageHandler = _MoveMessageHandler;
        }

        /// <summary>
        ///
        /// </summary>
        Rectangle CalcMoveRectanble_( Point p, Form form )
        {
            //-------------------------------------------------------
            // マウス位置から、新しい位置を計算する。
            p.Offset( -_mouseDownPosOnForm.X, -_mouseDownPosOnForm.Y );
            Point newLocation = form.PointToScreen( p );

            return new Rectangle( newLocation, form.Bounds.Size );
        }

        /// <summary>
        ///
        /// </summary>
        Point CalcMoveAdjustValue_( Rectangle newRectangle )
        {
            // 補正値を計算する。
            // 最大値より大きな値からスタートする。
            Point formOffset = new Point( _StickThreshold + 1, _StickThreshold + 1 );

            Screen activeScreen = Screen.FromPoint( newRectangle.Location );
            Rectangle workingRect = activeScreen.WorkingArea;

            // デスクトップ端への吸着
            Move_Stick_( workingRect, newRectangle, ref formOffset, false );

            // 他のフォームへの吸着
            foreach( Form form in _GrobalTargetFormSet )
            {
                if( form.Visible && form != this._ownerForm )
                {
                    Move_Stick_( form.Bounds, newRectangle, ref formOffset, true );
                }
            }

            // 操作がなければ、補正値をリセットします。
            if( formOffset.X >= _StickThreshold + 1 )
            {
                formOffset.X = 0;
            }

            if( formOffset.Y >= _StickThreshold + 1 )
            {
                formOffset.Y = 0;
            }

            return formOffset;
        }

        /// <summary>
        /// 移動を処理します。
        /// </summary>
        private void Move_( Point p )
        {
            // 新しい位置を計算します。
            Rectangle newRectangle = CalcMoveRectanble_( p, _ownerForm );

            // 補正量を計算します。
            Point formOffset = CalcMoveAdjustValue_( newRectangle );

            // 補正値を加算する。
            newRectangle.Offset( formOffset );

            // 位置を更新します。
            _ownerForm.Bounds = newRectangle;
        }

        /// <summary>
        /// 補正値の更新をします。
        /// </summary>
        static int CalcMinimamAdjustValue_( int from, int to, int currentAdjust )
        {
            int difference = to - from;
            if( Math.Abs( difference ) <= Math.Abs( currentAdjust ) )
            {
                return difference;
            }
            else
            {
                return currentAdjust;
            }
        }

        /// <summary>
        /// toRect 吸着対象矩形。
        /// formRect 吸着する矩形。
        /// formOffset 吸着のための操作量
        ///
        /// </summary>
        void Move_Stick_(
            Rectangle toRect,
            Rectangle formRect,
            ref Point formOffset,
            bool bInsideStick )
        {

            // X方向
            if( formRect.Top - _StickThreshold <= toRect.Bottom &&
                formRect.Bottom + _StickThreshold >= toRect.Top )
            {
                formOffset.X = CalcMinimamAdjustValue_( formRect.Left, toRect.Right, formOffset.X );
                formOffset.X = CalcMinimamAdjustValue_( formRect.Right, toRect.Left, formOffset.X );
                formOffset.X = CalcMinimamAdjustValue_( formRect.Left, toRect.Left, formOffset.X );
                formOffset.X = CalcMinimamAdjustValue_( formRect.Right, toRect.Right, formOffset.X );
            }

            // Y方向
            if( formRect.Right + _StickThreshold  >= toRect.Left &&
                formRect.Left  - _StickThreshold  <= toRect.Right )
            {
                formOffset.Y = CalcMinimamAdjustValue_( formRect.Top, toRect.Bottom, formOffset.Y );
                formOffset.Y = CalcMinimamAdjustValue_( formRect.Bottom, toRect.Top, formOffset.Y );
                formOffset.Y = CalcMinimamAdjustValue_( formRect.Top, toRect.Top, formOffset.Y );
                formOffset.Y = CalcMinimamAdjustValue_( formRect.Bottom, toRect.Bottom, formOffset.Y );
            }
        }

        #endregion

        #region サイズ変更
        /// <summary>
        ///
        /// </summary>
        bool StartResize_( ResizeDirection resDir )
        {
            _resizeDirection = resDir;

            if( !_ownerForm.Capture )
            {
                _ownerForm.Capture = true;
            }

            _currentMessageHandler = _ResizeMessageHandler;

            return true;
        }



        /// <summary>
        ///
        /// </summary>
        Rectangle CalcResizeRectanble_( Point mousePos )
        {
            Rectangle formRect = _ownerForm.Bounds;

            int origRight = formRect.Right;
            int origBottom = formRect.Bottom;

            // マウス位置から、新しい、位置とサイズを計算します。
            // このサイズを元に、補正計算をおこないます。
            if( IsResizeFlagEnabled_( ResizeDirection.Left ) )
            {
                formRect.Width = formRect.X - mousePos.X + formRect.Width;
                formRect.X = origRight - formRect.Width;
            }

            if( IsResizeFlagEnabled_( ResizeDirection.Right ) )
            {
                formRect.Width = mousePos.X - formRect.Left;
            }

            if( IsResizeFlagEnabled_( ResizeDirection.Top ) )
            {
                formRect.Height = formRect.Height - mousePos.Y + formRect.Top;
                formRect.Y = origBottom - formRect.Height;
            }

            if( IsResizeFlagEnabled_( ResizeDirection.Bottom ) )
            {
                formRect.Height = mousePos.Y - formRect.Top;
            }

            return formRect;
        }

        /// <summary>
        ///
        /// </summary>
        Rectangle CalcNewFormRectangle_( Form targetForm, Rectangle formRect, Rectangle formOffsetRect )
        {
            int origRight = formRect.Right;
            int origBottom = formRect.Bottom;
            //---------------------------------------------
            // 新しいサイズを計算します。
            int iNewWidth = formRect.Width + formOffsetRect.Width + formOffsetRect.X;
            if( IsResizeFlagEnabled_( ResizeDirection.Left ) )
            {
                if( targetForm.MaximumSize.Width != 0 )
                {
                    iNewWidth = Math.Min( iNewWidth, targetForm.MaximumSize.Width );
                }

                iNewWidth = Math.Min( iNewWidth, SystemInformation.MaxWindowTrackSize.Width );
                iNewWidth = Math.Max( iNewWidth, targetForm.MinimumSize.Width );
                iNewWidth = Math.Max( iNewWidth, SystemInformation.MinWindowTrackSize.Width );

                formRect.X = origRight - iNewWidth;
                formRect.Width = iNewWidth;
            }
            else
            {
                formRect.Width = iNewWidth;
            }


            int iNewHeight = formRect.Height + formOffsetRect.Height + formOffsetRect.Y;
            if( IsResizeFlagEnabled_( ResizeDirection.Top ) )
            {
                if( targetForm.MaximumSize.Height != 0 )
                {
                    iNewHeight = Math.Min( iNewHeight, targetForm.MaximumSize.Height );
                }

                iNewHeight = Math.Min( iNewHeight, SystemInformation.MaxWindowTrackSize.Height );
                iNewHeight = Math.Max( iNewHeight, targetForm.MinimumSize.Height );
                iNewHeight = Math.Max( iNewHeight, SystemInformation.MinWindowTrackSize.Height );

                formRect.Y = origBottom - iNewHeight;
                formRect.Height = iNewHeight;
            }
            else
            {
                formRect.Height = iNewHeight;
            }

            return formRect;
        }

        /// <summary>
        ///
        /// </summary>
        Rectangle CalcAdjustRectangle_( Point mousePos, Rectangle newRectangle )
        {
            // 調整値を格納する、矩形
            Rectangle formOffsetRect = Rectangle.Empty;
            formOffsetRect.X = _StickThreshold + 1;
            formOffsetRect.Y = _StickThreshold + 1;
            formOffsetRect.Height = 0;
            formOffsetRect.Width = 0;

            // スクリーンに対するスナップ
            Screen activeScr = Screen.FromPoint( mousePos );
            Resize_Stick_( activeScr.WorkingArea, newRectangle, ref formOffsetRect, false );

            // 他のフォームに対するスナップ
            foreach( Form sw in _GrobalTargetFormSet )
            {
                if( sw.Visible && sw != this._ownerForm )
                {
                    Resize_Stick_( sw.Bounds, newRectangle, ref formOffsetRect, true );
                }
            }

            // 変更がなかった場合は、補正値をゼロに設定します。
            if( Math.Abs( formOffsetRect.X ) >= Math.Abs( _StickThreshold + 1 ) )
            {
                formOffsetRect.X = 0;
            }
            if( Math.Abs( formOffsetRect.Width ) >= Math.Abs( _StickThreshold + 1 ) )
            {
                formOffsetRect.Width = 0;
            }
            if( Math.Abs( formOffsetRect.Y ) >= Math.Abs( _StickThreshold + 1 ) )
            {
                formOffsetRect.Y = 0;
            }
            if( Math.Abs( formOffsetRect.Height ) >= Math.Abs( _StickThreshold + 1 ) )
            {
                formOffsetRect.Height = 0;
            }

            return formOffsetRect;
        }

        /// <summary>
        ///
        /// </summary>
        private void Resize_( Point p )
        {
            Point mousePos = _ownerForm.PointToScreen( p );

            Rectangle newRectangle = CalcResizeRectanble_( mousePos );

            Rectangle formOffsetRect = CalcAdjustRectangle_( mousePos, newRectangle );

            _ownerForm.Bounds = CalcNewFormRectangle_( _ownerForm, newRectangle, formOffsetRect );
        }

        /// <summary>
        ///
        /// </summary>
        private void Resize_Stick_(
            Rectangle toRect,
            Rectangle formRect,
            ref Rectangle formOffsetRect,
            bool bInsideStick )
        {
            //--------------------------------------------
            // 垂直方向
            if( formRect.Right >= ( toRect.Left - _StickThreshold ) &&
                formRect.Left <= ( toRect.Right + _StickThreshold ) )
            {
                // Topが変更されるような処理
                if( IsResizeFlagEnabled_( ResizeDirection.Top ) )
                {
                    int newValue;

                    newValue = CalcMinimamAdjustValue_( formRect.Top, toRect.Top, formOffsetRect.Top );
                    if( newValue != formOffsetRect.Top )
                    {
                        formOffsetRect.Y = -newValue;
                    }
                    newValue = CalcMinimamAdjustValue_( formRect.Top, toRect.Bottom, formOffsetRect.Top );
                    if( newValue != formOffsetRect.Top )
                    {
                        formOffsetRect.Y = -newValue;
                    }
                }

                // Bottomが変更されるような処理
                if( IsResizeFlagEnabled_( ResizeDirection.Bottom ) )
                {
                    int newValue;

                    newValue = CalcMinimamAdjustValue_( formRect.Bottom, toRect.Top, formOffsetRect.Bottom );
                    if( newValue != formOffsetRect.Bottom )
                    {
                        formOffsetRect.Height = newValue;
                    }
                    newValue = CalcMinimamAdjustValue_( formRect.Bottom, toRect.Bottom, formOffsetRect.Bottom );
                    if( newValue != formOffsetRect.Bottom )
                    {
                        formOffsetRect.Height = newValue;
                    }
                }
            }

            //--------------------------------------------
            // 水平方向
            if( formRect.Bottom >= ( toRect.Top - _StickThreshold ) &&
                formRect.Top <= ( toRect.Bottom + _StickThreshold ) )
            {
                // Rightが変更されるような処理
                if( IsResizeFlagEnabled_( ResizeDirection.Right ) )
                {
                    int newValue;
                    newValue = CalcMinimamAdjustValue_( formRect.Right, toRect.Left, formOffsetRect.Right );
                    if( newValue != formOffsetRect.Right )
                    {
                        formOffsetRect.Width = newValue;
                    }
                    newValue = CalcMinimamAdjustValue_( formRect.Right, toRect.Right, formOffsetRect.Right );
                    if( newValue != formOffsetRect.Right )
                    {
                        formOffsetRect.Width = newValue;
                    }
                }

                // Leftが変更されるような処理
                if( IsResizeFlagEnabled_( ResizeDirection.Left ) )
                {
                    int newValue;
                    newValue = CalcMinimamAdjustValue_( formRect.Left, toRect.Right, formOffsetRect.Left );
                    if( newValue != formOffsetRect.Left )
                    {
                        formOffsetRect.X = -newValue;
                    }
                    newValue = CalcMinimamAdjustValue_( formRect.Left, toRect.Left, formOffsetRect.Left );
                    if( newValue != formOffsetRect.Left )
                    {
                        formOffsetRect.X = -newValue;
                    }
                }
            }
        }

        #endregion

        #region 独自のイベント処理
        /// <summary>
        /// 待機状態のウインドウハンドラ
        /// サイズ変更や移動をしていないときの処理。
        ///
        /// NCイベントを監視して、
        /// サイズ変更、移動などの開始を判定する。
        /// </summary>
        bool OnIdleMessageHandler_( ref Message m )
        {
            switch( m.Msg )
            {
                case LECore.Win32.WM.WM_NCLBUTTONDOWN:
                {
                    _ownerForm.Activate();
                    if( CheckNCLButtonDown_( (int)m.WParam, GetMousePoint_( m ) ) )
                    {
                        m.Result = (IntPtr)( ( _ResizingForm || _MovingForm ) ? 1 : 0 );
                        return true;
                    }
                    break;
                }
            }

            return false;
        }

        /// <summary>
        /// 移動時のハンドラ
        /// </summary>
        bool OnMoveMessageHandler_( ref Message m )
        {
            // 内部メッセージである
            if( !_ownerForm.Capture )
            {
                Cancel_();
                return false;
            }

            switch( m.Msg )
            {
                case LECore.Win32.WM.WM_LBUTTONUP:
                {
                    // 移動処理が完了した。
                    Cancel_();
                    break;
                }
                case LECore.Win32.WM.WM_MOUSEMOVE:
                {
                    // 移動処理中
                    Move_( GetMousePoint_( m ) );
                    break;
                }
            }

            return false;
        }

        /// <summary>
        /// サイズ変更時のハンドラ
        /// </summary>
        bool OnSizeChangeMessageHandler_( ref Message m )
        {
            if( !_ownerForm.Capture )
            {
                Cancel_();
                return false;
            }

            switch( m.Msg )
            {
                case LECore.Win32.WM.WM_LBUTTONUP:
                {
                    // サイズ変更終了
                    Cancel_();
                    break;
                }
                case LECore.Win32.WM.WM_MOUSEMOVE:
                {
                    Resize_( GetMousePoint_( m ) );
                    break;
                }
            }

            return false;
        }

        /// <summary>
        /// ウインドウプロシージャ
        /// まず、補正処理を試みます。
        /// 処理がされれば、既定の処理はおこなわれません。
        /// </summary>
        protected override void WndProc( ref Message m )
        {
            if( !_currentMessageHandler( ref m ) )
            {
                base.WndProc( ref m );
            }
        }
        #endregion 独自のイベント処理

        #endregion メッセージハンドラ

    }
}
